/* 
 * event.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "debug.h"
#include "upi.h"
#define	 MAX_EV	50
#include "system.h"
#include "process.h"

#ifdef XSIMUL
#include <sys/time.h>
extern struct itimer i_value, i_zero;
#endif

struct event {
int code;
  long    t;			/* time this event is to happen, delta */
  long	  i;			/* interval between events; 0=>no_repeat */
  Pfi     r;
  int	  a;
  int	  id;
  int	  useProcess;
  struct  event      *next;
  struct  event      *prev;
};

static int currentEventSize = 0;
static int nextEventId = 0x101;
Pfi	*todo_r = NULL;
int	*todo_a = NULL, *todo_id = NULL;

extern	int	x_errno;
static	struct  event  *ev_list=NULL;


/**********************************************************
/*  Internal routines for manipulating events
/*	event_delete() --- delete event from list
/*	event_insert() --- insert event into list
/***********************************************************/

static void event_delete(ev)
struct	event	*ev;
{
  if (!ev->next) {
    if (ev->prev)
      ev->prev->next = NULL;
    else 
      ev_list = NULL;
  } else {
    ev->next->t += ev->t;
    if (ev->prev) {
      ev->prev->next = ev->next;
      ev->next->prev = ev->prev;
    }
    else {
      ev_list = ev->next;
      ev->next->prev = NULL;
    }
  }
#ifdef XSIMUL
  if (!ev_list) setitimer(ITIMER_REAL,&i_zero,(struct itimer *)NULL);
#endif
}
    
static void event_insert(ev)
struct	event	*ev;
{
  struct  event  *ep;

  ev->prev = NULL;
  for (ep = ev_list; ep && ev->t >= ep->t; ep = ep->next) {
    ev->t -= ep->t;
    ev->prev = ep;
  }
  /*
   * Now, it goes before ep, and its prev pointer is right.
   */
  ev->next = ep;
  if (ev->prev == NULL) ev_list = ev; else ev->prev->next = ev;
  if (ep) {
    ep->prev = ev;
    ep->t -= ev->t;
  }
}

/***********************************************************
/*  clock_ih() --- clock interrupt handler; scans event list
/*		   and invokes routine for registered events
/*		   that have happened
/***********************************************************/

static expandEvents()
{
  currentEventSize += MAX_EV;
  if (todo_r) {
    todo_r = (Pfi *)realloc((char *)todo_r, (unsigned)(sizeof(Pfi) * currentEventSize));
    todo_a = (int *)realloc((char *)todo_a, (unsigned)(sizeof(int) * currentEventSize));
    todo_id = (int *)realloc((char *)todo_id, (unsigned)(sizeof(int) * currentEventSize));
  } else {
    todo_r = (Pfi *)malloc((unsigned)(sizeof(Pfi) * currentEventSize));
    todo_a = (int *)malloc((unsigned)(sizeof(int) * currentEventSize));
    todo_id = (int *)malloc((unsigned)(sizeof(int) * currentEventSize));
  }
}

void clock_ih()
{
  struct  event *ev;
  int	  i = 0, j;

  TRACE0(event, 2, "Clock interrupt");
  if (!ev_list) return;

  ev_list->t --;
  while (ev_list && ev_list->t <= 0) {
    TRACE1(event, 1, "Event %x expired", ev_list->r);
    if (ev_list->useProcess) {
      CreateKernelProcess(ev_list->r, 5, 2, ev_list->a, ev_list->id);
    } else {
      if (i >= currentEventSize) expandEvents();
      todo_r[i] = ev_list->r;
      todo_a[i] = ev_list->a;
      todo_id[i++] = ev_list->id;
    }
    ev = ev_list;
    ev_list = ev->next;
    if (ev_list != NULL) ev_list->prev = NULL;
    if (ev->i == 0) {				/* really delete event */
      free((char *)ev);
    } else {					/* reset event */
      ev->t = ev->i;
      event_insert(ev);
    }
  }
  for (j = 0; j < i; j++) {
    todo_r[j](todo_a[j], todo_id[j]);
  }
#ifdef XSIMUL
  if (!ev_list) setitimer(ITIMER_REAL,&i_zero,(struct itimer *)NULL);
#endif
  return;  
}

/**********************************************************
/*  event_remove() --- remove a registered event
/***********************************************************/

event_remove(r, a)
register Pfi	r;
register int	a;
{
  register struct event *ev;
  register int x = spl7();

  TRACE2(event, 2, "event remove: r = %x, a = %x", r, a);
  for (ev = ev_list; ev; ev = ev->next)  {
    /* events are identified by routine and argument */
    if (ev->r == r && ev->a == a) {	
      event_delete(ev);
      free((char *)ev);
      TRACE0(event, 2, "event remove done");
      splx(x);
      return(0);
    }
  }
  TRACE0(event, 2, "event remove, can't find event");
  x_errno = EVENT_NOT_REGISTERED;
  splx(x);
  return(-1);
}

/**********************************************************
/*  event_removeevent() --- remove a registered event
/***********************************************************/

event_removeevent(id)
int id;
{
  register struct event *ev;
  register int x = spl7();

  TRACE1(event, 2, "event removeevent: id = %x", id);
  for (ev = ev_list; ev; ev = ev->next)  {
    if (ev->id == id) {	
      event_delete(ev);
      free((char *)ev);
      TRACE0(event, 2, "event removeevent done");
      splx(x);
      return(0);
    }
  }
  TRACE0(event, 2, "event removeevent, can't find event");
  x_errno = EVENT_NOT_REGISTERED;
  splx(x);
  return(-1);
}

/**********************************************************
/*  event_register() --- register (delete) an event with
/*                        the event manager
/***********************************************************/

int event_register(r, a, interval, code)
Pfi	r;
int	a;
unsigned	interval;
int	code;
{
  register struct  event	*ev;
  register int x = spl7();
  TRACE4(event, 1, "RegisterEvent: r = %x, a = %x, i = %d, code = %x", r, a,
    interval, code);
#ifndef XSIMUL
  interval = (interval + 20 - 1) / 20;
#else
  interval = (interval + 100 - 1) / 100;
#endif

  switch (code & 0xff) {
    case EV_REMOVE:
      for (ev = ev_list; ev; ev = ev->next)  {
	/* events are identified by routine and argument */
	if (ev->r == r && ev->a == a) {	
	  event_delete(ev);
	  free((char *)ev);
	  TRACE0(event, 2, "RegisterEvent (remove) done");
	  
	  splx(x);
	  return(0);
	}
      }
      TRACE0(event, 2, "RegisterEvent (remove), can't find event");
      x_errno = EVENT_NOT_REGISTERED;
      splx(x);
      return(-1);
    case EV_ONCE:
    case EV_REPEAT:
      ev = (struct event *) malloc(sizeof(struct event));
      ev->r = r;
      ev->a = a;
      ev->t = interval;
      ev->i = ((code & 0xff) == EV_ONCE ? 0 : interval);
      ev->useProcess = code & EV_CREATEPROCESS;
      ev->id = nextEventId++;
#ifdef XSIMUL
      if (!ev_list) setitimer(ITIMER_REAL,&i_value,(struct itimer *)NULL);
#endif
      event_insert(ev);
      TRACE2(event, 2, "RegisterEvent done, id = %x, ticks = %d",ev->id,ev->t);
      splx(x);
      return(ev->id);
    default:
      x_errno = INVALID_EV_CODE;
      TRACE0(event, 2, "RegisterEvent invalid code");
      splx(x);
      return(-1);
  }
}


/***************************************************************************
/*  event_reschedule()  -- reschedule an evnet with new time interval.
/* 				if the event is expired, insert a new event;
/*				otherwise, move to correct place.
/*	
/***************************************************************************/
int
event_reschedule(id,r,a,interval,code)
int id;
Pfi r;
int a;
unsigned interval;
int code;
{
  register struct event *ev, *oev, *rev;
  register int x=spl7();
  register int t;
  int found=0;
#ifndef XSIMUL
  interval = (interval + 20 - 1) / 20;
#else
  interval = (interval + 100 - 1) / 100;
#endif

  t=interval;
  oev=rev=NULL;
  for(ev=ev_list;ev;ev=ev->next) {
    if(ev->id==id) 
      oev=ev;
    if(!found && t>ev->t) {
      t-=ev->t;
      rev=ev;
    } else found=1;
    if(oev!=NULL&&found)
      break;
  }

  
  if(oev!=NULL &&(oev==rev || oev==ev)) {

    /* if event exist and at the right place.... */
    if(oev==rev) t+=rev->t;

  } else {

    /* if not at right place ... */
    if(oev!=NULL)  event_delete(oev);
    else  {
    /* if not exist .... */
    	   oev=(struct event *)malloc(sizeof(struct event));
           oev->id=id;
    }
    /* put on the right place */
    oev->next=ev;
    oev->prev=rev;
    if(rev==NULL) ev_list = oev; else oev->prev->next = oev;
    if(ev) {
       ev->prev=oev;
       ev->t -= oev->t;
    }
  }

  /* update information */
  oev->i=((code&0xff)==EV_ONCE?0:interval);
  oev->useProcess=code&EV_CREATEPROCESS;
  oev->r=r;
  oev->a=a;
  oev->t=t;
#ifdef XSIMUL
  if(!ev_list)setitimer(ITIMER_REAL,&i_value,(struct itimer *)NULL);
#endif

  splx(x);
  return(oev->id);
}

    
