/*
 * nfs_xdr.c
 *
 * Copyright (C) 1984, Sun Microsystems, Inc.
 *
 */


/* we must define KERNEL in at least mbuf.h and param.h to get everything
 * we need */

#define KERNEL

#include <sys/param.h>

#include <sys/systm.h>
#include <sys/user.h>
#include <sys/vnode.h>
#include <sys/file.h>
#include <sys/dir.h>

#include <sys/mbuf.h>

#include <sys/vfs.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <nfs/nfs.h>
#include <netinet/in.h>
#include <rpcsvc/mount.h>
#include "nfs_xdr.h"

#ifdef NFSDEBUG
extern int stderr;

char *xdropnames[] = {"encode", "decode", "free"};
#endif

/*
 * These are the XDR routines used to serialize and deserialize
 * the various structures passed as parameters accross the network
 * between NFS clients and servers.
 */

/*
 * File access handle
 * The fhandle struct is treated a opaque data on the wire
 */
bool_t
  xdr_fhandle(xdrs, fh)
XDR *xdrs;
fhandle_t *fh;
{
  
  if (xdr_opaque(xdrs, (caddr_t)fh, NFS_FHSIZE)) {
#		ifdef NFSDEBUG
    printf("xdr_fhandle: %s %x\n",
	   xdropnames[(int)xdrs->x_op], fh);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf("xdr_fhandle %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}


/*
 * Arguments to remote write and writecache
 */
bool_t
  xdr_writeargs(xdrs, wa)
XDR *xdrs;
struct nfswriteargs *wa;
{
  if (xdr_fhandle(xdrs, &wa->wa_fhandle) &&
      xdr_long(xdrs, (long *)&wa->wa_begoff) &&
      xdr_long(xdrs, (long *)&wa->wa_offset) &&
      xdr_long(xdrs, (long *)&wa->wa_totcount) &&
      xdr_bytes(xdrs, &wa->wa_data, (u_int *)&wa->wa_count,
		NFS_MAXDATA) ) {
#		ifdef NFSDEBUG
    printf("xdr_writeargs: %s off %d ct %d\n",
	   xdropnames[(int)xdrs->x_op],
	   wa->wa_offset, wa->wa_totcount);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf("xdr_writeargs: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}


/*
 * Arguments to remote read
 */
bool_t
  xdr_readargs(xdrs, ra)
XDR *xdrs;
struct nfsreadargs *ra;
{
  
  if (xdr_fhandle(xdrs, &ra->ra_fhandle) &&
      xdr_long(xdrs, (long *)&ra->ra_offset) &&
      xdr_long(xdrs, (long *)&ra->ra_count) &&
      xdr_long(xdrs, (long *)&ra->ra_totcount) ) {
#		ifdef NFSDEBUG
    printf("xdr_readargs: %s off %d ct %d\n",
	   xdropnames[(int)xdrs->x_op],
	   ra->ra_offset, ra->ra_totcount);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf("xdr_raedargs: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}

/*
 * Info necessary to free the bp which is also an mbuf
 */
struct rrokinfo {
  int	(*func)();
  int	done;
  struct vnode *vp;
  struct buf *bp;
};


/*
 * File attributes which can be set
 */
bool_t
  xdr_sattr(xdrs, sa)
XDR *xdrs;
struct nfssattr *sa;
{
  
  if (xdr_u_long(xdrs, &sa->sa_mode) &&
      xdr_u_long(xdrs, &sa->sa_uid) &&
      xdr_u_long(xdrs, &sa->sa_gid) &&
      xdr_u_long(xdrs, &sa->sa_size) &&
      xdr_timeval(xdrs, &sa->sa_atime) &&
      xdr_timeval(xdrs, &sa->sa_mtime) ) {
#		ifdef NFSDEBUG
    printf( 
	   "xdr_sattr: %s mode %o uid %d gid %d size %d\n",
	   xdropnames[(int)xdrs->x_op], sa->sa_mode, sa->sa_uid,
	   sa->sa_gid, sa->sa_size);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_sattr: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}

struct xdr_discrim attrstat_discrim[2] = {
  { (int)NFS_OK, xdr_fattr },
  { __dontcare__, NULL_xdrproc_t }
};

/*
 * Reply status with file attributes
 */
bool_t
  xdr_attrstat(xdrs, ns)
XDR *xdrs;
struct nfsattrstat *ns;
{
  
  if (xdr_union(xdrs, (enum_t *)&(ns->ns_status),
		(caddr_t)&(ns->ns_attr), attrstat_discrim, xdr_void) ) {
#		ifdef NFSDEBUG
    printf( "xdr_attrstat: %s stat %d\n",
	   xdropnames[(int)xdrs->x_op], ns->ns_status);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_attrstat: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}


struct xdr_discrim rdlnres_discrim[2] = {
  { (int)NFS_OK, xdr_srok },
  { __dontcare__, NULL_xdrproc_t }
};


#define	nextdp(dp)	((struct direct *)((int)(dp) + (dp)->d_reclen))
#undef DIRSIZ
#define DIRSIZ(dp)	(sizeof(struct direct) - MAXNAMLEN + (dp)->d_namlen)


#define	roundtoint(x)	(((x) + (sizeof(int) - 1)) & ~(sizeof(int) - 1))
#define	reclen(dp)	roundtoint(((dp)->d_namlen + 1 + sizeof(u_long) +\
				    2 * sizeof(u_short)))


/*
 * Arguments for directory operations
 */
bool_t
  xdr_diropargs(xdrs, da)
XDR *xdrs;
struct nfsdiropargs *da;
{
  
  if (xdr_fhandle(xdrs, &da->da_fhandle) &&
      xdr_string(xdrs, &da->da_name, NFS_MAXNAMLEN) ) {
#		ifdef NFSDEBUG
    printf( "xdr_diropargs: %s '%s'\n",
	   xdropnames[(int)xdrs->x_op], da->da_name);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_diropargs: FAILED\n");
#	endif
  return (FALSE);
}


/*
 * arguments to create and mkdir
 */
bool_t
  xdr_creatargs(xdrs, argp)
XDR *xdrs;
struct nfscreatargs *argp;
{
  
  if (xdr_diropargs(xdrs, &argp->ca_da) &&
      xdr_sattr(xdrs, &argp->ca_sa) ) {
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_creatargs: FAILED\n");
#	endif
  return (FALSE);
}


/*
 * NFS_OK part of statfs operation
 */
xdr_fsok(xdrs, fsok)
     XDR *xdrs;
     struct nfsstatfsok *fsok;
{
  
  if (xdr_long(xdrs, (long *)&fsok->fsok_tsize) &&
      xdr_long(xdrs, (long *)&fsok->fsok_bsize) &&
      xdr_long(xdrs, (long *)&fsok->fsok_blocks) &&
      xdr_long(xdrs, (long *)&fsok->fsok_bfree) &&
      xdr_long(xdrs, (long *)&fsok->fsok_bavail) ) {
#		ifdef NFSDEBUG
    printf( 
	   "xdr_fsok: %s tsz %d bsz %d blks %d bfree %d bavail %d\n",
	   xdropnames[(int)xdrs->x_op], fsok->fsok_tsize,
	   fsok->fsok_bsize, fsok->fsok_blocks, fsok->fsok_bfree,
	   fsok->fsok_bavail);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_fsok: FAILED\n");
#	endif
  return (FALSE);
}

struct xdr_discrim statfs_discrim[2] = {
  { (int)NFS_OK, xdr_fsok },
  { __dontcare__, NULL_xdrproc_t }
};




struct xdr_discrim rdres_discrim[2] = {
  { (int)NFS_OK, xdr_rrok },
  { __dontcare__, NULL_xdrproc_t }
};

/*
 * Reply from remote read
 */
bool_t
  xdr_rdresult(xdrs, rr)
XDR *xdrs;
struct nfsrdresult *rr;
{
  
#	ifdef NFSDEBUG
  printf( "xdr_rdresult: %s\n", xdropnames[(int)xdrs->x_op]);
#	endif
  if (xdr_union(xdrs, (enum_t *)&(rr->rr_status),
		(caddr_t)&(rr->rr_ok), rdres_discrim, xdr_void) ) {
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_rdresult: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}


struct xdr_discrim diropres_discrim[2] = {
  { (int)NFS_OK, xdr_drok },
  { __dontcare__, NULL_xdrproc_t }
};

/*
 * Results from directory operation 
 */
bool_t
  xdr_diropres(xdrs, dr)
XDR *xdrs;
struct nfsdiropres *dr;
{
  if (xdr_union(xdrs, (enum_t *)&(dr->dr_status),
		(caddr_t)&(dr->dr_drok), diropres_discrim, xdr_void) ) {
#		ifdef NFSDEBUG
    printf( "xdr_diropres: %s stat %d\n",
	   xdropnames[(int)xdrs->x_op], dr->dr_status);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_diropres: FAILED\n");
#	endif
  return (FALSE);
}


/*
 * File attributes
 */
bool_t
  xdr_fattr(xdrs, na)
XDR *xdrs;
register struct nfsfattr *na;
{
  register long *ptr;
  
#	ifdef NFSDEBUG
  printf( "xdr_fattr: %s\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  if (xdrs->x_op == XDR_ENCODE) {
    ptr = XDR_INLINE(xdrs, 17 * BYTES_PER_XDR_UNIT);
    if (ptr != NULL) {
      IXDR_PUT_ENUM(ptr, na->na_type);
      IXDR_PUT_LONG(ptr, na->na_mode);
      IXDR_PUT_LONG(ptr, na->na_nlink);
      IXDR_PUT_LONG(ptr, na->na_uid);
      IXDR_PUT_LONG(ptr, na->na_gid);
      IXDR_PUT_LONG(ptr, na->na_size);
      IXDR_PUT_LONG(ptr, na->na_blocksize);
      IXDR_PUT_LONG(ptr, na->na_rdev);
      IXDR_PUT_LONG(ptr, na->na_blocks);
      IXDR_PUT_LONG(ptr, na->na_fsid);
      IXDR_PUT_LONG(ptr, na->na_nodeid);
      IXDR_PUT_LONG(ptr, na->na_atime.tv_sec);
      IXDR_PUT_LONG(ptr, na->na_atime.tv_usec);
      IXDR_PUT_LONG(ptr, na->na_mtime.tv_sec);
      IXDR_PUT_LONG(ptr, na->na_mtime.tv_usec);
      IXDR_PUT_LONG(ptr, na->na_ctime.tv_sec);
      IXDR_PUT_LONG(ptr, na->na_ctime.tv_usec);
      return (TRUE);
    }
  } else {
    ptr = XDR_INLINE(xdrs, 17 * BYTES_PER_XDR_UNIT);
    if (ptr != NULL) {
      na->na_type = IXDR_GET_ENUM(ptr, enum nfsftype);
      na->na_mode = IXDR_GET_LONG(ptr);
      na->na_nlink = IXDR_GET_LONG(ptr);
      na->na_uid = IXDR_GET_LONG(ptr);
      na->na_gid = IXDR_GET_LONG(ptr);
      na->na_size = IXDR_GET_LONG(ptr);
      na->na_blocksize = IXDR_GET_LONG(ptr);
      na->na_rdev = IXDR_GET_LONG(ptr);
      na->na_blocks = IXDR_GET_LONG(ptr);
      na->na_fsid = IXDR_GET_LONG(ptr);
      na->na_nodeid = IXDR_GET_LONG(ptr);
      na->na_atime.tv_sec = IXDR_GET_LONG(ptr);
      na->na_atime.tv_usec = IXDR_GET_LONG(ptr);
      na->na_mtime.tv_sec = IXDR_GET_LONG(ptr);
      na->na_mtime.tv_usec = IXDR_GET_LONG(ptr);
      na->na_ctime.tv_sec = IXDR_GET_LONG(ptr);
      na->na_ctime.tv_usec = IXDR_GET_LONG(ptr);
      return (TRUE);
    }
  }
  if (xdr_enum(xdrs, (enum_t *)&na->na_type) &&
      xdr_u_long(xdrs, &na->na_mode) &&
      xdr_u_long(xdrs, &na->na_nlink) &&
      xdr_u_long(xdrs, &na->na_uid) &&
      xdr_u_long(xdrs, &na->na_gid) &&
      xdr_u_long(xdrs, &na->na_size) &&
      xdr_u_long(xdrs, &na->na_blocksize) &&
      xdr_u_long(xdrs, &na->na_rdev) &&
      xdr_u_long(xdrs, &na->na_blocks) &&
      xdr_u_long(xdrs, &na->na_fsid) &&
      xdr_u_long(xdrs, &na->na_nodeid) &&
      xdr_timeval(xdrs, &na->na_atime) &&
      xdr_timeval(xdrs, &na->na_mtime) &&
      xdr_timeval(xdrs, &na->na_ctime) ) {
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_fattr: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}



/*
 * Status OK portion of remote read reply
 */
bool_t
  xdr_rrok(xdrs, rrok)
XDR *xdrs;
struct nfsrrok *rrok;
{
  
  if (xdr_fattr(xdrs, &rrok->rrok_attr)) {
    /* client side */
    if (xdr_bytes(xdrs, &rrok->rrok_data,
		  (u_int *)&rrok->rrok_count, NFS_MAXDATA) ) {
#				ifdef NFSDEBUG
      printf( 
	     "xdr_rrok: %s %d addr %x\n",
	     xdropnames[(int)xdrs->x_op],
	     rrok->rrok_count, rrok->rrok_data);
#				endif
      return (TRUE);
    }
    
  }
#	ifdef NFSDEBUG
  printf( "xdr_rrok: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}


/*
 * NFS_OK part of read sym link reply union
 */
bool_t
  xdr_srok(xdrs, srok)
XDR *xdrs;
struct nfssrok *srok;
{
  
  if (xdr_bytes(xdrs, &srok->srok_data, (u_long *)&srok->srok_count,
		NFS_MAXPATHLEN) ) {
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_srok: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}

/*
 * Time structure
 */
bool_t
  xdr_timeval(xdrs, tv)
XDR *xdrs;
struct timeval *tv;
{
  
  if (xdr_long(xdrs, &tv->tv_sec) &&
      xdr_long(xdrs, &tv->tv_usec) ) {
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_timeval: FAILED\n");
#	endif
  return (FALSE);
}


/*
 * NFS_OK part of directory operation result
 */
bool_t
  xdr_drok(xdrs, drok)
XDR *xdrs;
struct nfsdrok *drok;
{
  if (xdr_fhandle(xdrs, &drok->drok_fhandle)) {
    if (xdr_fattr(xdrs, &drok->drok_attr)) {
      return (TRUE);
    }
  }
#	ifdef NFSDEBUG
  printf( "xdr_drok: FAILED\n");
#	endif
  return (FALSE);
}

/* 
 * reply from mount
 */
struct xdr_discrim fhstatus_discrim[2] = {
  { (int) NFS_OK, xdr_fhandle },
  { __dontcare__, NULL_xdrproc_t }
};

bool_t
  xdr_fhstatus(xdrs, rm)
XDR *xdrs;
struct fhstatus *rm;
{
  if (xdr_union(xdrs, (enum_t *) &(rm->fhs_status),
		(caddr_t) &(rm->fhs_fh), fhstatus_discrim, xdr_void) ) {
    return TRUE;
  }
  
  return FALSE;
}


bool_t
  xdr_linkargs(xdrs, la)
XDR *xdrs;
struct nfslinkargs *la;
{
  if (xdr_fhandle(xdrs, &la->la_from) &&
      xdr_diropargs(xdrs, &la->la_to)) {
    return (TRUE);
  }
  return (FALSE);
}


bool_t
  xdr_slargs(xdrs, la)
XDR  *xdrs;
struct nfsslargs *la;
{
  
  if (xdr_diropargs(xdrs, &la->sla_from)  &&
      xdr_string(xdrs, &la->sla_tnm, NFS_MAXNAMLEN) &&
      xdr_sattr(xdrs, &la->sla_sa)) {
    return(TRUE);
  }
  return(FALSE);
}

bool_t 
  xdr_rnmargs(xdrs, la)
XDR    *xdrs;
struct nfsrnmargs *la;
{
  
  if (xdr_diropargs(xdrs, &la->rna_from) &&
      xdr_diropargs(xdrs, &la->rna_to)) {
    return(TRUE);
  }
  return(FALSE);
}


bool_t
  xdr_saargs(xdrs, la)
XDR   *xdrs;
struct nfssaargs *la;
{
  if (xdr_fhandle(xdrs, &la->saa_fh) &&
      xdr_sattr(xdrs, &la->saa_sa)) {
    return(TRUE);
  }
  return(FALSE);
}

bool_t 
  xdr_rdlnres(xdrs, rl)
XDR   *xdrs;
struct nfsrdlnres *rl;
{
  
  if (xdr_union(xdrs, (enum_t *)&(rl->rl_status),
		(caddr_t)&(rl->rl_srok), rdlnres_discrim, xdr_void)) {
    return(TRUE);
  }
  return(FALSE);
}


bool_t
  xdr_nfsrddirargs(xdrs, objp)
XDR *xdrs;
struct nfsrddirargs *objp;
{
  if (!xdr_fhandle(xdrs, &objp->rda_fh)) {
    return (FALSE);
  }
  if (!xdr_u_long(xdrs, &objp->rda_offset)) {
    return (FALSE);
  }
  if (!xdr_u_long(xdrs, &objp->rda_count)) {
    return (FALSE);
  }
  return (TRUE);
}
extern bool_t xdr_char();


bool_t
  xdr_filename(xdrs, objp)
XDR *xdrs;
char  *objp;
{
  if (!xdr_string(xdrs, objp, NFS_MAXNAMLEN)) {
    return (FALSE);
  }
  return (TRUE);
}

bool_t
  xdr_nfscookie(xdrs, objp)
XDR *xdrs;
nfscookie objp;
{
  if (!xdr_opaque(xdrs, objp, NFS_COOKIESIZE)) {
    return (FALSE);
  }
  return (TRUE);
}



bool_t
  xdr_entry(xdrs, objp)
XDR *xdrs;
entry *objp;
{
  if (!xdr_u_int(xdrs, &objp->fileid)) {
    return (FALSE);
  }
  if (!xdr_filename(xdrs, &objp->name)) {
    return (FALSE);
  }
  if (!xdr_nfscookie(xdrs, objp->cookie)) {
    return (FALSE);
  }
  if (!xdr_pointer(xdrs, (char **)&objp->nextentry, sizeof(entry), xdr_entry)) {
    return (FALSE);
  }
  return (TRUE);
}




bool_t
  xdr_dirlist(xdrs, objp)
XDR *xdrs;
dirlist *objp;
{
  if (!xdr_pointer(xdrs, (char **)&objp->entries, sizeof(entry), xdr_entry)) {
    return (FALSE);
  }
  if (!xdr_bool(xdrs, &objp->eof)) {
    return (FALSE);
  }
  return (TRUE);
}


struct xdr_discrim nfsrddirres_discrim[2] = {
  { (int)NFS_OK, xdr_dirlist },
  { __dontcare__, NULL_xdrproc_t }
};
bool_t
  xdr_nfsrddirres(xdrs, ns)
XDR *xdrs;
struct readdirres *ns;
{
  
  if (xdr_union(xdrs, (enum_t *)&(ns->status),
		(caddr_t)&(ns->readdirres_u.reply), nfsrddirres_discrim, xdr_void) ) {
#		ifdef NFSDEBUG
    printf( "xdr_nfsrddirres: %s stat %d\n",
	   xdropnames[(int)xdrs->x_op], ns->rd_status);
#		endif
    return (TRUE);
  }
#	ifdef NFSDEBUG
  printf( "xdr_nfsrddirres: %s FAILED\n",
	 xdropnames[(int)xdrs->x_op]);
#	endif
  return (FALSE);
}



