/* 
 * arp.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "xkernel.h"
#include "eth.h"
#include "ip.h"
#include "arp.h"
#include "arp_internal.h"

int tracearpp;

static char *codenames[] = {
  "impossible",
  "req",
  "rply",
  "rreq",
  "rrply"
};

/* global data of arp protocol */

static	ETHaddr arp_me = { 0x0806 };
static	ETHaddr rarp_me ={ 0x8035 };
static	ETHhost myeaddr;
static	struct arppak arp_pak;
static	struct arpent arp_table[ARP_TAB];

/* misc */

#ifdef XSIMUL
extern ETHhost BCAST;
#else
static ETHhost BCAST = { 0xffff, 0xffff, 0xffff};
#endif
IPhost	myipaddr;
static XObj ARP=0;
#define ETH (ARP->down[0])

arp_print(s, m)
char *s;
register arppak *m;
{
  printf("%s arp %s (%d) message:\n", s,
    codenames[ntohs(m->arp_op)], ntohs(m->arp_op));
  printf("  source %4x.%4x.%4x @ %d.%d.%d.%d\n",
    m->arp_sha.high, m->arp_sha.mid, m->arp_sha.low,
    m->arp_spa.a, m->arp_spa.b, m->arp_spa.c, m->arp_spa.d);
  printf("  target %4x.%4x.%4x @ %d.%d.%d.%d\n",
    m->arp_tha.high, m->arp_tha.mid, m->arp_tha.low,
    m->arp_tpa.a, m->arp_tpa.b, m->arp_tpa.c, m->arp_tpa.d);
}

arp_init(self)
XObj self;
{
  int  	i;
  Part	part[2];

#ifdef XSIMUL
  /* Check the ROM */
  char **rp;
  int inetaddr;
  i = 0;
  for (rp = rom; *rp; rp += 3) {
    if (!strcmp(rp[0], "arp")) {
      inetaddr = inet_addr(rp[1]);
      arp_table[i].arp_Iad = *(IPhost *) &inetaddr;
      arp_table[i].arp_Ead = *(ETHhost *)(rp[2]);
      arp_table[i].arp_state = ARP_RSLVD;
      TRACE4(arpp, 5, "arp: loaded (%d.%d.%d.%d) from rom file",
	    arp_table[i].arp_Iad.a, arp_table[i].arp_Iad.b,
	    arp_table[i].arp_Iad.c, arp_table[i].arp_Iad.d);
      i++;
    }
  }
#endif
  ARP = self;
  if (arp_table[0].arp_state == ARP_FREE) {
    arp_table[0].arp_state = ARP_ALLOC;
    arp_table[1].arp_state = ARP_ALLOC;
    arp_table[2].arp_state = ARP_ALLOC;
  }
  init_partlist(part, 1, ETHaddr);
  set_part(part, 0, arp_me);
  x_openenable(ARP, ARP->down[0], part);
  set_part(part, 0, rarp_me);
  x_openenable(ARP, ARP->down[0], part);
  for (i=1; i<ARP_TAB; i++) {
    InitSemaphore(&arp_table[i].s, 0);
  }
  if (arp_table[0].arp_state == ARP_ALLOC && arp_learn_me() == -1) {
    Kabort("cannot learn my IP address");
    return(-1);
  }
  /* initialize template for arp packets */

  arp_pak.arp_hrd=htons(1);
  arp_pak.arp_prot=0x0800;
  arp_pak.arp_hlen=htons(6);
  arp_pak.arp_plen=htons(4);
  arp_pak.arp_sha = arp_table[0].arp_Ead;
  arp_pak.arp_spa = arp_table[0].arp_Iad;
  myipaddr = arp_table[0].arp_Iad;

  TRACE4(arpp, 1, "Using IP address %d.%d.%d.%d",
    myipaddr.a, myipaddr.b, myipaddr.c, myipaddr.d);
  return(0);
}

/*ARGSUSED*/
Sessn arp_open(self,hlp, p)
XObj  self;
XObj  hlp;
Part *p;
{
  assert(x_is_protocol(hlp));
  assert(x_is_protocol(self));
  return x_createsession(hlp, self, 1);
}

arp_close(s)
XObj	s;
{
  TRACE1(arpp, 1, "Close arp session %x", s);
  assert(x_is_session(s));
  if (--s->rcnt > 0) {
    TRACE1(arpp, 2, "Still referenced %d times", s->rcnt);
  } else {
    TRACE0(arpp, 2, "Destroying the session");
    x_destroysession(s);
  }
  return(0);
}

arp_demux(self,s, msg)
XObj	self;
XObj	s;
Msg	msg;
{
  Msg	rep_msg;
  struct arppak	reply;
  struct arppak Amsg;
#ifdef ANSWER_RARPS
  int pos;
#endif

  assert(x_is_session(s));
  assert(x_is_protocol(self));

  msg_peek(msg, 0, ARP_HLEN, (char *)&Amsg);

  IFTRACE(arpp, 3) arp_print("received", &Amsg);
  switch(ntohs(Amsg.arp_op)) {
    case ARP_REQ:
      save_binding(&Amsg.arp_sha, &Amsg.arp_spa);    
      if (! bcmp((char *)&Amsg.arp_tpa, (char *)&myipaddr, sizeof(IPhost))) {
	bcopy((char *)&arp_pak, (char *)&reply, ARP_HLEN);
        /* Rice change */
        reply.arp_tha = Amsg.arp_sha;
        reply.arp_tpa = Amsg.arp_spa;
        /* end Rice change */
	reply.arp_op = htons(ARP_RPLY);
        msg_make_allstack(rep_msg,MSG_SSIZE,(char *)&reply,ARP_HLEN);
	TRACE2(arpp, 3, "replying with arp message with op %d (%s)", 
	  ntohs(reply.arp_op), codenames[ntohs(reply.arp_op)]);
	x_push(s,rep_msg,(Msg *)0);
      }
      break;
  
    case ARP_RPLY:
      save_binding(&Amsg.arp_sha, &Amsg.arp_spa);
      break;
  
    case ARP_RREQ:
#ifdef ANSWER_RARPS
      pos = arp_lookup_eth(&Amsg.arp_sha);
      if (pos != -1 && arp_table[pos].arp_state == ARP_RSLVD) {
	reply.arp_hrd=htons(1);
	reply.arp_prot=htons(0x0800);
	reply.arp_hlen=htons(6);
	reply.arp_plen=htons(4);
	reply.arp_op = htons(ARP_RRPLY);
	reply.arp_sha = arp_table[0].arp_Ead;  
	reply.arp_spa = arp_table[0].arp_Iad;  
	reply.arp_tha = arp_table[pos].arp_Ead;  
	reply.arp_tpa = arp_table[pos].arp_Iad;  
        msg_make_allstack(rep_msg,MSG_SSIZE,(char *)&reply,ARP_HLEN);
	TRACE1(arpp, 3, "replying with arp message with op %d", 
	  ntohs(reply.arp_op));
	x_push(s,rep_msg,(Msg *)0);
      }
#endif
      break;
  
    case ARP_RRPLY:
      save_binding(&Amsg.arp_sha, &Amsg.arp_spa);
      save_binding(&Amsg.arp_tha, &Amsg.arp_tpa);
      break;
  
    default:
      {/*do nothing*/}
      break;
  }
  IFTRACE(arpp,3) {
    printf("x_close 1\n");
    x_printxobj(s);
  }
  x_close(s);
  msg_free(msg);
  return(0);
}

arp_timeout(s)
XObj	s;
{
  struct  arpstate *arp_state;
  Msg	msg;

  TRACE2(arpp, 3, "Arp timeout s = %x, state = %x", s, s->state);
  arp_state = (struct arpstate *) s->state;
  if (arp_table[arp_state->pos].arp_state == ARP_RSLVD) {
    TRACE1(arpp, 5, "Arp session %x already resolved", s);
    event_remove(arp_timeout, (int)s);
    x_close(s);
    return(x_close(arp_state->down_s));
  }
  if ((arp_state->type==ARP_ARP && (++arp_state->tries) > ARP_RTRY) ||
      (arp_state->type==ARP_RARP && (++arp_state->tries) > ARP_RRTRY)) {
    event_remove(arp_timeout, (int)s);
    x_close(s);
    TRACE0(arpp, 1, "arp timeout: quiting");
    /* wake up waiters */
    VAll(&arp_table[arp_state->pos].s);
    arp_table[arp_state->pos].arp_state = ARP_FREE;
    return(x_close(arp_state->down_s));
  }
  TRACE0(arpp, 3, "arp timeout: trying again");
  msg_make_allstack(msg,MSG_SSIZE,(char *)&arp_state->req_msg,ARP_HLEN);
  IFTRACE(arpp, 3) arp_print("sending", &arp_state->req_msg);
  return(x_push(arp_state->down_s, msg,(Msg *)0));
}

arp_controlprotl(self,op, buf, len)
XObj 	self;
int	op;
char	*buf;
int	len;
{
  IPhost  tmpip;
  ETHhost tmpeth;
  int reply;
  Sessn s;
  assert(x_is_protocol(self));
  switch (op) {

  case RESOLVE:
    checkLen(len, sizeof(ETHhost));
    s = x_createsession(self, self,1);
    tmpip = *(IPhost *)buf;
    reply = arp_resolve(s, &tmpip, (ETHhost *)buf);
    IFTRACE(arpp,3) {
      printf("x_close 2\n");
      x_printxobj(s);
    }
    x_close(s);
    break;

    case RRESOLVE:
    checkLen(len, sizeof(ETHhost));
    s = x_createsession(self, self, 1);
    tmpeth = *(ETHhost *)buf;
    reply = arp_rresolve(s, (IPhost *)buf, &tmpeth, 0);
    IFTRACE(arpp,3) {
      printf("x_close 3\n");
      x_printxobj(s);
    }
    x_close(s);
    break;

  case ARP_INSTALL:
    {
      struct foo {
	IPhost ipaddr;
	ETHhost  eaddr;
      } *ptr;
      checkLen(len, sizeof *ptr);
      ptr = (struct foo *) buf;
      save_binding(&ptr->eaddr, &ptr->ipaddr);
      reply = 0;
    }
    break;

  case ARP_GETIPINTERFACES:
    checkLen(len, sizeof(int));
    *(int *)buf = ETH->index;
    reply = sizeof(int);
    break;

  case ARP_GETIPADDRS:
    checkLen(len, sizeof(IPhost));
    *(IPhost *)buf = myipaddr;
    reply = sizeof(IPhost);
    break;

  default:
    x_errno = INVALID_OPCODE;
    reply = -1;
  }
  TRACE2(arpp, 3, "Arp control %s returns %d", 
    op == RESOLVE ? "resolve":
    op == ARP_GETIPINTERFACES ? "getipinterfaces" :
    op == ARP_GETIPADDRS ? "getipaddrs" :
    op == RRESOLVE ? "rresolve" :
    "unknown", reply);
  return(reply);
}

arp_controlsessn(s, op, buf, len)
XObj	s;
int	op;
char	*buf;
int	len;
{
  IPhost   tmpip;
  ETHhost  tmpeth;

  assert(x_is_session(s));
  switch (op) {
  case RESOLVE:
    checkLen(len, sizeof(ETHhost));
    tmpip = *(IPhost *)buf;
    return arp_resolve(s, &tmpip, (ETHhost *)buf);

  case RRESOLVE:
    checkLen(len, sizeof(ETHhost));
    tmpeth = *(ETHhost *)buf;
    return arp_rresolve(s, (IPhost *)buf, &tmpeth, 0);

  case ARP_INSTALL:
    {
      struct foo {
	IPhost ipaddr;
	ETHhost  eaddr;
      } *ptr;
      checkLen(len, sizeof *ptr);
      ptr = (struct foo *) buf;
      save_binding(&ptr->eaddr, &ptr->ipaddr);
      return 0;
    }

    default:
      x_errno = INVALID_OPCODE;
      return -1;
  }
}

/************************************************
/*
/* Arp Internal Routines
/*
/************************************************/

arp_resolve(s, IPad, Ead)	/* main "arp" interface */
Sessn	s;
IPhost	*IPad;
ETHhost	*Ead;
{
  Part	part[3];
  struct arpstate *arp_state;
  Msg	req_msg;
  int	pos, reply;
  ETHaddr remoteaddr;
  
  TRACE4(arpp, 3, "Resolving %d.%d.%d.%d", IPad->a, IPad->b, IPad->c, IPad->d);
  if ((pos=arp_lookup_ip(IPad)) != -1 && arp_table[pos].arp_state==ARP_RSLVD) {
    *Ead = arp_table[pos].arp_Ead;
    TRACE0(arpp, 4, "Already had it");
    reply = 0;
  } else if (pos != -1) {
    /* someone else must have requested it */
    TRACE0(arpp, 4, "Someone else requested it");
    reply = wait_for_answer(pos);
    *Ead = arp_table[pos].arp_Ead;
  } else if (! IP_NETEQ(*IPad, myipaddr)) {
    TRACE0(arpp, 4, "Not on this network, failing");
    reply = -1;
  } else {
    TRACE0(arpp, 4, "Sending request message");
    arp_state = (struct arpstate *) malloc(sizeof(struct arpstate));
    if ((pos=arp_get_pos()) == -1) {
      Kabort("panic in arp_resolve\n");
      /*NOTREACHED*/
    }
    arp_table[pos].arp_Iad = *IPad;
    init_partlist(part, 2, ETHaddr);
    remoteaddr.type = arp_me.type;
    remoteaddr.host = BCAST;
    set_part(part, 0, arp_me);
    set_part(part, 1, remoteaddr);

    arp_state->down_s = x_open(ARP, ARP->down[0], part);
    arp_state->pos = pos;
    bcopy((char *)&arp_pak, (char *)&arp_state->req_msg, ARP_HLEN);
    arp_state->req_msg.arp_op = htons(ARP_REQ);
    arp_state->req_msg.arp_tpa = *IPad;
    arp_state->tries = 1;
    arp_state->type = ARP_ARP;
    msg_make_allstack(req_msg,MSG_SSIZE,(char *)&arp_state->req_msg,ARP_HLEN);
    s->state = (char *) arp_state;
    s->rcnt++;
    event_register(arp_timeout, (int)s, ARP_TIME, EV_REPEAT);
    x_push(arp_state->down_s, req_msg,(Msg *)0);
    reply = wait_for_answer(pos);
    *Ead = arp_table[pos].arp_Ead;
  }
  TRACE5(arpp, 3, "Resolved %d.%d.%d.%d, result %d",
    IPad->a, IPad->b, IPad->c, IPad->d, reply);
  return(reply);
}

arp_rresolve(s, IPad, Ead, isme)		/* main "rarp" interface */
Sessn	s;
IPhost	*IPad;
ETHhost	*Ead;
int isme;
{
  Part	part[3];
  struct arpstate *arp_state;
  Msg	req_msg;
  int	pos, reply;
  ETHaddr remoteaddr;
  
  TRACE3(arpp, 3, "RResolving %4x.%4x.%4x", Ead->high, Ead->mid, Ead->low);
  if ((pos=arp_lookup_eth(Ead)) != -1 && arp_table[pos].arp_state==ARP_RSLVD) {
    *IPad = arp_table[pos].arp_Iad;
    TRACE0(arpp, 4, "Already had it");
    TRACE4(arpp, 4, "found %d.%d.%d.%d", IPad->a, IPad->b, IPad->c, IPad->d);
    reply = 0;
  } else if (pos != -1 && !isme) {
    /* someone else must have requested it */
    TRACE0(arpp, 4, "Someone else requested it");
    reply = wait_for_answer(pos);
    *IPad = arp_table[pos].arp_Iad;
#define DONTRARPOTHERS
#ifdef DONTRARPOTHERS
  } else if (!isme) {
    reply = -1;
#endif
  } else {
    TRACE0(arpp, 4, "Sending request message");
    if (isme) {
      pos = 0; 
    } else {
      if ((pos=arp_get_pos()) == -1) {
	Kabort("panic in arp_resolve\n");
	/*NOTREACHED*/
      }
    }
    arp_table[pos].arp_Ead = *Ead;
    remoteaddr.type = rarp_me.type;
    remoteaddr.host = BCAST;
    init_partlist(part, 2, ETHaddr);
    set_part(part, 0, rarp_me);
    set_part(part, 1, remoteaddr);
    arp_state = (struct arpstate *) malloc(sizeof(struct arpstate));
    arp_state->req_msg.arp_hrd=htons(1);
    arp_state->req_msg.arp_prot=htons(0x0800);
    arp_state->req_msg.arp_hlen=htons(6);
    arp_state->req_msg.arp_plen=htons(4);
    arp_state->req_msg.arp_op = htons(ARP_RREQ);
    arp_state->down_s = x_open(ARP, ARP->down[0], part);
    arp_state->pos = pos;
    arp_state->req_msg.arp_sha = myeaddr;
    arp_state->req_msg.arp_tha = *Ead;
    if (isme) {
      bzero((char *)&arp_state->req_msg.arp_spa, sizeof (IPhost));
    } else {
      arp_state->req_msg.arp_spa = myipaddr;
    }
    bzero((char *)&arp_state->req_msg.arp_tpa, sizeof (IPhost));
    arp_state->tries = 1;
    arp_state->type = ARP_RARP;
    msg_make_allstack(req_msg,MSG_SSIZE,(char *)&arp_state->req_msg,ARP_HLEN);
    s->state = (char *) arp_state;
    s->rcnt++;
    event_register(arp_timeout, (int)s, ARP_TIME, EV_REPEAT);
    x_push(arp_state->down_s, req_msg,(Msg *)0);
    if (isme) {
      while (arp_table[pos].arp_state == ARP_ALLOC) kSwitch();
      reply = arp_table[pos].arp_state == ARP_RSLVD ? 0 : -1;
    } else {
      reply = wait_for_answer(pos);
    }
    *IPad = arp_table[pos].arp_Iad;
  }
  TRACE4(arpp, 3, "Resolved %4x.%4x.%4x, result %d",
    Ead->high, Ead->mid, Ead->low, reply);
  return reply;
}

arp_learn_me()			/* find my ip address */
{				/* put binding in position 0 of table */
				/* put broadcast binding in position 1 and 2 */
  Sessn s;
  x_controlprotl(ARP->down[0], GETMYADDR, (char *)&myeaddr, sizeof myeaddr);
  TRACE3(arpp, 3, "My eaddr = %4x %4x %4x", myeaddr.high, myeaddr.mid,
    myeaddr.low);
  arp_table[0].arp_Ead = myeaddr;
  arp_table[1].arp_Ead = BCAST;
  arp_table[2].arp_Ead = BCAST;
  s = x_createsession(ARP, ARP,1);
  if (arp_rresolve(s, &arp_table[0].arp_Iad, &arp_table[0].arp_Ead, 1) == 0) {
    assert (arp_table[0].arp_state == ARP_RSLVD);
    arp_table[1].arp_Iad = arp_table[0].arp_Iad;
    arp_table[2].arp_Iad = arp_table[0].arp_Iad;
    arp_table[1].arp_Iad.d = 0;
    arp_table[2].arp_Iad.d = 255;
    arp_table[1].arp_state = ARP_RSLVD;
    arp_table[2].arp_state = ARP_RSLVD;
    VAll(&arp_table[1].s);
    VAll(&arp_table[2].s);
    IFTRACE(arpp,3) {
      printf("x_close 4\n");
      x_printxobj(s);
    }
    x_close(s);
    return 0;
  } else {
    IFTRACE(arpp,3) {
      printf("x_close 5\n");
      x_printxobj(s);
    }
    x_close(s);
    return -1;
  }
}

wait_for_answer(pos)		/* block until answer arrives */
int	pos;
{
  P(&arp_table[pos].s);
  return arp_table[pos].arp_state == ARP_RSLVD ? 0 : -1;
}
  
/* wake people up */
save_binding(eth,ip)			/* put binding in table */
ETHhost	*eth;
IPhost	*ip;
{
  int	i,e;

  TRACE4(arpp, 4, "Saving %4x.%4x.%4x -> %X", eth->high, eth->mid, eth->low,
    *(long *)ip);
  if ((i=arp_lookup_ip(ip)) != -1 && arp_table[i].arp_state==ARP_RSLVD){
    TRACE0(arpp, 5, "save_binding: already have it");
    return(0);
  }

  e = arp_lookup_eth(eth);

  if (i == -1 && e == -1) {
    if ((i=arp_get_pos()) == -1) {
      Kabort("panic in save_binding\n");
      return(-1);
    }
    arp_table[i].arp_Iad = *ip;
    arp_table[i].arp_Ead = *eth;
    arp_table[i].arp_state = ARP_RSLVD;
    e = i;
  } else if (i != -1) {
    arp_table[i].arp_Ead = *eth;
    arp_table[i].arp_state = ARP_RSLVD;
    e = i;
  } else {
    arp_table[e].arp_Iad = *ip;
    arp_table[e].arp_state = ARP_RSLVD;
    i = e;
  }
  VAll(&arp_table[i].s);
  return(0);
}

arp_lookup_ip(IPad)			/* find IP addr in table */
IPhost	*IPad;
{
  int	i;

  for (i=0; i<ARP_TAB; i++)
    if (arp_table[i].arp_state != ARP_FREE &&
	!bcmp((char *)IPad, (char *)&arp_table[i].arp_Iad, sizeof(IPhost))) {
      return(i);
    }
  return(-1);
}

arp_lookup_eth(Ead)			/* find ETH addr in table */
ETHhost	*Ead;
{
  int	i;

  for (i=0; i<ARP_TAB; i++) {
    if (arp_table[i].arp_state != ARP_FREE &&
	!bcmp((char *)Ead, (char *)&arp_table[i].arp_Ead, sizeof(ETHhost))) {
      return(i);
    }
  }
  return(-1);
}

arp_get_pos()			/* get a position for binding */
{
  int	i;
  
  for (i=0; i<ARP_TAB; i++)
    if (arp_table[i].arp_state == ARP_FREE) {
      arp_table[i].arp_state = ARP_ALLOC;
      return(i);
    }
  /* normally replace some position */
  return(-1);
}



arp_getproc(p,type)
XObj p;
XObjType type;
{
  if (type == Protocol) {
    p->instantiateprotl = noop;
    p->init = arp_init;
    p->close = noop;
    p->push = noop;
    p->pop = noop;
    p->control = arp_controlprotl;
  } else {
    p->push = noop;
    p->pop = noop;
    p->instantiateprotl = noop;
    p->init = noop;
    p->close = arp_close;
    p->control = arp_controlsessn;
  }
  p->open = (Pfi) arp_open;
  p->openenable = noop;
  p->opendone = noop;
  p->closedone = noop;
  p->opendisable = noop;
  p->demux = arp_demux;
  p->getproc = arp_getproc;
}

