/* 
 * process.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "process.h"
#include "memory.h"

Process *IdleProcessPtr, Idle_process;
short DoSwitch;

/*
 * Process management
 */

Process *Active;		/* This will have to go in multi-p land */
Process *ProcessDescriptors;	/* Pointer to array of process descriptors */
Process *EndPds;		/* Ptr to the end of the process descriptors */
unsigned long MaxProcesses;

/* Free PD list needs to be synchronized for multiple processor because
 * PDs are allocated and freed as aliens in the IKC code.
 */
SyncQueue FreePdq;		/* Queue of free PDs */
Process *PdIndex;		/* Ptr to next pd to check on timer int. */

/* Ready queue head and active process pointer */
SyncQueue Readyq;
unsigned FreePds;		/* Count of number of free PD's */

InitProcesses()
{
  register Process *pd;
  TRACE0(processcreation, 1, "Allocate pds");
  Ms_AllocatePds();
  InitPDs();

  pd = &Idle_process;
  Active = pd;
  pd->link = NULL;
  pd->pid = 0; 
  pd->state = READY;
  pd->priority = NULL_PROC_PRIORITY; /* Lowest priority*/
  TRACE1(processswitch, 1, "Idle process aspace is %x",GetAddressableAspace());
  pd->aspace = GetAddressableAspace();
}

int lastPid = 0x1000;
#define NEXTPID() (++lastPid > 0x2000 ? lastPid = 0x1001 : lastPid)

/*forward*/
Process *ICreateProcess();

Process *CreateProcess(aspace, entry, extrastuff, priority, kernelP, nargs, firstarg)
Aspace *aspace;
int (*entry)(), extrastuff;
unsigned int priority;
int kernelP;
int nargs;
int firstarg;
{
  return ICreateProcess(aspace, entry, extrastuff, priority, kernelP, nargs, &firstarg);
}

/*
 * Create a process to execute in the kernel.
 */
Process *standbyProcessList = NULL;
int numKernelProcesses = 0;
int numFailedKernelCreates = 0;

Process *CreateKernelProcess(entry, priority, nargs, firstarg)
int (*entry)();
unsigned int priority;
int nargs, firstarg;
{
  register Process *pd;
  Process *result;
  int x = spl7();
#ifdef DEBUGRQ
  Ms_CheckOutRQ(Active);
#endif
  TRACE3(processcreation, 1,
    "CreateKernelProcess entry %x %dargs priority %d", entry, nargs, priority);
  if ((pd = standbyProcessList) == NULL) {
    if (numKernelProcesses <= 50) {
      ++numKernelProcesses;
      result = ICreateProcess(Idle_process.aspace, entry, 0, priority, 1,
	nargs, &firstarg);
    } else {
      ++numFailedKernelCreates;
      TRACE0(processcreation, 1, "CreateKernelProcess failed, too many");
      result = (Process *)0;
    }
  } else {
    /* re-use an existing one */
    TRACE2(processcreation, 3, 
      "Create kernel process reusing process #%x, entry %x", pd->pid, entry);
    standbyProcessList = pd->link;
    pd->link = NULL;
    pd->priority = priority;
#ifdef DEBUGRQ
    Ms_CheckOutRQ(pd);
#endif
    Ms_CreateProcess(Idle_process.aspace, pd, entry, 1, nargs, &firstarg);
#ifdef DEBUGRQ
    Ms_CheckOutRQ(pd);
#endif
    Ms_Addready(pd);
    result = pd;
  }
  splx(x);
  return(result);
}

/*
 * Destroy the current process
 */
DestroyProcess()
{
  register Process *pd = Active;
  TRACE1(processcreation, 1, "Destroying process number %x", pd->pid);
  disable();
#ifdef DEBUGRQ
  Ms_CheckOutRQ(pd);
#endif
  pd->state = DEAD;
#ifndef NDEBUG
  assert(pd->queuePtr == 0);
  pd->queuePtr = (SyncQueue *)1;
#endif
  if (pd->aspace == Idle_process.aspace) {
    /* this is a kernel process */
    TRACE0(processcreation, 2, "Putting it on standby list");
    pd->link = standbyProcessList;
    standbyProcessList = pd;
  } else {
    /* unlink it from the geneology structure */
    /* not done yet */
#ifdef __STDC__
    KernelStack();
#else
    KernelStack(0);
#endif
    TRACE1(processcreation, 4, "Calling Ms_DestroyProcess pd = %x", pd);
    Ms_DestroyProcess(pd);
    TRACE1(processcreation, 4, "Calling FreePd pd = %x", pd);
    FreePd(pd);
  }
  TRACE0(processcreation, 2, "Running next process");
  Ms_ActivateReadyqHead();
}

Wait(s)
register Semaphore *s;
{
  P(s);
}

Signal(s)
register Semaphore *s;
{
  V(s);
}

InitSemaphore(s, n)
register Semaphore *s;
unsigned n;
{
  s->count = n;
  Q_INIT(s);
}

VAll(s)
register Semaphore *s;
{
  register Process *p;
  register int sr = spl7();
  while (s->count < 0) {
    Q_REMOVEFIRST(s, p);
    if (p) Ms_Addready(p);
    s->count ++;
  }
  splx(sr);
}

Kabort(s)
register char *s;
{
  Ms_Kabort(s);
}

/*********************************************************
* Internal routines
**********************************************************/

InitPDs()
{
  register Process *pd;
  register unsigned i;

  /* Initialize process descriptors. */
  IdleProcessPtr = &Idle_process;
  pd = ProcessDescriptors;
  PdIndex = pd;
  FreePdq.head = NULL;
  FreePdq.tail = pd;

  for ( i = 0; i < MaxProcesses; ++i ) {
    pd->link = NULL;
    pd->pid = 0;
    pd->next = FreePdq.head;
    FreePdq.head = pd;
    Ms_InitPd(pd);
    ++pd;
  }
  FreePds = MaxProcesses;
}

Process *AllocatePd()
/* Allocate a process descriptor and return a pointer to same
 * if one is available, else return NULL.
 * This needs to be synchronized.
 */
{
  register Process *pd;
  int x = spl7();
  if( (pd = FreePdq.head) == NULL ) {
#ifdef LANCE
PrintQueues();
#endif
    splx(x);
    return( NULL );
  }
  FreePdq.head = pd->next;
  pd->next = NULL;
  --FreePds;
  splx(x);
  pd->pid = NEXTPID();
  Ms_InitPd(pd);
  return( pd );
}

FreePd(pd)
Process *pd;
{
  /*
   * Free up the process descriptor by setting its pid field to 0.
   */
  extern unsigned FreePds;
  int x = spl7();
  ++FreePds;
  TRACE2(processcreation, 3, "FreePD, pd = %x, FreePdq.head = %x", pd,
	 FreePdq.head);
  pd->next = FreePdq.head;
  FreePdq.head = pd;
  pd->pid = 0;
  splx(x);
}

Process *ICreateProcess(aspace, entry, extrastuff, priority, kernelP, nargs, argv)
Aspace *aspace;
int (*entry)(), extrastuff;
unsigned int priority;
int kernelP;
int nargs;
int *argv;
{
  register Process *pd;
  register int x;

  x = spl7();
  if ((pd = AllocatePd()) == NULL) {
    printf("Couldn't allocate descriptor for process\n");
    (void) splx(x);
    return(0);
  } 
  TRACE4(processcreation, 1, "Allocated pd %x pid %x entry %x aspace %x",
    pd, pd->pid, entry, aspace);
  pd->aspace = aspace;
  pd->extrastuff = extrastuff;
  pd->priority = priority > NULL_PROC_PRIORITY ? NULL_PROC_PRIORITY : priority;
  /* Link into the tree of processes */
  pd->link = NULL;
  pd->queuePtr = NULL;
  Ms_CreateProcess(aspace, pd, entry, kernelP, nargs, argv);
  Ms_Addready(pd);
  (void) splx(x);;
  return pd;
}

RemoveSemaphore(pd, sem)
Process *pd;
Semaphore *sem;
{
  int x = spl7();
  register Process *follow, *run;
  sem->count--;
  follow = 0;
  run = sem->head;
  while (run && run != pd) {
    follow = run;
    run = run->link;
  }
  assert(run == pd);
  if (follow) {
    follow->link = pd->link;
  } else {
    sem->head = pd->link;
  }
  if (sem->tail == &pd->link) {		/* It was the last one */
    if (follow) {			/* There is another one */
      sem->tail = &follow->link;
    } else {				/* No processes */
      assert(sem->count == 0);
      sem->tail = &sem->head;
    }
  }
  splx(x);
}

/*
 * Terminate an arbitrary process
 */
TerminateProcess(pd)
Process *pd;
{
  int x = (int) pd->queuePtr;
  if (pd == Active) DestroyProcess();
  switch (x) {
    case 0:
      /* no queue, do nothing */
      break;
    case 1:
      Ms_RemoveReady(pd);
      break;
    default:
      RemoveSemaphore(pd, (Semaphore *)x);
      break;
  }
  pd->state = DEAD;
  if (pd->aspace == Idle_process.aspace) {
    /* this is a kernel process */
    TRACE0(processcreation, 2, "Putting it on standby list");
    pd->link = standbyProcessList;
    standbyProcessList = pd;
  } else {
    Ms_DestroyProcess(pd);
    FreePd(pd);
  }
  
}

realV(s)
register Semaphore *s;
{
  register Process *p;
  register int sr = spl7();
#ifdef DEBUGRQ
    Ms_CheckOutRQ(Active);
#endif
  if (s->count <= 0) {
    Q_REMOVEFIRST(s, p);
    if (p) Ms_Addready(p);
  }
#ifdef DEBUGRQ
    Ms_CheckOutRQ(Active);
#endif
  splx(sr);
}




