;;; xwem-selections.el --- Support for X selections.

;; Copyright (C) 2004 by Free Software Foundation, Inc.

;; Author: Zajcev Evgeny <zevlg@yandex.ru>
;; Created: Wed May  5 17:06:41 MSD 2004
;; Keywords: xwem
;; X-CVS: $Id: xwem-selections.el,v 1.4 2005/01/01 04:43:00 youngs Exp $

;; This file is part of XWEM.

;; XWEM is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XWEM is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:

;; 

;;; Code:

(require 'xwem-load)
(require 'xwem-help)

(defgroup xwem-selections nil
  "Group to customize `xwem-selections'."
  :prefix "xwem-selections-"
  :group 'xwem)

(defcustom xwem-selections-maximum 20
  "Maximum number of saved selections."
  :type 'number
  :group 'xwem-selections)

;;; Internal variables


(defvar xwem-selections nil
  "Ring of saved selections.
Actually alist")

(defvar xwem-selection-xwin nil
  "X-Win used to operate on selections.")

(defun xwem-init-selections ()
  "Initialize selections mechanism."
  (setq xwem-selection-xwin
        (XCreateWindow (xwem-dpy) (xwem-rootwin)
                       0 0 1 1 0 nil nil nil
                       (make-X-Attr :override-redirect t
                                    :event-mask (Xmask-or XM-StructureNotify))))

  (X-Win-EventHandler-add-new xwem-selection-xwin 'xwem-selection-get nil
                              (list X-SelectionNotify))
  )

(defun xwem-selection-req (sel &optional targ prop)
  "Issue XConvertSelection."
  (unless targ
    (setq targ XA-string))

  (XConvertSelection (xwem-dpy)
                     (if (X-Atom-p sel) sel (XInternAtom (xwem-dpy) sel t))
                     (if (X-Atom-p targ) targ (XInternAtom (xwem-dpy) targ t))
                     (XInternAtom (xwem-dpy) (or prop "XWEM_SELECTION_PROPERTY") t)
                     xwem-selection-xwin)
  )

(defun xwem-selection-get (xdpy xwin xev)
  "On display XDPY and window XWIN process SelectionNotify event XEV."
  (xwem-message 'note "here prop=%d" (X-Atom-id (X-Event-xselection-property xev)))
  (let (target prov)
    (if (not (= (X-Atom-id (X-Event-xselection-property xev)) X-None))
        (progn
          (if (X-Atom-equal (X-Event-xselection-target xev) (XInternAtom xdpy "XA_TARGETS" t))
              (setq target XA-atom)
            (setq target (X-Event-xselection-target xev)))

          (setq prov (XGetWindowProperty xdpy (X-Event-xselection-requestor xev)
                                         (X-Event-xselection-property xev) nil nil nil target))

          (xwem-message 'note "Got prov=%S, prop=%S target=%S" prov (X-Atom-id (X-Event-xselection-property xev)) (X-Atom-id target)))
      )))

;;;###autoload(autoload 'xwem-help-cutbuffers "xwem-selections" "Display help about cutbuffers." t)
(define-xwem-command xwem-help-cutbuffers ()
  "Show help buffer about cutbuffers."
  (xwem-interactive)

  (xwem-help-display "cutbuffers"
    (insert "X cutbuffers:\n\n")
    (insert "NUMBER   VALUE\n")
    (insert "------   -----\n")
    (mapc (lambda (n)
            (let ((cbval (x-get-cutbuffer n)))
              (when cbval
                (insert (format "%-9d%S\n" n cbval)))))
          '(0 1 2 3 4 5 6 7))

    (insert "\n")

    (insert "XWEM selections:\n\n")
    (insert "NUMBER   VALUE\n")
    (insert "------   -----\n")
    (let ((nsel 0))
      (mapc (lambda (s)
              (insert (format "%-9d%S\n" nsel s))
              (incf nsel))
            xwem-selections))))

;;;###autoload(autoload 'xwem-copy-cutbuffer "xwem-selections" "Copy CUTBUFFER0 to `xwem-selections'." t)
(define-xwem-command xwem-copy-cutbuffer (&optional which-one)
  "Copy current cut"
  (xwem-interactive "p")

  (decf which-one)
  (let ((cb0 (x-get-cutbuffer which-one)))
    (if (not cb0)
        (xwem-message 'note "No active CUTBUFFER")
      (push cb0 xwem-selections)
      (xwem-message 'info "Copying CUTBUFFER%d: %S" which-one cb0))))

;;;###autoload(autoload 'xwem-paste-cutbuffer "xwem-selections" "Paste CUTBUFFER0 to `xwem-selections'." t)
(define-xwem-command xwem-paste-cutbuffer (&optional no-remove)
  "Paste's most recent cutbuffer from `xwem-selections' to selected client.
cutbuffer is removed from `xwem-selections', unless NO-REMOVE is non-nil."
  (xwem-interactive "_P")

  (if (null xwem-selections)
      (xwem-message 'info "No selections in `xwem-selections'")

    (xwem-kbd-force-mods-release)
    (mapc 'xwem-unread-command-event
          (if no-remove
              (car xwem-selections)
            (pop xwem-selections)))))


(provide 'xwem-selections)

;;; xwem-selections.el ends here
