(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: Tipe.m3                                               *)
(* Last Modified On Mon Mar  2 11:35:43 PST 1992 By kalsow     *)
(*      Modified On Tue Nov 27 22:16:56 1990 By muller         *)

MODULE Tipe;

IMPORT Value, ValueRep, Scope, Error, String, OpaqueType;
IMPORT MBuf, Token, Type, Decl, Scanner;
FROM Scanner IMPORT GetToken, Fail, Match, Match1, MatchID, cur;

TYPE
  T = Value.T BRANDED "Tipe.T" OBJECT 
        value : Type.T;
      OVERRIDES
        typeCheck   := Check;
	class       := MyClass;
        fingerprint := FPrinter;
        load        := ValueRep.NoLoader;
        write       := ValueRep.NoWriter;
        declare0    := Compile;
	toExpr      := ValueRep.NoExpr;
	toType      := ToType;
        typeOf      := ValueRep.TypeVoid;
      END;

PROCEDURE Parse (READONLY fail: Token.Set; att: Decl.Attributes) =
  VAR t: T;  fail2: Token.Set;  id: String.T;
  BEGIN
    IF att.isExternal THEN Error.Msg ("a type cannot be external"); END;
    IF att.isInline   THEN Error.Msg ("a type cannot be inline"); END;

    Match (Token.T.tTYPE, fail, Token.Set {Token.T.tSEMI});
    fail2 := fail + Token.Set {Token.T.tSEMI};

    WHILE (cur.token = Token.T.tIDENT) DO
      id := MatchID (fail2, Token.TypeStart + Token.Set {Token.T.tEQUAL,
                                              Token.T.tDOT, Token.T.tSUBTYPE});
      t := Create (id);
      t.unused := att.isUnused;
      t.obsolete := att.isObsolete;
      CASE cur.token OF
      | Token.T.tEQUAL =>
          GetToken (); (* = *)
	  t.value := Type.Parse (fail2);
      | Token.T.tSUBTYPE =>
          GetToken (); (* <: *)
          t.value := OpaqueType.New (Type.Parse (fail2));
      ELSE
          Fail ("missing \'=\' or \'<:\'", fail);
      END;
      Scope.Insert (t);
      Type.NoteDeclaration (t.value, t);
      Match1 (Token.T.tSEMI, fail);
    END;

  END Parse;

PROCEDURE Create (name: String.T): T =
  VAR t: T;
  BEGIN
    t := NEW (T);
    ValueRep.Init (t, name);
    t.readonly := TRUE;
    t.value    := NIL;
    RETURN t;
  END Create;

PROCEDURE Define (name: TEXT;  type: Type.T;  reserved: BOOLEAN) =
  VAR t: T;
  BEGIN
    t := Create (String.Add (name));
    t.value := type;
    Scope.Insert (t);
    Type.NoteDeclaration (type, t);
    IF (reserved) THEN Scanner.NoteReserved (t.name, t) END;
  END Define;

PROCEDURE Check (t: T;  <*UNUSED*> VAR cs: Value.CheckState) =
  BEGIN
    Type.Check (t.value);
  END Check;

PROCEDURE Compile (t: T): BOOLEAN =
  BEGIN
    Type.Compile (t.value);
    (************ the pre linker generates all named types...  **********
    Emit.OpF ("typedef @", t.value);
    Emit.OpN (" @;\n", t);
    *********************************************************************)
    RETURN TRUE;
  END Compile;

PROCEDURE MyClass (<*UNUSED*> t: T): Value.Class =
  BEGIN
    RETURN Value.Class.Type;
  END MyClass;

PROCEDURE FPrinter (t: T; map: Type.FPMap; wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "TYPE ");
    Type.Fingerprint (t.value, map, wr);
  END FPrinter;

PROCEDURE ToType (t: T): Type.T =
  BEGIN
    RETURN t.value;
  END ToType;

BEGIN
END Tipe.
