(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: AndExpr.m3                                            *)
(* Last modified on Tue Jan  7 08:18:20 PST 1992 by kalsow         *)
(*      modified on Thu Nov 29 03:30:36 1990 by muller         *)

MODULE AndExpr;

IMPORT Expr, ExprRep, Type, Bool, Error, Emit, EnumExpr, Temp, MBuf;

TYPE
  P = ExprRep.Tab BRANDED "AndExpr.P" OBJECT
      OVERRIDES
        typeOf       := ExprRep.NoType;
        check        := Check;
        compile      := Compile;
        evaluate     := Fold;
        fprint       := FPrinter;
        write        := ExprRep.NoWriter;
        isEqual      := ExprRep.EqCheckAB;
        getBounds    := ExprRep.NoBounds;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsNever;
	isZeroes     := ExprRep.IsNever;
	note_write   := ExprRep.NotWritable;
	genLiteral   := ExprRep.NoLiteral;
      END;

PROCEDURE New (a, b: Expr.T): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.a := a;
    p.b := b;
    p.type := Bool.T;
    RETURN p;
  END New;

PROCEDURE Check (p: P;  VAR cs: Expr.CheckState) =
  VAR ta, tb: Type.T;
  BEGIN
    Expr.TypeCheck (p.a, cs);
    Expr.TypeCheck (p.b, cs);
    ta := Type.Base (Expr.TypeOf (p.a));
    tb := Type.Base (Expr.TypeOf (p.b));
    IF (ta # Bool.T) OR (tb # Bool.T) THEN
      Error.Msg ("operands of AND must be of type BOOLEAN");
    END;
  END Check;

PROCEDURE Compile (p: P): Temp.T =
  VAR t1, t2: Temp.T;
  BEGIN
    t1 := Expr.Compile (p.a);
    t2 := Temp.Alloc (p);
    Emit.OpTT ("@ = 0;\nif (@) {\001\n", t2, t1);
    Temp.Free (t1);
    t1 := Expr.Compile (p.b);
    Emit.OpTT ("@ = @;\n\002}\n", t2, t1);
    Temp.Free (t1);
    RETURN t2;
  END Compile;

PROCEDURE Fold (p: P): Expr.T =
  VAR e1, e2: Expr.T;  i1, i2: INTEGER;  t1, t2: Type.T;
  BEGIN
    e1 := Expr.ConstValue (p.a);
    e2 := Expr.ConstValue (p.b);
    IF (e1 = NIL) OR (e2 = NIL)
      OR (NOT EnumExpr.Split (e1, i1, t1))
      OR (NOT EnumExpr.Split (e2, i2, t2))
      OR (Type.Base (t1) # Bool.T)
      OR (Type.Base (t2) # Bool.T) THEN
      RETURN NIL;
    ELSE
      RETURN Bool.Map[(ORD (TRUE) = i1) AND (ORD (TRUE) = i2)];
    END;
  END Fold;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "AND ");
    Expr.Fingerprint (p.a, map, wr);
    Expr.Fingerprint (p.b, map, wr);
  END FPrinter;

BEGIN
END AndExpr.
