/*           HyperText Browser for Dumb Terminals     		     HTBrowse.c
**           ====================================
**  
**  Authors:
**	NP  Nicola Pellow  Tech.Student CERN 1990, 91
**	TBL Tim Berners-Lee CERN
**
**      Copyright CERN 1990,91   See Copyright.html 
**
**  History:
**
**   4 Dec 90	Written from scratch (NP)
**  11 Feb 91   Code written by Tim BL so that the browser could be linked with
**              code produced by Bernd Pollermann, enabling access to the
**              data on CERNVM. This involved changing the code to handle file
**              numbers rather than file pointers.
**  18 Mar 91   The feature of history mechanism was included, enabling a  
**              record of previous nodes visited to be kept.
**   6 Apr 91   When a node is accessed, it is immediately read into a 
**              buffer, in an unformatted state, as soon as the connection is   
**              made, so that the server is freed as quickly as possible. 
**              The program now also uses the additional modules HTBufferFile.c
**              and HTBufferFile.h.
**  17 Apr      Can be used on machines running ANSI C and ordinary C.
**  10 May      Formatted text is stored in a linked list buffer which allows
**              scrolling and better page breaks in the middle of text.
**              Code incorporated by Tim BL, to enable anonymous FTP.          
**  21 May 91   Accepts various parameters on the command line.
**  19 Aug 91   Currently available in Unix, VAX/VMS and MVS environments.
**  21 Nov 91	Character grid uses new architecture. (TBL)
**		added -w option, new commands, print,
**	...	See Features.html for further details
**
** Compilation-time macro options
**
**	REF_MARK	Printf string to be used for printing anchor numbers
**	END_MARK	String to be used to denote the end of a document
*/
#ifndef VERSION
#define VERSION "1.0"
#endif

/* Default Addresses */
/* ================= */

#define LOGICAL_DEFAULT "WWW_HOME"            /* Defined to be the home page */

#ifdef vms
#define DEFAULT_ADDRESS \
 "http://info.cern.ch./hypertext/WWW/LineMode/Defaults/default.html"
#define REMOTE_ADDRESS \
 "http://info.cern.ch./hypertext/WWW/LineMode/Defaults/remote.html"
#endif

#ifdef VM
#define DEFAULT_ADDRESS \
 "http://info.cern.ch./hypertext/WWW/LineMode/Defaults/default.html"
#define REMOTE_ADDRESS \
 "http://info.cern.ch./hypertext/WWW/LineMode/Defaults/remote.html"
#endif

#ifndef DEFAULT_ADDRESS
#define DEFAULT_ADDRESS  "file:/usr/local/lib/WWW/default.html"
#define REMOTE_ADDRESS  "file:/usr/local/lib/WWW/remote.html"
#endif

/* If run from telnet daemon and no -l specified, use this file:
*/
#ifndef DEFAULT_LOGFILE
#define DEFAULT_LOGFILE	"/usr/adm/www-log/www-log"
#endif

/* Check Statements */
/* ================ */

#define NAME_CHECK 0                /* Trace to show NAME anchors */


/* Include Files */
/* ============= */

#include <ctype.h>
#include "HTUtils.h"		    /* WWW general purpose macros */
#include "HTString.h"

#ifdef SHORT_NAMES 
#include "HTShort.h"
#endif

#include "GridText.h"		    /* Hypertext definition */

#include "tcp.h"		    /* TCP/IP and file access */
#include "WWW.h"		    /* WWW project constants etc */
#include "HTTCP.h"		    /* TCP/IP utilities */

#include "HTParse.h"                /* WWW address manipulation */
#include "HTAccess.h"               /* WWW document access network code */
#include "HTHistory.h"		    /* Navigational aids */
#include "HTML.h"		    /* For parser */

extern HTStyleSheet * styleSheet;

/* Define Statements */
/* ================= */

#ifndef REF_MARK		/* May be redefined on command line */
#ifdef VM
#define REF_MARK " <%d>"	/* IBM terminals can't handle [] well */
#else
#define REF_MARK "[%d]"
#endif
#endif

#ifndef END_MARK
#ifdef VM
#define END_MARK "     <End>"
#else
#define END_MARK "     [End]"
#endif
#endif

#ifndef EOF
#define EOF (-1)                    /* End of file character defined as -1 */
#endif

#define WHITE_SPACE(c) ((c==' ')||(c=='\t')||(c=='\n')||(c=='\r'))
                                   /* Definition for any kind of white space */



#ifdef NEWLIB
#define SCREEN_WIDTH 78
#endif

#ifndef SCREEN_WIDTH      
#define SCREEN_WIDTH 79            /* width of the screen */ 
#endif
#ifndef SCREEN_HEIGHT
#define SCREEN_HEIGHT 24 /* Default Number of lines to the screen */
#endif

#ifdef VM                          /* Needed to flush out the prompt line..*/
#ifndef NEWLIB
#define NEWLINE_PROMPT             /* before input */
#endif				   /* Except on NEWLIB which can wrap. */
#endif

#define INFINITY 1024		   /* BUG @@@@@@@@@ */
#define	ADDRESS_LENGTH INFINITY	   /* Maximum address length of node */
#define TITLE_LENGTH INFINITY           /* Maximum length of a title */
#define RESPONSE_LENGTH INFINITY        /* Maximum length of users response */


/*	Public Variables
**	================
*/

PUBLIC  int  WWW_TraceFlag = 0;         /* Off unless -v option given */
PUBLIC  int  HTScreenWidth = SCREEN_WIDTH;	/* By default */
PUBLIC  int  HTScreenHeight = SCREEN_HEIGHT;	/* By default */
PUBLIC BOOL  display_anchors = YES;	/* anchor will be shown in text? */
PUBLIC  BOOL interactive = YES;         /*  e.g. shows prompts etc */
PUBLIC  BOOL end_of_file = NO;	       /* EOF read on input stream */

PUBLIC  BOOL end_of_formatted_buffer = NO; 
                                        /* Is there more text in the formatted
                                           buffer to be displayed ? */
PUBLIC  int  output_line_count;         /* Counts the lines as they are output
                                           on the screen */
					   
PUBLIC FILE * logfile = 0;		/* File to output one-liners to */

PUBLIC char * HTClientHost = 0;	/* Name or number of telnetting host */
PUBLIC char * reference_mark = REF_MARK;      /* Format string for  [1] &c */
PUBLIC char * end_mark = END_MARK;      /* Format string for  [End] */

 

/* Structure for Accumulating HyperText References */
/* =============================================== */

typedef struct anchor_struct {

       char                  *address;                /* Pointer to the address
                                                         of an anchor */
       } anchor; 


/* Arrays for storing the HyperText References */ 


PRIVATE char chosen_reference[ADDRESS_LENGTH]; /* Address of requested node */


PRIVATE char         choice[RESPONSE_LENGTH];    /* Users response  */

PRIVATE char *	     logfile_root = 0;	    /* Log file name */
PRIVATE char *	     logfile_name = 0;	    /* Root of log file name */

/* Forward Declaration of Functions */
/* ================================ */

void History_List NOPARAMS; 
void Selection_Prompt NOPARAMS;
BOOL Check_User_Input PARAMS((char *s));          
void Error_Selection NOPARAMS;
void help_screen NOPARAMS;
void Reference_List NOPARAMS;
BOOL Select_Reference PARAMS((int ref_num));



/* 				MAIN PROGRAM
**				============
*/

int main ARGS2(int argc, char *argv[])

{
    int  arg;		               	       /* Argument number as we scan */
    BOOL first_keyword = YES;
    BOOL argument_found = NO;
    char* default_default=0;	           /* @@ Parse home relative to this */

    StrAllocCopy(default_default, "file://");
    StrAllocCat(default_default, HTHostName());    /* Eg file://cernvax.cern.ch */

#ifdef MAXPATHLEN  
    {
        char wd[MAXPATHLEN];
	extern char * getwd();
        char * result = getwd(wd);
	if (result) {
	    StrAllocCat(default_default, wd);
	} else {
	    fprintf(stderr, "HTBrowse: Can't read working directory.\n");
	}
	StrAllocCat(default_default, "/default.html");
    }
#else
    if (TRACE) fprintf(stderr,
    	"HTBrowse: This platform does not support getwd()\n");
    StrAllocCat(default_default, "/");
#endif
	
/*	Check for command line options
**      ------------------------------
*/

    for (arg=1; arg<argc ; arg++) {
        if (*argv[arg]=='-'){

	    if (0==strcmp(argv[arg], "-v")){
                WWW_TraceFlag = 1;		   /* Verify: Turns on trace */

	    } else if (argv[arg][1] == 'p') {	   /* Page size */
		if (sscanf(argv[arg]+2, "%d", &HTScreenHeight) <1)  /* fail */
		    HTScreenHeight = 999999;	   /* Turns off paging */

	    } else if (argv[arg][1] == 'w') {	   /* Page width */
		if (sscanf(argv[arg]+2, "%d", &HTScreenWidth) <1)  /* fail */
		    HTScreenWidth = SCREEN_WIDTH;

	    } else if (argv[arg][1] == 'h') {	   /* Telnet from */
		if (++arg <argc)
		   HTClientHost = argv[arg];	  /* Use original host name */

	    } else if (argv[arg][1] == 'l') {	   /* Log file */
		if (++arg <argc) {
		   logfile_root = argv[arg];
		}
	    } else if (0==strcmp(argv[arg], "-n")) { /* Non-interactive */
		interactive = NO;	             /* Turns off interaction */

	    } else if (0==strcmp(argv[arg], "-a")) { /* Non-interactive */
		if (++arg <argc)
		   reference_mark = argv[arg];	  /* Change representation */

	    } else if (0==strcmp(argv[arg], "-na")) { /* No anchors */
		display_anchors = NO;	          /* Turns off anchor display */
	    }

        } else {

/*      Check for main argument
**      -----------------------
*/
	    if (!argument_found) {
						    	              /* TBL */
		char * ref = HTParse(argv[arg], default_default, 
		    PARSE_ACCESS|PARSE_HOST|PARSE_PATH|PARSE_PUNCTUATION);
		HTSimplify(ref);
		strcpy(chosen_reference,ref);
		free(ref);
                argument_found = YES;

	    } else { 

/*      Check for succesive keyword arguments
**      -------------------------------------
*/
                char * s;
                char * p;
                char * q;
                
                p = HTStrip(argv[arg]);

                for (q=p; *q; q++){
                    if (WHITE(*q)){
                        *q = '+';
                    }
                }  
                if (first_keyword){
                    s=strchr(chosen_reference,'?'); /* Find old search string */
                    if (s) *s =0;                   /* Chop old search off */
                    strcat(chosen_reference,"?");   /* Start new search */
                    first_keyword = NO;
                } else {
                    strcat(chosen_reference,"+");
                }
                strcat(chosen_reference,p);

            } /* Keywords */
        } /* Not an option '-'*/
    } /* End of arguement loop */


/*	Open Log File if necessary
**	--------------------------
*/
    if (!logfile_root && HTClientHost) {
        logfile_root = DEFAULT_LOGFILE;
    }
    
        
    if (logfile_root) {
	logfile_name = (char*) malloc(strlen(logfile_root)+20);
#ifdef VM
	sprintf(logfile_name, "%s", logfile_root);  /* No getpid() */
#else
	sprintf(logfile_name, "%s-%d", logfile_root, getpid());
#endif
	logfile = fopen(logfile_name, "a");
	if (!logfile)
	    fprintf(stderr, "WWW: Can't open log file %s\n",
		    logfile_name);
    }


/*	Make home page address
*/    
    if (!argument_found){
	char * my_home = (char *)getenv(LOGICAL_DEFAULT);
	char * ref = HTParse(my_home ?	my_home :
			HTClientHost ?	REMOTE_ADDRESS :
					DEFAULT_ADDRESS,
		    default_default,
		    PARSE_ACCESS|PARSE_HOST|PARSE_PATH|PARSE_PUNCTUATION);
	HTSimplify(ref);
	if (TRACE) fprintf(stderr,
		"HTBrowse: Using custom home page %s i.e. address %s\n",
		my_home, ref);
	strcpy(chosen_reference, ref);
	free(ref);
    }
    if (TRACE) fprintf(stderr,
    	"HTBrowse: DefaultDefault is %s, Initial page is %s\n",
    	 default_default, chosen_reference);

/*	Load default page
*/
    if (HTLoadAbsolute(chosen_reference)){ 
        HTHistory_record((HTAnchor *)HTMainAnchor);
	
    } else {		      /* Failure in accessing */
	printf("\nWWW: Can't access `%s'\n", chosen_reference);
	if (!HTMainText) exit(2);		/* Can't get first page */
    }
    
/* 	Main "Event Loop"
*/

     if (interactive) while (YES) Selection_Prompt();
     
#ifdef vms
    return 1
#else
     return 0;		/* Good */
#endif

} /* main() */



/*      Produce an error message
**	------------------------
*/                       

#ifdef __STDC__
void Error_Selection(void)
#else
void Error_Selection()
#endif

{    
    printf("%s", "Bad command, for list of commands type help.\n");
 
    return;
}


/*		HELP Screen
**		-----------
**
** Produce a help screen, displaying the current document address and a list of 
** of available commands. 
*/

PRIVATE void help_screen NOARGS

{
    char * current_address = HTAnchor_address((HTAnchor*)HTMainAnchor);
    CONST char * title = HTAnchor_title(HTMainAnchor);
    printf("\n\nWWW LineMode Browser version %s:     COMMANDS AVALIABLE\n\n",
    	 VERSION);
    if (title) printf(
"You are reading a document whose address is\n    '%s' \n\n", current_address);
    else printf(
        "You are reading\n \"%s\"\nwhose address is\n  %s\n\n",
	title, current_address);
	
    if (HText_canScrollDown(HTMainText))
        printf(
"  <RETURN>     Move down one page within the document.\n");
    
    if (HText_canScrollUp(HTMainText)) {
        printf(
"  Top          Return to the first page of the document.\n");
        printf(
"  Up           Move up one page within the document\n");
    }
    if (HText_sourceAnchors(HTMainText) != 0) {
        printf(
"  List         List the references from this document\n");
        printf(
"  <number>     Select a referenced document by number (from 1 to %i)\n",
			HText_sourceAnchors(HTMainText));
    }
    if (HTAnchor_isIndex(HTMainAnchor)) {
	printf(
"  Key <words>  Search this index for given words (separated by spaces)\n"); 
    }
    if (HTHistory_canBacktrack) {
        printf(
"  Recall       List documents visited.\n");
        printf(
"  Recall <number>\n");
        printf(
"               Return to a previously visited document\n");
        printf(
"               as numbered in the recall list.\n");
        printf("  HOme         Return to the starting document\n");
        printf("  Back         Move back to the last document\n");
    }
    if (HTHistory_canMoveBy(+1))
        printf("  Next         Take next link from last document.\n");
    if (HTHistory_canMoveBy(-1))
        printf("  Previous     Take previous link from last document.\n");
#ifdef unix
    if (!HTClientHost) {	/* NOT for telnet guest! */
        printf("  > file       Save the text of this document in a file.\n");
        printf("  | command    Pipe this document to a shell command.\n");
        printf("  ! command    Execute shell command without leaving www.\n");
    }
#endif
    printf    ("  Quit         Quits www.\n");
    printf("\n");
    free(current_address);
}


/*		Select Reference
**		----------------
**
**  After a reference is selected by the user, opens, links into the history
**  list and displays.
**
**  On Entry,
**       int  reference_num   Number corresponding to the hypertext reference
**                            given in the text.
*/

BOOL Select_Reference ARGS1(int  reference_num)

{  
    HTAnchor * destination;
    HTChildAnchor * source = HText_childNumber(HTMainText, reference_num);
    
    if (!source) return NO;		/* No anchor */
    destination = HTAnchor_followMainLink((HTAnchor*) source);
    if (!HTLoadAnchor(destination)) return NO;	/* No link */
    HTHistory_leavingFrom((HTAnchor*) source);
    HTHistory_record(destination);
    
    return YES;
    
} /* Selecte_Reference*/


/* 		Reference List
**		--------------
**  Print out a list of HyperText References accumulated within the text.
*/

#ifdef __STDC__
void Reference_List(void)
#else
void Reference_List()
#endif

{
    int  n;

    if (HText_sourceAnchors(HTMainText) == 0) {
        printf("\n\n     There are no references from this document.\n\n");
    } else {

	printf("\n\n     References from this document:-\n\n");
	
	for (n=1; n<=HText_sourceAnchors(HTMainText); n++) {
	    HTAnchor * destination =
	    	HTAnchor_followMainLink(
		        (HTAnchor *)HText_childNumber(HTMainText, n)
		    );
	    HTParentAnchor * parent = HTAnchor_parent(destination);
	    char * address =  HTAnchor_address(destination);
	    CONST char * title = HTAnchor_title(parent);
	    printf("[%d]  %s%s\n", n,
	    	((HTAnchor*)parent!=destination) && title ? "in " : "",
		title ? title : address);
	    free(address);
	}
	printf("\n");
    }
}      


/*		HISTORY LIST
**		------------
*/
/*	Display a history list of nodes visited during the session.
**
**	If anchors have titles, should use them. @@@
*/

PRIVATE void History_List NOARGS

{
    int  history_number = 1; 
    printf("\n  Documents you have visited:-\n\n");
    do {

	char * address;
	CONST char * title;
	HTAnchor * anchor = HTHistory_read(history_number);
	HTParentAnchor * parent;
	if (!anchor) break;
	parent = HTAnchor_parent(anchor);
	title = HTAnchor_title(parent);
	address = HTAnchor_address(anchor);
	printf("R %2d)   %s%s\n",
		history_number,
		((HTAnchor*)parent!=anchor) && title ? "in " : "",
		title ? title : address);
	free(address);
	
	history_number++;
	
    } while (YES);

    printf("\n");
}      


/*
**  		Check command word
**		------------------
**
** Accepts shortened versions of commands.
**
**  On Entry,
**       char *s  Correct and full version of command, to which the users 
**                input is compared.
**
**  On Exit,
**         returns  YES  Users input corresponds to the command.
**                   NO  Not a recognized command.  
*/

PRIVATE BOOL Check_User_Input ARGS1(char *s)
   
{   int k=0;
    int match=0;

    while (choice[k] == ' ') k++;

    for (; *s; s++){

        if (*s == toupper(choice[k])){
            match++;
            k++;
        } else {
            if (((choice[k] == '\n')||(choice[k] == ' '))&&(match>0)){
                return YES;          
            } else {
                return NO;
            }
        }
    }
    return YES;
}     


/*			USER INTERFACE
**			==============
*/

/*
**   Produces a prompt at the bottom of a page full of text. The prompt varies 
**   depending on the options avaliable.
*/

#ifdef __STDC__
void Selection_Prompt(void)
#else
void Selection_Prompt()
#endif

{ 
    int length_of_prompt = 0;
    BOOL is_index = HTAnchor_isIndex(HTMainAnchor);

    HText_setStale(HTMainText);	/* We corrupt the display */
    
    if (  !HText_canScrollDown(HTMainText) &&
    	  !HTAnchor_hasChildren(HTMainAnchor) &&
    	  ! is_index && 
          (!HTHistory_canBacktrack())){
        printf("\n");      
	exit(0);		             /* Exit if no other options */
    }

    if (is_index){	
        printf("K <keywords>, ");
        length_of_prompt = length_of_prompt + 14;
    }
    if (HTAnchor_hasChildren(HTMainAnchor)!=0){
	int refs = HText_sourceAnchors(HTMainText);
	if (refs>1) {
	    printf("1-%d, ", refs);
	    length_of_prompt = length_of_prompt + 6;	/* Roughly */
	} else {
	    printf("1, ");	
	    length_of_prompt = length_of_prompt + 3;
        }
    }
    if (HTHistory_canBacktrack()){
        printf("Back, ");
        length_of_prompt = length_of_prompt + 6;
    }
    if (HText_canScrollUp(HTMainText)){
        printf("Up, ");
        length_of_prompt = length_of_prompt + 4;
    }
    if (HText_canScrollDown(HTMainText)){
        printf("<RETURN> for more, ");
        length_of_prompt = length_of_prompt + 19;
    }
    if (length_of_prompt <= 47){
        printf("Quit, ");
    }
    printf("or Help: ");	


/*  	Read in the users input, and deal with as necessary.
**	-----------------------
**
**	Any Command which works returns from the routine. If nothing
**	works then a serach or error message donw at the bottom.
*/

    {   
	int  reference_num; 
	char *first_word = choice;	/* First word of command */
	char * second_word;		/* Second word */
    
    
#ifdef NEWLINE_PROMPT
	printf("\n");           /* For use on VM to flush out the prompt */ 
#endif
	fgets(choice,RESPONSE_LENGTH,stdin);	/* Read User Input */
    
    /*	Clear the screen (on screen-mode systems)
    */
#ifdef VM
        clear_screen();
#endif		/* VM */
    
	while (*first_word==' '||*first_word=='\t')
		first_word++; /* Skip leading space */
	second_word = first_word;
	while (!WHITE(*second_word) &&*second_word)
		second_word++; /* Skip first word */
	while (*second_word==' ' || *second_word=='\t') /* Skip space */
		second_word++; /* Skip leading space */

	switch(first_word[0]) {
    
	case '0':                               /* Selection of a reference */
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	    sscanf(first_word,"%d",&reference_num);
	    if ((reference_num>=1)&&
	    	(reference_num<=HText_sourceAnchors(HTMainText))){
		if (Select_Reference(reference_num))
		    return;
	    }            
	    break;
    
    
	case '\n':                              /* Continue with next page of   
						    document */
	    if (HText_canScrollDown(HTMainText))
	        HText_scrollDown(HTMainText);
	    return;
    
    
	case 'b':                    
	case 'B':                    
						/* Return to previous node ? */
	    if (Check_User_Input("BACK")) {
		if (!HTHistory_canBacktrack()){
		    printf("\n   The BACK command cannot be used,");
		    printf(" as there are no previous documents\n");
		    return; 
		}
		HTLoadAnchor(HTHistory_backtrack());
		return;
	    }
	    break;
	    
	case 'e':                                /* Quit program ? */
	case 'E':                                /* Alternative command */
	    if (Check_User_Input("EXIT"))
		exit(0);
	    break;
    
	case 'f':                                /* Keyword search ? */
	case 'F':
	    if (is_index && Check_User_Input("FIND")){
		    if (HTSearch(second_word))
		        HTHistory_record((HTAnchor*)HTMainAnchor);
		    return;
	    }
	    break;
    
	case 'g':                                /* GOTO */
	case 'G':
	    if (Check_User_Input("GOTO")){
		    if (HTLoadRelative(second_word))
		        HTHistory_record((HTAnchor*)HTMainAnchor);;
		    return;
	    }
	    break;
    
	case 'h':                             
	case 'H':
    
	    if (Check_User_Input("HELP")){  /* help menu, ..*/
		help_screen();                /* or a keyword search ? */
		return; 
	    }

	    if (Check_User_Input("HOME")){
		if (!HTHistory_canBacktrack()){ 
		    HText_scrollTop(HTMainText);
		} else {
		    HTLoadAnchor(HTHistory_recall(1));
		}
		return;
	    } /* if HOME */
	    break;
    
	
	case 'k':                                /* Keyword search ? */
	case 'K':
	    if (is_index && Check_User_Input("KEYWORDS")){
		    if (HTSearch(second_word))
		        HTHistory_record((HTAnchor*)HTMainAnchor);
		    return;
	    }
	    break;
    
	case 'l':                               /* List of references ? */
	case 'L':
	    if (Check_User_Input("LIST")){
		Reference_List();
		return;
	    }
	    break;
    
	case 'n':                    
	case 'N':                    
	    if (Check_User_Input("NEXT")) {
		if (!HTHistory_canMoveBy(+1)){   /* No nodes to jump back to */
		    printf("\n\n   Can't take the NEXT link from the last");
		    if (!HTHistory_canBacktrack())
		        printf(" document as there is no last");
		    printf(" document.\n");
		    return; 
		}
		HTLoadAnchor(HTHistory_moveBy(+1));
		return;
	    }
	    break;
	    
	case 'p':                    
	case 'P':                    
	    if (Check_User_Input("PREVIOUS")) {
		if (!HTHistory_canMoveBy(+1)){ 
		    printf(
		      "\n\n   Can't take the PREVIOUS link from the last");
		    if (!HTHistory_canBacktrack())
		        printf(" document as there is no last");
		    printf(" document.\n");
		    return;
		}
		HTLoadAnchor(HTHistory_moveBy(-1));
		return;
	    }
	    
#ifdef unix	    
	    if (!HTClientHost && Check_User_Input("PRINT")) {
	        char * address = HTAnchor_address((HTAnchor*)HTMainAnchor);
		char * command;
		char * template = (char*)getenv("WWW_PRINT_COMMAND");
		int result;
		
		if (!template) template = "www -n -na -p66 %s | enscript";
		command  = (char *) malloc(
			strlen(address)+strlen(template)+20);
	        sprintf(command, template, address);
		result = system(command);
		free(address);
		free(command);
		if (result) printf("  %s\n  returns %d\n", command, result);
		return;
	    }
#endif
	    break;
	    
    
	case 'q':                                /* Quit program ? */
	case 'Q':
	    if (Check_User_Input("QUIT"))
		exit(0);
	    break;
    
	case 'r':                            
	case 'R':
	    {
		int  recall_node_num;
		int status = 0;
    
		if (Check_User_Input("RECALL")){
		    if (!HTHistory_canBacktrack()){    /* No nodes to recall */
			printf("\n\n   No other documents to recall.\n");
			return;
    		    }
/* Is there a previous node number to recall, or does the user just require
 a list of nodes visited? */
 
		    status = sscanf(second_word, "%d",&recall_node_num);

		    if (status == 1){	/* Good parameter */

			HTLoadAnchor(HTHistory_recall(recall_node_num));

		    } else if (status <1) {
			History_List();
		    } else {
			Error_Selection();
		    }
		    return;
		}  /* if CHECK("RECALL") */
	    }	/* scope of status */
	    break;
    
	case 't':
	case 'T':
	    if (Check_User_Input("TOP")) {   /* Return to top  */
		HText_scrollTop(HTMainText);
		return;
	    }
	    break;
    
	case 'u':
	case 'U':
	    if (Check_User_Input("UP")) {   /* Scroll up one page  */
		HText_scrollUp(HTMainText);
		return;
	    }
	    break;
    
#ifdef unix	    
	case '>':
	case '|':
	    if (!HTClientHost) {	/* Local only!!!! */
	        char * address = HTAnchor_address((HTAnchor*)HTMainAnchor);
		char * command;
		int result;
		
		command  = (char *) malloc(
			strlen(address)+strlen(first_word)+30);
	        sprintf(command, "www -n -na -p %s %s", address, first_word);
		result = system(command);
		free(command);
		free(address);
		if (result) printf("  %s\n  returns %d\n", command, result);
		return;
	    }
	case '!':
	    if (!HTClientHost) {	/* Local only!!!!!!! */
		int result;
		result = system(&first_word[1]);
		if (result) printf("  %s\n  returns %d\n", &first_word[1], result);
		return;
	    }
#endif
    
	    
	default :
	    break;
	}	/* Switch on 1st character */
	
	if (is_index && *first_word){                       
	    if (HTSearch(first_word))
		HTHistory_record((HTAnchor*)HTMainAnchor);
	} else {             
	    Error_Selection();
	}
	
    }	/* Get user selection */
} /* Selection_Prompt */


