/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 * $Header: vl.c[4.26] Fri Feb 14 17:15:34 1992 axel@cs.tu-berlin.de accessed $
 */

#ifndef lint
static char *AtFSid = "$Header: vl.c[4.26] Fri Feb 14 17:15:34 1992 axel@cs.tu-berlin.de accessed $";
#ifdef CCFLGS
  static char *Cflags = CCFLGS;
#endif
#endif

#include <atfs.h>
#include <stdio.h>
#include <ParseArgs.h>
#include <atfsapp.h>

#define MAXCOLUMNS 80		/* should be determined by an ioctl() call */
#define LINECONT '>'            /* precedes multiline uda-vals */

extern char *re_comp(), *getwd(), *malloc();
extern char *vlversion(), *af_version(), *at_version();

static char *progname;
static char *form_String;
static char **names;		/* for muli-column output */
static int Aflag, Dflag, Fflag, Hflag, Iflag, Intentflag, Sflag;
static int Qflag, Lflag, Allflag, Crossflag, Rflag, Tflag;
static int Uflag, Ulflag, Hiddenflag, Vflag, Lockerflag;
static int Envflag, Columnflag, Xflag, Recursiveflag, Authorflag;
static int Archflag, Modeflag, Fmtflag, Sinceflag = 0;
static char cwd[MAXPATHLEN+2];
static char msg[2*MAXPATHLEN+1];
static int nameidx;
static int maxlen;
static int atty;		/* 1 iff stdout is a tty */

static struct d_desc {
  char *since_string;
  int  since_what, since_version;
  time_t since_when;
} since_desc;
#define SINCE_DATE 01
#define SINCE_SYMBOLIC 02
#define SINCE_VERSION 04

void logmsg(), logwarn(), logerr(), cleanup(), thin_out();
int handle_format_action();
int handle_since_action();
int IsPattern();


static void cleanup()
{
  (void) fflush(stdout);
  af_cleanup();
}

static void byebye(i)
     int i;
{
  cleanup();
  exit(i);
}

interrupt_action ()
{
  byebye (1);
}

static void die (i)
     int i;
{
  cleanup();
  (void) signal(i, SIG_DFL);
  (void) kill(getpid(),i);
}

static int printversion ()
{
  (void) printf("This is %s version %s.\nAtFS toolkit lib version %s.\n\
AtFS version %s.\n",
		progname, vlversion(), at_version(), af_version());
  return 1;
}

static int handle_A_opt ()
{
  Lflag = Authorflag = 1;	/* force -l flag */
  return 0;
}

static int handle_L_opt ()
{
  Lflag = Lockerflag = 1;	/* force -l flag */
  return 0;
}

static OptDesc optdesc[] = {
  { "#", PSWITCH|PSET, NULL, &Crossflag, NULL}, /* display only rev number */
  { "vnum", PSWITCH|PSET, NULL,  &Crossflag, NULL}, /* dito. */
  { "A", PSWITCH, handle_A_opt, NULL, NULL }, /* print author */
  { "C", PSWITCH|PSET, NULL, &Columnflag, NULL }, /* print columnized */
  { "F", PSWITCH|PSET, NULL, &Fflag, NULL }, /* symbolic file type */
  { "H", PSWITCH|PSET, NULL, &Hflag, NULL }, /* print history */
  { "L", PSWITCH, handle_L_opt, NULL, NULL }, /* print locker */
  { "R", PSWITCH|PSET, NULL, &Recursiveflag, NULL }, /* traverse directory */
  { "S", PSWITCH|PSET, NULL, &Sflag, NULL }, /* version state verbosely */
  { "U", PSWITCH|PSET, NULL, &Ulflag, NULL }, /* print name of uda + val */
  { "a", PSWITCH|PSET, NULL, &Aflag, NULL }, /* display hidden files */
  { "all", PSWITCH|PSET, NULL, &Allflag, NULL },	/* display all */
  { "archive", PSWITCH|PSET, NULL, &Archflag, NULL}, /* archive files */
  { "d", PSWITCH|PSET, NULL, &Dflag, NULL }, /* only directory */
  { "format", PARG, handle_format_action, &Fmtflag, NULL }, /* custom format */
  { "h", PFAIL, NULL, NULL, "files ..."},	/* help */
  { "i", PSWITCH|PHIDDEN|PSET, NULL, &Iflag, NULL }, /* interactive mode */
  { "intent", PSWITCH|PSET, NULL, &Intentflag, NULL}, /* print intention */
  { "l", PSWITCH|PSET, NULL, &Lflag, NULL }, /* list in long format */
  { "m", PSWITCH|PSET, NULL, &Modeflag, NULL },	/* print file mode */
  { "noenv", PSWITCH|PSET, NULL, &Envflag, NULL }, /* no environment var */
  { "q", PSWITCH|PSET, NULL, &Qflag, NULL }, /* be quiet */
  { "r", PSWITCH|PSET, NULL, &Rflag, NULL }, /* sort reverse */
  { "t", PSWITCH|PSET, NULL, &Tflag, NULL }, /* sort by time */
  { "since", PARG, handle_since_action, &Sinceflag, NULL },
  { "u", PSWITCH|PSET, NULL, &Uflag, NULL }, /* print name of uda */
  { "ux", PSWITCH|PSET, NULL, &Hiddenflag, NULL }, /* print also hidden udas */
  { "v", PSWITCH|PSET, NULL, &Vflag, NULL }, /* print attribute value only */
  { "version", PSWITCH, printversion, NULL },
  { "x", PSWITCH|PSET, NULL, &Xflag, NULL }, /* print left to right */
  { (char *) NULL, NULL, NULL, NULL, NULL},
};

/***/
int main (ac, av)
     int ac;
     char *av[];
{
  int i, nac, success, err_reported;
  char **nav;
  char *cp;
  char path[MAXPATHLEN+1], name[MAXNAMLEN+1], type[MAXTYPLEN+1];
  Af_set set, tmpset;
  Af_attrs attr;
  
  int Order(), DisplayInfo(), IsDirectory(), traverse();
  void vl_initattrs();

  progname = (cp = rindex(av[0], '/')) ? ++cp : av[0];

  if (!strcmp(progname, "vlog")) Hflag++;
  else if (!strcmp(progname, "vinfo")) Iflag++;
  else if (!strcmp(progname, "pattr")) Vflag++;
  if (getuid() == 0) Aflag++;
  if (at_ParseArgs(ac, av, &nac, &nav, optdesc)) byebye(1);
  if (!Envflag)
    if (ParseEnv(progname, "VLOPTS", optdesc))
      logwarn("invalid option specification in env variable VLOPTS");
  (void) getwd(cwd);
  success = 0;
  err_reported = 0;
  atty = isatty(fileno(stdout));

  /* to not confuse other progs we do not allow -F in a pipe. */
  if (Fflag && !atty) Fflag = 0;

  /* print uda value regardless if uda is a hidden attribute. */
  if (Vflag) {
    Hiddenflag++;
    if (!at_nuda && !Fmtflag) {
      logerr("You must specify at least one attribute name via option -uda\n");
      byebye(1);
    }
  }

  /* can we print a columnized output ?  */
  Columnflag =
    (!(Hflag||Lflag||Uflag||Ulflag||Vflag||Allflag||
       Crossflag||Modeflag||Intentflag) &&
     (atty || Columnflag));

  if (Iflag) {			/* switch to interactive mode */
    logerr("interactive mode not yet implemented.\n");
    byebye(0);
  }

  (void)signal (SIGHUP, die);
  (void)signal (SIGQUIT, die);
  (void)signal (SIGINT, die);
  (void)signal (SIGBUS, die);
  (void)signal (SIGSEGV, die);
  (void)signal (SIGFPE, die);
  (void)signal (SIGTERM, die);
  
  (void) vl_initattrs(&attr);
  (void) af_initset(&set);
  
  if (nac)
    qsort((char *) nav, nac, sizeof(char *), Order);
  else {
    if (at_find(&attr, cwd, cwd, (Dflag)?".":(char *)NULL,
		(char *)NULL, &set) == -1)
      byebye(1);
    success += DisplayInfo(&set, 0, Aflag || Dflag);

    if (Recursiveflag) {
      Af_attrs dirwarrant;
      Af_set dirset;
      int pdirs;

      (void)af_initattrs (&dirwarrant);
      dirwarrant.af_mode = S_IFDIR;
      if ((pdirs = af_find (&dirwarrant, &dirset)) == -1) {
	af_perror ("af_find (directories)");
      }
      if (af_union (&set, &dirset, &set) == -1) {
	af_perror ("af_union(set, &dirset, set)");
      }

      success += traverse("", &set);
    }

    if (!success) {
      logwarn("nothing appropriate found.\n");
      byebye(1);
    }
    byebye(0);
  }

  for (i = 0; i < nac; i++) {
    if (nav[i][strlen(nav[i])-1] == '/') nav[i][strlen(nav[i])-1] = '\0';
    (void) strcpy(path, af_afpath(cp=nav[i]));
    (void) strcpy(name, af_afname(cp));
    (void) strcpy(type, af_aftype(cp));

    if (!at_ispattern(cp) &&	
	(af_access(path, name, type, (at_Bflag)?AF_DERIVED:AF_SOURCE) == -1)) {
      (void) sprintf(msg,"%s not found.\n", cp);
      logwarn(msg);
      err_reported++;
      continue;
    }

    if (IsDirectory(cp) && !Dflag) {
      if (af_nrofkeys(&set)) {	/* print now info about plain files */
	success += DisplayInfo(&set, 1, 1);
	logmsg("\n");
	(void) af_dropset(&set);
	(void) af_initset(&set);
      }
      if (nac > 1) {
	(void) sprintf(msg,"%s:\n", cp);
	logmsg(msg);
      }
      (void) at_find(&attr, cp, cp, (char *) NULL, (char *) NULL, &set);
      success += DisplayInfo(&set, 0, Aflag);
      if (Recursiveflag) {
	if (cp[strlen(cp) - 1] == '/')
	  success += traverse(cp, &set);
	else {
	  char cp2[MAXPATHLEN+1];
	  (void) strcpy(cp2, cp);
	  (void) strcat(cp2, "/");
	  success += traverse(cp2, &set);
	}
      }
      else
	(void) af_dropset(&set); /* set is freed in traverse() */

      (void) af_initset(&set);
      if (i + 1 < nac) logmsg("\n");
    }
    else {
      af_initset(&tmpset);
      (void) at_find(&attr, cp, path, name, type, &tmpset);
      (void) af_union(&set, &tmpset, &set);
      (void) af_dropset(&tmpset);
    }
  }

  if (af_nrofkeys(&set)) success += DisplayInfo(&set, 1, 1);

  if (success) byebye(0);

  if (!err_reported) logwarn("nothing appropriate found.\n");
  byebye(1);
}

static void vl_initattrs (attr)
     Af_attrs *attr;
{
  (void) af_initattrs(attr);

  if (Intentflag) {
    attr->af_udattrs[0] = "__Intent__="; /* should be INTENT, but */
					 /* INTENT has a partial value. */
    attr->af_udattrs[1] = (char *) NULL;
  }
}

static int IsDirectory (name)
     char *name;
{
  struct stat buf;

  if (stat(name, &buf)) return 0;
  else return (buf.st_mode & S_IFDIR);
}

/***/
static int Order (left, right)
     char **left, **right;
{
  /* Order sorts in ls(1) manner. */
  int dir;

  /* if both are different file types then file has precedence */
  if ((dir = IsDirectory(*left)) ^ IsDirectory(*right))
    return dir ? 1 : -1;

  /* otherwise, sort in lexical order. */
  return strcmp(*left, *right);
}

static void PrintColumnized ()
{
  int i, j, len, wordsperline, nblanks, colwidth, this, lines, nl;

  if (!nameidx) {
    (void) free((char *) names);
    return;
  }

  colwidth = maxlen + 2;
  if (!(wordsperline = MAXCOLUMNS / colwidth)) wordsperline++;
  if (wordsperline > nameidx) wordsperline = nameidx;

  nl = 0;
  if (Xflag) {
    for (i = 0; i < nameidx; i++) {
      (void) printf("%s", names[i]);
      len = strlen(names[i]);
      (void) free(names[i]);

      if ((i % wordsperline == wordsperline - 1) || ( i == nameidx - 1)) {
	(void) putchar('\n');
	nl = 0;
	continue;
      }

      nl++;
      nblanks = colwidth - len;
      while (nblanks--) (void) putchar(' ');
    }
  }
  else {
    lines = (nameidx / wordsperline);
    if (nameidx  % wordsperline) lines++;

    for (i = 0; i < lines; i++) {
      for (j = 0; j < wordsperline; j++) {
	if ((this = j * lines + i) < nameidx) {
	  (void) printf("%s", names[this]);
	  len = strlen(names[this]);
	  (void) free(names[this]);
	}
	else
	  len = colwidth;

	if (j < wordsperline - 1) {
	  nl++;
	  nblanks = colwidth - len;
	  while (nblanks--) (void) putchar(' ');
	}
	else {
	  nl = 0;
	  (void) putchar('\n');
	}
      }
    }
  }

  if (nl) (void) putchar('\n');
  (void) free((char *) names);
}

static void RegisterName(name)
     char *name;
{
  int len = strlen(name) - 1;

  name[len] = '\0';
  if ((names[nameidx] = malloc((unsigned) (len + 1))) == (char *) NULL) {
    logerr("Out of memory.\n");
    byebye(1);
  }

  (void) strcpy(names[nameidx++], name);
  if (len > maxlen) maxlen = len;
}

static int DisplayInfo (set, ppath, phid)
     Af_set *set;
     int ppath, phid;
{
  Af_attrs attrs;
  Af_key key;
  int i, j, k, success, nkeys;
  char *cp, *note = NULL, *intent = NULL;
  char *af_rnote();

  void DisplayAll();

  key.af_lpos = ERROR;   /* key starts off as undefined */

  if (Sinceflag) thin_out (set, &since_desc);

  nkeys = af_nrofkeys(set);
  if (nkeys == 0) return 0; /* fail */

  success = 0;

  if (Archflag || (Hflag && Intentflag)) nkeys = at_last(set, set);
  
  if (af_sortset(set, Tflag ? AF_ATTMTIME : AF_ATTHUMAN) == -1) {
    af_perror("DisplayInfo: af_sortset"); byebye(1);
  }

  if (Tflag ^ Rflag) at_revset(set); /* ??? */

  if (Columnflag) {
    names = (char **) malloc((unsigned)(sizeof(char *) * (nkeys + 1)));
    nameidx = 0; maxlen = 0;
    if (names == (char **) NULL) {
      logerr("Out of memory.\n");
      byebye(1);
    }
  }

  for (i = 0; i < nkeys; i++) {
    (void) af_dropkey (&key); /* doesn't hurt if key is not valid */
    if (af_setgkey(set, i, &key) == -1) {
      af_perror("DisplayInfo: af_setgkey:");
      byebye(1);
    }

    if (af_gattrs(&key, &attrs) == -1) {
      af_perror("DisplayInfo: af_gattrs:");
      byebye(1);
    }

    if (Hflag)
      if (((note = af_rnote(&key)) == (char *) NULL)
	  && (attrs.af_state == AF_BUSY))
	note = af_rudattr(&key, INTENT);
    
    if (Intentflag) intent = af_rudattr(&key, INTENT);

    /* key is no longer neccessary, free memory */
    /* (void) af_dropkey(&key); */
    
    if ((attrs.af_name[0] == '.') && !phid) /* skip hidden aso's */
      continue;

    success = 1;		/* asume success */

    if (Intentflag && intent) {
      (void) printf("%s:\n%s",
		    at_getbndvers(&attrs,
				  !(ppath && strcmp(cwd, attrs.af_syspath))),
		    intent);
      if (intent[strlen(intent)-1] != '\n') putchar('\n');
      (void) free(intent);
      continue;
    }

    if (Archflag) {
      (void) printf("%s\n", at_getasoname(&attrs, 0)); /* w/o path */
      continue;
    }
      
    if (Qflag)			/* if quiet option set */
      continue;

    if (Crossflag) {		/* print only version number */
      (void) printf("%s\n", at_revision(&attrs));
      continue;
    }

    if (Modeflag) {		/* print file mode, octal representation */
      (void) printf("%o\n", attrs.af_mode & 0007777);
      continue;
    }
    if (Allflag) { /* if -all option set print everything we have */
      DisplayAll (&attrs);
      continue;
    }

    /* if we wish to print udas and we have none skip this aso */
/*    if ((Uflag || Ulflag || Vflag) && !attrs.af_udattrs[0])
      continue;
*/
    /* if we want a log message history and this aso has none skip it. */
    if (Hflag && (note == (char *) NULL))
      continue;

    if (!Vflag) {		/* we do not want the name iff -v */
      if (Lflag) /* long output required via option -l */
	(void) sprintf(msg, "%s %s %s%8d %s %s%s%s\n",
		       at_getmode(&attrs),
		       at_getversstate(&attrs, Sflag),
		       Lockerflag ? at_getuser(&attrs.af_locker) :
		       Authorflag ? at_getuser(&attrs.af_author) :
		       at_getuser(&(attrs.af_owner)),
		       attrs.af_size,
		       Lockerflag ? at_getlockdate(&attrs) :
		       ((attrs.af_state == AF_BUSY) || at_isDerived(&attrs))
			? at_getmoddate(&attrs) :
		          at_getsavedate(&attrs),
		       at_getbndvers(&attrs, !(ppath &&
					       strcmp(cwd, attrs.af_syspath))),
		       Fflag ? at_symbfiletype(&attrs) : "",
		       (Hflag || Uflag || Ulflag || Vflag) ? ":" : "");
      else
	(void) sprintf(msg,"%s%s%s%s\n",
		       at_getbndvers(&attrs, !(ppath &&
					       strcmp(cwd, attrs.af_syspath))),
		       (Fflag) ? at_symbfiletype(&attrs) : "",
		       (Sflag) ? at_getversstate(&attrs, Sflag) : "",
		       (Hflag || Uflag || Ulflag || Vflag) ? ":" : "");

      if (Columnflag) {
	RegisterName(msg);
	continue;
      }
      else
	logmsg(msg);
    }

    if (Hflag && note) {	/* now print the note iff required */
      (void) printf("%s", note);
      if (note[strlen(note)-1] != '\n') (void) putchar('\n');

      free(note);
    }

    if (Vflag||Uflag||Ulflag)  /* print the uda */ {
      /* XXX this code needs reimplementation */
      int h;
      if (Fmtflag) {
	WriteXPand (form_String, strlen (form_String), stdout, &key);
	goto formatexit;
      }
      for (h = 0; at_udas[h]; h++) {
	if (at_is_stdattr (at_udas[h]) && Vflag) {
	  (void) substitute (at_udas[h], &key, &attrs, stdout);
	  if (isatty (fileno (stdout))) (void) putchar ('\n');
	}
      }

      for (j = 0; attrs.af_udattrs[j]; j++) {
	char **flatted, *xflatted[2];
	int idx;

	if (Vflag||Ulflag)
	  flatted = (char **) at_flatten_uda(attrs.af_udattrs[j]);
	else {
	  xflatted[0] = attrs.af_udattrs[j];
	  xflatted[1] = (char *) NULL;
	  flatted = xflatted;
	}
	
	for (idx = 0; cp = flatted[idx]; idx++) {
	  /* if a uda is specified via option -uda, skip the other udas. */
	  if (at_Udaflag) {
	    for (k = 0; at_udas[k]; k++) {
	      if (at_udapattern) {
		(void) re_comp(at_udas[k]);
		if (re_exec(cp)) break;
	      }
	      else if (!strncmp(at_udas[k], cp, strlen(at_udas[k]))) break;
	    }
	    if (at_udas[k] == (char *) NULL)
	      continue;
	  }

	  /* if uda is a hidden uda skip it except -ux options is set */
	  if (cp && (cp[0] == '_') && (cp[1] == '_') && !Hiddenflag)
	    continue;

	  if (Vflag) {		/* print only the value, skip name */
 	    if (!(cp = index(cp, '='))) *cp = '\0';
	  }
	  else			/* print the name of the uda */
	    while(cp && *cp != '=') (void) putchar(*cp++);
	  ++cp;
	  if (Ulflag || Vflag)
	    if (Vflag) {	/* print value w/o a leading '=' char */
	      (void) substitute (at_udas[k], &key, &attrs, stdout);
	      break;
	    }
	    else {
	      (void) printf("=%s\n", cp);
	      for (;cp = flatted[++idx];) {
		if (!(cp = index(cp, '='))) cp = flatted[idx];
		else cp++;
		(void) printf ("%c%s\n", LINECONT, cp);
	      }
	      break;
	    }
	}
      }
    formatexit:;
    }
  } /* end of main for-loop */
  (void) af_dropkey (&key); 

  if (Columnflag) PrintColumnized();

  return success;
}

static void DisplayAll (attrs)
     Af_attrs *attrs;
{
  int i;

  (void) printf("****************************************\n%s:%s",
		attrs->af_host,
		at_getbndvers(attrs, 0));

  (void) printf(" %s",at_getversstate(attrs, 1));

  if (attrs->af_locker.af_username[0])
    (void) printf(" locked by %s\n", at_getuser(&(attrs->af_locker)));
  else
    (void) printf(" not locked\n");

  (void) printf("Owner: %s", at_getuser(&(attrs->af_owner)));
  (void) printf("\tAuthor: %s\n", at_getuser(&(attrs->af_author)));

  (void) printf("Size: %d\tMode: %s\n", attrs->af_size, at_getmode(attrs));

  if (attrs->af_mtime != -1)
    (void) printf("mtime: %s\n", at_getmoddate(attrs));
  if (attrs->af_atime != -1)
    (void) printf("atime: %s\n", at_getaccdate(attrs));
  if (attrs->af_ctime != -1)
    (void) printf("ctime: %s\n", at_getcreatdate(attrs));
  if (attrs->af_stime != -1)
    (void) printf("stime: %s\n", at_getsavedate(attrs));
  if (attrs->af_ltime != -1)
    (void) printf("ltime: %s\n", at_getlockdate(attrs));

  if (attrs->af_udattrs[0]) {
    (void) printf("\nUser defined attributes:\n");
    for (i = 0; attrs->af_udattrs[i]; i++)
      (void) printf("%s\n", attrs->af_udattrs[i]);
  }
}

static void thin_out (set, desc) Af_set *set; struct d_desc *desc; {
  int nkeys = af_nrofkeys (set);
  register int i;
  time_t disc_date;
  Af_key this_key, *baseline_key, *at_sym2key();
  Af_attrs these_attrs;

  i = 0;
  switch (desc->since_what) {
  case SINCE_DATE:
    while ( i < nkeys ) {
      if (af_setgkey (set, i, &this_key) < 0) {
	af_perror ("thin_out: af_setgkey");
	return;
      }
      if (af_gattrs (&this_key, &these_attrs) < 0) {
	af_perror ("thin_out: af_gattrs");
	return;
      }
      udafree (&these_attrs);
      if (these_attrs.af_stime <= desc->since_when) {
	if (af_setrmkey (set, &this_key) < 0) {
	  af_perror ("thin_out: af_setrmkey");
	  af_dropkey (&this_key);
	  i++; continue;
	}
	nkeys--; continue;
      }
      af_dropkey (&this_key);
      i++;
    }
    break;
  case SINCE_VERSION:
    while ( i < nkeys ) {
      if (af_setgkey (set, i, &this_key) < 0) {
	af_perror ("thin_out: af_setgkey");
	return;
      }
      if (mkvno (vnum(&this_key)) <= desc->since_version) {
	if (af_setrmkey (set, &this_key) < 0) {
	  af_perror ("thin_out: af_setrmkey");
	  af_dropkey (&this_key);
	  i++; continue;
	}
	nkeys--; continue;
      }
      af_dropkey (&this_key);
      i++;
    }
    break;
  case SINCE_SYMBOLIC:
    /*
     * For each subset of keys in set that are from the same history,
     * remove all versions from set that are older or same age as the
     * the version (from the subset) that has symbolic name "since_string".
     */
    {
      char cur_histname[MAXPATHLEN], histname[MAXPATHLEN];
      cur_histname[0] = (char)0; histname[0] = (char)0;
      
      while ( i < nkeys ) {
	/* get next key from set */
	if (af_setgkey (set, i, &this_key) < 0) {
	  af_perror ("thin_out: af_setgkey");
	  return;
	}
	
	/* get historyname of this_key */
	mkvstring (histname, &this_key);
	*(index (histname, '[')) = '\0';
	
	/* is it in the current history ? */
	if (strcmp (histname, cur_histname)) {
	  
	  /* ...no! It's a new history. Reset discriminating date. */
	  
	  /* first, find version with symbolic name in new history. */
	  
	  baseline_key = at_sym2key (histname, desc->since_string);
	  if (baseline_key == (Af_key *)NULL)
	    /* 
	     * if no baseline version exists, discriminate all versions
	     * from current history.
	     */
	    disc_date = (time_t)0;
	  else {
	    /*
	     * determine savedate of baseline version.
	     */
	    if (af_gattrs (baseline_key, &these_attrs) < 0) {
	      af_perror ("thin_out: af_gattrs");
	      af_dropkey (&this_key);
	      af_dropkey (baseline_key);
	      return;
	    }
	    udafree (&these_attrs);
	    disc_date = these_attrs.af_stime;
	  }
	  af_dropkey (baseline_key);
	  (void) strcpy (cur_histname, histname);
	}
	
	/*
	 * ASSERT: disc_date is the date discriminating the versions prior or
	 * equal to the baseline version, defined by symbolic name.
	 */
	if (af_gattrs (&this_key, &these_attrs) < 0) {
	  af_perror ("thin_out: af_gattrs");
	  af_dropkey (&this_key);
	  return;
	}
	udafree (&these_attrs);
	if ((disc_date == (time_t)0) || (these_attrs.af_stime <= disc_date)) {
	  if (af_setrmkey (set, &this_key) < 0) {
	    af_perror ("thin_out: af_setrmkey");
	    af_dropkey (&this_key);
	    i++; continue;
	  }
	  nkeys--; continue;
	}
	af_dropkey (&this_key);
	i++;
      }
    }
    break;
  default:
    logerr ("In thin_out: unresolved switch....");
    exit (1);
  }
}

Af_key *at_sym2key (fname, sym) char *fname, *sym; {
  Af_attrs warrant;
  Af_set   hits;
  static Af_key   key;
  char symname[SYMNAMLEN], messg[MAXPATHLEN + 80];
  int rc;

  af_initattrs (&warrant);
  (void)sprintf (symname, "%s=%s", SYMNAME, sym);
  warrant.af_udattrs[0] = symname;
  warrant.af_udattrs[1] = (char *)NULL;
  (void)strcpy (warrant.af_syspath, af_afpath(fname)); 
  (void)strcpy (warrant.af_name, af_afname(fname));
  (void)strcpy (warrant.af_type, af_aftype(fname));
  if ((rc=af_find (&warrant, &hits)) > 1) { 
    /* this shouldn't happen, as symbolic names are supposed to be */
    /* within one ASO's history */
    af_dropset (&hits);
    (void)sprintf (messg, 
     "%s's history inconsistent.\nSymbolic name \"%s\" multiply assigned.",
	     fname, sym);
    logerr (messg);
    return (Af_key*)NULL;
  }
  else if (rc < 0) {
    af_perror ("af_find (in SymnameInUse)");
    return (Af_key*)NULL;
  }
  else if (rc == 0) {
    af_dropset (&hits);
    return (Af_key*)NULL;
  }
  else { /* rc must be 1 -- thats just fine */
    af_setgkey (&hits, 0, &key);
    af_dropset (&hits);
    return &key;
  }
}

void logmsg (msg)
     char *msg;
{
  if (Qflag) return;
  (void) printf(msg);
}

void logerr (msg)
     char *msg;
{
  (void) fflush(stdout);
  (void) fprintf(stderr, "%s: %s", progname, msg);
}

static void logwarn (msg)
     char *msg;
{
  if (Qflag) return;
  (void) fflush(stdout);
  (void) fprintf(stderr, "%s: %s", progname, msg);
}

static int traverse (rootdir, set)
     char *rootdir;
     Af_set *set;
{
  Af_attrs attrs;
  Af_set dset;
  Af_key key;
  char **dirs;
  char *dir, ldir[MAXPATHLEN+1];
  int i, ndirs, pdirs, success, nkeys;

  
  if ((nkeys = af_nrofkeys(set)) == 0) return 0;
  success = 0;

  dirs = (char **) malloc((unsigned) (sizeof(char *) * (nkeys + 1)));
  if (dirs == (char **) NULL) {
    logerr("Out of memory\n");
    byebye(1);
  }
  dirs[ndirs = 0] = (char *) NULL;

  for (i = 0; i < nkeys; i++) {
    if (af_setgkey(set, i, &key) == -1) {
      af_perror("traverse: af_setgkey:");
      byebye(1);
    }
    
    if (af_gattrs(&key, &attrs) == -1) {
      af_perror("traverse: af_gattrs:");
      byebye(1);
    }

    (void) af_dropkey(&key);
    
    if ((attrs.af_mode & S_IFDIR) &&
	(strcmp(attrs.af_name, ".")) && 
	(strcmp (attrs.af_name, "..")) &&
	(strcmp(attrs.af_name, AF_SUBDIR))) {

      dir = at_getbndvers(&attrs, 0);

      dirs[ndirs] = malloc((unsigned) (strlen(dir) + 1));
      if (dirs[ndirs] == (char *) NULL) {
	logerr("Out of memory\n");
	byebye(1);
      }
      (void) strcpy(dirs[ndirs], dir);
      dirs[++ndirs] = (char *) NULL;
    }
  }

  (void) af_dropset(set);

  for (i = 0; i < ndirs; i++) {
    Af_attrs dirwarrant;
    Af_set dirset;

    dir = dirs[i];
    (void) sprintf(ldir, "%s%s", rootdir, af_afname(dir));

    if (chdir(dir) == -1) {
      (void) sprintf(msg,"\n%s unreadable\n", ldir);
      logmsg(msg);
      continue;
    }

    (void) vl_initattrs(&attrs);
    (void) af_initset(&dset);
    (void) sprintf(msg, "\n%s:\n", ldir);
    logmsg(msg);
    

    if (at_find(&attrs, dir, dir, (char *) NULL, (char *) NULL, &dset))
      success += DisplayInfo(&dset, 0, Aflag);

    free(dirs[i]);

    (void)af_initattrs (&dirwarrant);
    dirwarrant.af_mode = S_IFDIR;
    if ((pdirs = af_find (&dirwarrant, &dirset)) == -1) {
      af_perror ("af_find (directories)");
    }
    if (pdirs) {
      (void) strcat(ldir,"/");
      success += traverse(ldir, &dirset);
    }

    (void) chdir("..");
  }

  free((char *) dirs);
  return success;
}

static handle_since_action (o, a) char *o, *a; {
  time_t at_mktime();
  char junk[MAXPATHLEN];

  if (a && *a) {
    if ((since_desc.since_string = 
	 malloc ((unsigned) (strlen (a) + 2))) == NULL) {
      logerr ("Not enough memory.\n");
      byebye(1);
    }
    (void) strcpy (since_desc.since_string, a);
    Sinceflag = 1;
    if (since_desc.since_when = at_mktime (since_desc.since_string))
      since_desc.since_what = SINCE_DATE;
    else if (since_desc.since_version = mkvno (since_desc.since_string))
      since_desc.since_what = SINCE_VERSION;
    else
      if (!BoundVersion (since_desc.since_string, junk, junk)) {
	since_desc.since_what = SINCE_SYMBOLIC;
      }
      else
	Sinceflag = 0;
  }
  if (Sinceflag)
    return 0;
  logerr ("No baseline specified -- \"-since\" ignored.");
  return 0;
}

static handle_format_action (o, a) char *o, *a; {
  char *convert_special();
  if (!Vflag) {
    Vflag = 1;
    Fflag = Hflag = Iflag = Intentflag = Sflag = Lflag = Allflag = 
      Crossflag = Uflag = Ulflag = Lockerflag = Envflag = Columnflag = 
	Xflag = Authorflag = Aflag = Archflag = Modeflag = 0;
  }
  if (!a || !*a) {
    logerr ("You must specify a format string.\n");
    byebye(1);
  }
  if ((form_String = malloc ((unsigned) (strlen (a) + 2))) == NULL) {
    logerr ("Not enough memory.\n");
    byebye(1);
  }
  (void)strcpy (form_String, convert_special(a));
  /*(void)strcat (form_String, "\n"); */
  Fmtflag = 1;
  /* Just for now. \n should be in format specification. */
  return 0;
}

#define tcc(strp) \
{ register char *cpx = strp+1, *cpy = strp+2; do *cpx++ = *cpy; \
    while (*cpy++); }

static char *convert_special (s) char *s; {
  char bsl, *index();
  register char *cp = s;
  
  while ((cp = index (cp, '\\')) && *(cp+1)) {
    switch (bsl = *(cp+1)) {
    case 'n':
      *cp = '\n';
      tcc(cp);
      break;
    case 't':
      *cp = '\t';
      tcc(cp);
      break;
    case '\\':
      tcc(cp);
      break;
    }
    cp++;
  }
}
