;;; Name:              shapetools.el
;;; Author:            shape@cs.tu-berlin.de
;;; Version:           1.30 
;;; State:             accessed
;;; Last modification: Tue Jan 28 17:33:44 1992 by nickel@cs.tu-berlin.de

;;; Shapetools mode for GNU Emacs

;;; Copyright (C) 1985, 1986 Free Software Foundation, Inc. -- this
;;; code is based on DIRED.

;;; This file is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY.  No author or distributor accepts
;;; responsibility to anyone for the consequences of using it or for
;;; whether it serves any particular purpose or works at all, unless
;;; he says so in writing.  Refer to the GNU Emacs General Public
;;; License for full details.

;;; Everyone is granted permission to copy, modify and redistribute
;;; this file, but only under the conditions described in the GNU
;;; Emacs General Public License.  A copy of this license is supposed
;;; to have been given to you along with GNU Emacs so you can know
;;; your rights and responsibilities.  It should be in a file named
;;; COPYING.  Among other things, the copyright notice and this notice
;;; must be preserved on all copies.

(provide 'shapetools)
(require 'callp-err)

;;; BUGS: 
;;;
;;; shape-retrv deletes display of last saved version if busy version
;;; already exists

;;; To configure the emacs interface change the following two
;;; definitions (in most cases this will not be necessary):

(defvar shape-bin-dir ""
  "String: directory from which the shapetools binaries are called.
May be an empty string if the directory is in the standard search path.
If non-empty, it *must* end with a slash.")

(defvar shape-interface-dir ""
  "String: directory from which the Emacs interface files are loaded.
May be an empty string if the directory is in Emacs' load-path.
If non-empty, it *must* end with a slash.")

;;; If you want to see the directory in folded form initially, set the
;;; value of the following variable to a non-nil value:

(defvar shape-fold-initially nil 
  "*Non-nil means the directory is shown in folded form initially.")

(defvar shape-vl-command (concat shape-bin-dir "vl"))
(defvar shape-vadm-command (concat shape-bin-dir "vadm"))
(defvar shape-vcat-command (concat shape-bin-dir "vcat"))
(defvar shape-save-command (concat shape-bin-dir "save"))
(defvar shape-retrv-command (concat shape-bin-dir "retrv"))
(defvar shape-submit-command (concat shape-bin-dir "sbmt"))
(defvar shape-vlog-command (concat shape-bin-dir "vlog"))
(defvar shape-vdiff-command (concat shape-bin-dir "vdiff"))
(defvar shape-pattr-command (concat shape-bin-dir "pattr"))
(defvar shape-bourne-shell "/bin/sh" "Shell compatible with Bourne Shell")

(defvar shape-listing-switches "-al"
  "Switches passed to ls for shape. MUST contain the 'l' option.
	CANNOT contain the 'F' option.")
(defvar shape-compare-file1 nil)

(defvar shape-wish-address "shape-cr@coma.cs.tu-berlin.de"
  "The mail address to report a wish.")
(defvar shape-bug-address "shape-cr@coma.cs.tu-berlin.de" 
  "The mail address to report a bug.")
(defvar shape-bug-description 
  "Description:\n\nRepeat-By:\n\nFix:\n\nShape Toolkit version:\n\n" 
  "Formular to report a bug")


(defun shape-readin (dirname buffer)
  (save-excursion
    (set-buffer buffer)
    (let ((buffer-read-only nil))
      (widen)
      (erase-buffer)
      (setq dirname (expand-file-name dirname))
      (if (file-directory-p dirname)
	  (call-process-with-error (concat shape-vl-command " "
					   shape-listing-switches " "
					   dirname)
				   buffer)
	(let ((default-directory (file-name-directory dirname)))
	  (call-process-with-error (concat shell-file-name " -c '"
					   shape-vl-command " "
					   shape-listing-switches " "
					   (file-name-nondirectory dirname) 
					   "'")
				   buffer)))
      (goto-char (point-min))
      (while (not (eobp))
	(insert "  ")
	(move-to-column 28)
	(delete-char 7)
	(move-to-column 49)
	(delete-char 3)
	(forward-line 1))
      (if shape-fold-initially
	  (shape-fold-directory))
      (goto-char (point-min))
      (shape-move-to-filename)
      (message ""))))

(defun shape-find-buffer (dirname)
  (let ((blist (buffer-list))
	found)
    (while blist
      (save-excursion
	(set-buffer (car blist))
	(if (and (eq major-mode 'shape-mode)
		 (equal shape-directory dirname))
	    (setq found (car blist)
		  blist nil)
	  (setq blist (cdr blist)))))
    (or found
	(progn (if (string-match "/$" dirname)
		   (setq dirname (substring dirname 0 -1)))
	       (create-file-buffer (file-name-nondirectory dirname))))))

(defun shapetools (dirname) 
"\"Edit\" directory DIRNAME and version histories with the Shape Toolkit\
 commands.
Shape displays a list of files and saved versions in DIRNAME.  You can
move around in it with the usual commands.  You can save, retrieve,
and compare versions, execute Shape, and issue several file
manipulation commands.
Type `h' after entering shapetools for more info."
  (interactive "DShapetools (directory): ")
  ;; Some day this will support 
  (switch-to-buffer (shape-noselect dirname)))

(defun shape-other-window (dirname)
  "\"Edit\" directory DIRNAME.	Like M-x shape but selects in another window."
  (interactive (list (read-file-name "Shapetools in other window (directory): "
				     nil default-directory nil)))
  (switch-to-buffer-other-window (shape-noselect dirname)))

(defun shape-noselect (dirname)
  "Like M-x shapetools but returns the shape buffer as value, does not select it."
  (or dirname (setq dirname default-directory))
  (if (string-match "./$" dirname)
      (setq dirname (substring dirname 0 -1)))
  (setq dirname (expand-file-name dirname))
  (and (not (string-match "/$" dirname))
       (file-directory-p dirname)
       (setq dirname (concat dirname "/")))
  (let ((buffer (shape-find-buffer dirname)))
    (save-excursion
      (set-buffer buffer)
      (shape-readin dirname buffer)
      (shape-move-to-filename)
      (shape-mode dirname))
    buffer))

(defun shape-revert (&optional arg noconfirm)
  (message "Reading...")
  (let ((opoint (point))
	(ofile (shape-get-filename t t))
	(buffer-read-only nil))
    (erase-buffer)
    (shape-readin shape-directory (current-buffer))
    (or (and ofile (re-search-forward (concat " " (regexp-quote ofile) "$")
				      nil t))
	(goto-char opoint))
    (beginning-of-line)))

(defvar shape-mode-map nil "Local keymap for shape-mode buffers.")
(if shape-mode-map
    nil
  (setq shape-mode-map (make-keymap))
  (suppress-keymap shape-mode-map)
  (define-key shape-mode-map " "  'shape-next-line)
  (define-key shape-mode-map "?" 'shape-summary)
  (define-key shape-mode-map "A" 'shape-vadm-change-author)
  (define-key shape-mode-map "B" 'shape-mail-bugs)
  (define-key shape-mode-map "C" 'shape-compare)
  (define-key shape-mode-map "E" 'shape-execute)
  (define-key shape-mode-map "F" 'shape-fold)
  (define-key shape-mode-map "L" 'shape-vadm-lock)
  (define-key shape-mode-map "M" 'shape-vadm-change-mode)
  (define-key shape-mode-map "O" 'shape-vadm-change-owner)
  (define-key shape-mode-map "P" 'shape-vadm-promote)
  (define-key shape-mode-map "R" 'shape-retrv)
  (define-key shape-mode-map "S" 'shape-save)
  (define-key shape-mode-map "U" 'shape-vadm-unpromote)
  (define-key shape-mode-map "V" 'shape-vadm)
  (define-key shape-mode-map "W" 'shape-mail-wishes)
  (define-key shape-mode-map "X" 'shape-unfold)
  (define-key shape-mode-map "a" 'shape-add-attribute)
  (define-key shape-mode-map "c" 'shape-copy-file)
  (define-key shape-mode-map "d" 'shape-flag-file-deleted)
  (define-key shape-mode-map "e" 'shape-find-file)
  (define-key shape-mode-map "f" 'shape-find-file)
  (define-key shape-mode-map "g" 'revert-buffer)
  (define-key shape-mode-map "h" 'describe-mode)
  (define-key shape-mode-map "l" 'shape-vlog)
  (define-key shape-mode-map "n" 'shape-next-line)
  (define-key shape-mode-map "o" 'shape-find-file-other-window)
  (define-key shape-mode-map "p" 'shape-previous-line)
  (define-key shape-mode-map "q" '(lambda () (interactive) 
				    (kill-buffer (current-buffer))))
  (define-key shape-mode-map "r" 'shape-rename-file)
  (define-key shape-mode-map "s" 'shape-show-attribute)
  (define-key shape-mode-map "u" 'shape-unflag)
  (define-key shape-mode-map "v" 'shape-view-file)
  (define-key shape-mode-map "x" 'shape-do-deletions)
  (define-key shape-mode-map "\177" 'shape-backup-unflag)
  (define-key shape-mode-map "\C-d" 'shape-flag-file-deleted)
  (define-key shape-mode-map "\C-n" 'shape-next-line)
  (define-key shape-mode-map "~" 'shape-flag-backup-files))
  


;; Shape mode is suitable only for specially formatted data.
(put 'shape-mode 'mode-class 'special)

(defun shape-mode (dirname)
  "\
- M change file's mode.			 - ~ flag backup files for deletion.
- G change group.			 - d flag a file for deletion.        
- O change owner.			 - u unflag a file (remove its D flag).
- A change author.			 - x execute the deletions requested.  
- P promote a saved version.		 - e edit file or list directory.      
- U unpromote a saved version.		 - o find file/directory other window. 
- C compare two files.			 - W mail wishes (B to mail a bug).    
- R retrieve a version.			 - c copy a file.                      
- L lock a history.                      - r rename a file.                    
- S save a busy version.		 - v view a file in View mode.
- F fold directory			 - g read the directory again.
- X unfold file or directory		 - E execute shape
- l show logentry			 - a add an attribute
Space and Rubout can be used to move down and up by lines.
\\{shape-mode-map}"
  (kill-all-local-variables)	
  (make-local-variable 'revert-buffer-function)
  (setq revert-buffer-function 'shape-revert)
  (setq major-mode 'shape-mode)
  (setq mode-name "Shape")
  (make-local-variable 'shape-directory)
  (setq shape-directory dirname)
  (setq default-directory 
	(if (file-directory-p dirname)
	    dirname (file-name-directory dirname)))
  (setq mode-line-buffer-identification '("Shape Tools: %17b"))
  (setq case-fold-search nil)
  (setq buffer-read-only t)
  (use-local-map shape-mode-map)
  (run-hooks 'shape-mode-hook))

(defun shape-repeat-over-lines (arg function)
  (beginning-of-line)
  (while (and (> arg 0) (not (eobp)))
    (setq arg (1- arg))
    (save-excursion
      (beginning-of-line)
      (and (bobp) (looking-at "	 total")
	   (error "No file on this line"))
      (funcall function))
    (forward-line 1)
    (shape-move-to-filename))
  (while (and (< arg 0) (not (bobp)))
    (setq arg (1+ arg))
    (forward-line -1)
    (shape-move-to-filename)
    (save-excursion
      (beginning-of-line)
      (funcall function))))

(defun shape-flag-file-deleted (&optional arg)
  "In shape, flag the current line's file for deletion.
With arg, repeat over several lines."
  (interactive "p")
  (shape-repeat-over-lines (or arg 1)
    '(lambda ()
       (let ((buffer-read-only nil))
	 (cond ((file-folded-p (shape-get-filename))
		(shape-move-to-filename)
		(sit-for 1)
		(message "Folded histories may not be deleted."))
	       ((looking-at "  d") nil)
	       ((or (looking-at "  .......... s ")
		    (looking-at "  .......... b "))
		(delete-char 1)
		(insert "D"))
	       (t (sit-for 1)
		  (message "Only saved or busy versions may be deleted")))))))

(defun shape-summary ()
  (interactive)
  ;;>> this should check the key-bindings and use 
  ;;   substitute-command-keys if non-standard
  (message
   "Commands: ACFGMOPSUX cdegoruvx \(h for more help\)"))

(defun shape-unflag (arg)
  "In shape, remove the current line's delete flag then move to next line."
  (interactive "p")
  (shape-repeat-over-lines arg
    '(lambda ()
       (let ((buffer-read-only nil))
	 (delete-char 1)
	 (insert " ")
	 (forward-char -1)))))

(defun shape-backup-unflag (arg)
  "In shape, move up a line and remove deletion flag there."
  (interactive "p")
  (shape-unflag (- arg)))

(defun shape-next-line (arg)
  "Move down ARG lines then position at filename."
  (interactive "p")
  (next-line arg)
  (shape-move-to-filename))

(defun shape-previous-line (arg)
  "Move up ARG lines then position at filename."
  (interactive "p")
  (previous-line arg)
  (shape-move-to-filename))

(defun shape-find-file ()
  "In shape, visit the file or directory named on this line."
  (interactive)
  (if (file-folded-p (shape-get-filename))
      (let ((shape-fold-initially nil))
	(shapetools (substring (shape-get-filename) 0 -3)))
    (if (file-AFS-p (shape-get-filename))
	(message "Can't edit a version or folded history")
      (find-file (shape-get-filename)))))

(defun shape-view-file ()
  "In shape, examine a file in view mode, returning to shape when done."
  (interactive)
  (if (file-directory-p (shape-get-filename))
      (shapetools (shape-get-filename))
    (if (file-folded-p (shape-get-filename))
	(let ((shape-fold-initially nil))
	  (message "Reading...")
	  (shapetools (substring (shape-get-filename) 0 -3)))
      (if (file-AFS-p (shape-get-filename))
	  (shape-vcat)
	(view-file (shape-get-filename))))))
	    
(defun shape-find-file-other-window ()
  "In shape, visit this file or directory in another window."
  (interactive)
  (if (file-folded-p (shape-get-filename))
      (let ((shape-fold-initially nil))
	(shape-other-window (substring (shape-get-filename) 0 -3)))
    (if (file-AFS-p (shape-get-filename))
	(message "Can't edit a version")
      (if (file-DIR-p)
	  (shape-other-window (shape-get-filename))
	(find-file-other-window (shape-get-filename))))))

(defun shape-get-filename (&optional localp no-error-if-not-filep)
  "In shape, return name of file mentioned on this line.
Value returned normally includes the directory name.
A non-nil 1st argument means do not include it.	 A non-nil 2nd argument
says return nil if no filename on this line, otherwise an error occurs."
  (let (eol)
    (save-excursion
      (end-of-line)
      (setq eol (point))
      (beginning-of-line)
      (if (re-search-forward 
	   (concat "\\(Jan\\|Feb\\|Mar\\|Apr\\|May\\|Jun\\|" 
		   "Jul\\|Aug\\|Sep\\|Oct\\|Nov\\|Dec\\)[ ]+[0-9]+")
	   eol t)
	  (progn (skip-chars-forward " ")
		 (skip-chars-forward "^ " eol)
		 (skip-chars-forward " " eol)
		 (skip-chars-forward "^ " eol)
		 (skip-chars-forward " " eol)
		 (let ((beg (point)))
		   (skip-chars-forward "^ \n")
		   (if localp
		       (buffer-substring beg (point))
		     ;; >> uses default-directory, could lose on cd, multiple.
		     (concat default-directory 
			     (buffer-substring beg (point))))))
	(if no-error-if-not-filep nil
	  (error "No file on this line"))))))

(defun shape-move-to-filename ()
  "In shape, move to first char of filename on this line.
Returns position (point) or nil if no filename on this line."
  (let ((eol (progn (end-of-line) (point))))
    (beginning-of-line)
    (if (re-search-forward
	 (concat "\\(Jan\\|Feb\\|Mar\\|Apr\\|May\\|Jun\\|Jul\\|"
		 "Aug\\|Sep\\|Oct\\|Nov\\|Dec\\)[ ]+[0-9]+")
	 eol t)
	(progn
	  (skip-chars-forward " ")
	  (skip-chars-forward "^ " eol)
	  (skip-chars-forward " " eol)
	  (skip-chars-forward "^ " eol)
	  (skip-chars-forward " " eol)
	  (point)))))

(defun shape-map-shape-file-lines (fn)
  "perform fn with point at the end of each non-directory line:
arguments are the short and long filename"
  (save-excursion
    (let (filename longfilename (buffer-read-only nil))
      (goto-char (point-min))
      (while (not (eobp))
	(save-excursion
	  (and (not (looking-at "  d"))
	       (not (eolp))
	       (setq filename (shape-get-filename t t)
		     longfilename (shape-get-filename nil t))
	       (progn (end-of-line)
		      (funcall fn filename longfilename))))
	(forward-line 1)))))

(defun shape-collect-file-versions (ignore fn)
  "If it looks like fn has versions, we make a list of the versions.
We may want to flag some for deletion."
    (let* ((base-versions
	    (concat (file-name-nondirectory fn) ".~"))
	   (bv-length (length base-versions))
	   (possibilities (file-name-all-completions
			   base-versions
			   (file-name-directory fn)))
	   (versions (mapcar 'backup-extract-version possibilities)))
      (if versions
	  (setq file-version-assoc-list (cons (cons fn versions)
					      file-version-assoc-list)))))

(defun shape-trample-file-versions (ignore fn)
  (let* ((start-vn (string-match "\\.~[0-9]+~$" fn))
	 base-version-list)
    (and start-vn
	 (setq base-version-list	; there was a base version to which 
	       (assoc (substring fn 0 start-vn) ; this looks like a 
		      file-version-assoc-list)) ; subversion
	 (not (memq (string-to-int (substring fn (+ 2 start-vn)))
		    base-version-list)) ; this one doesn't make the cut
	 (shape-flag-this-line-for-DEATH))))

(defun shape-flag-this-line-for-DEATH ()
  (beginning-of-line)
  (delete-char 1)
  (insert "D"))

(defun shape-rename-file (to-file)
  "Rename this file to TO-FILE."
  (interactive "FRename to: ")
  (setq to-file (expand-file-name to-file))
  (let ((file (shape-get-filename))
	(buffer-read-only nil))
    (if (file-AFS-p file)
	(message "Can't rename a version or folded history")
      (rename-file file to-file)
      (beginning-of-line)
      (delete-region (point) (progn (forward-line 1) (point)))
      (setq to-file (expand-file-name to-file))
      (shape-add-entry (file-name-directory to-file)
		       (file-name-nondirectory to-file)))))

(defun shape-copy-file ()
  "Copy this file to TO-FILE."
  (interactive)
  (let ((from-file (shape-get-filename t)))
  (if (file-AFS-p (shape-get-filename t))
      (message "Can't copy saved files")
    (setq to-file (read-string (concat "Copy " from-file " to: ")
			       default-directory))
    (copy-file (shape-get-filename) to-file)
    (setq to-file (expand-file-name to-file))
    (shape-add-entry (file-name-directory to-file)
		     (file-name-nondirectory to-file)))))

(defun shape-add-entry (directory filename)
  ;; If tree shape is implemented, this function will have to do
  ;; something smarter with the directory.  Currently, just check
  ;; default directory, if same, add the new entry at point.  With tree
  ;; shape, should call 'shape-current-directory' or similar.  Note
  ;; that this adds the entry 'out of order' if files sorted by time,
  ;; etc.
  (if (string-equal directory default-directory)
      (let ((buffer-read-only nil))
	(beginning-of-line)
	(if (file-AFS-p filename)
	    (call-process-with-error (concat shape-vl-command " "
					     shape-listing-switches " "
					     directory filename)
				     t)
	  (call-process-with-error (concat shape-vl-command " "
					   shape-listing-switches " "
					   "-sb "
					   directory filename)
				   t))
	(forward-line -1)
	(insert "  ")
	(move-to-column 28)
	(delete-char 7)
	(move-to-column 49)
	(delete-char 3)
	(shape-move-to-filename)
	(let* ((beg (point))
	       (end (progn (end-of-line) (point))))
	  (setq filename (buffer-substring beg end))
	  (delete-region beg end)
	  (insert (file-name-nondirectory filename)))
	(beginning-of-line))))

(defun shape-chgrp (group)
  "Change group of this file."
  (interactive "sChange to Group: ")
  (let ((buffer-read-only nil)
	(file (shape-get-filename)))
    (call-process-with-error (concat "/bin/chgrp " group " " file) nil)
    (shape-redisplay file)))

(defun shape-goto-file (file)
  "Go to entry of FILE, filename position."
  (beginning-of-buffer)
  (if (re-search-forward (concat " " (regexp-quote file) "$") (point-max) t)
      (shape-move-to-filename)
    (error "File %s not found" file)))

(defun shape-redisplay (file)
  "Redisplay this file."
  (shape-goto-file file2)
  (beginning-of-line)
  (delete-region (point) (progn (forward-line 1) (point)))
  (if file (shape-add-entry default-directory
			    (file-name-nondirectory file)))
  (shape-move-to-filename))

(defun shape-do-deletions ()
  "In shape, delete the files flagged for deletion."
  (interactive)
  (let (delete-list answer)
    (save-excursion
     (goto-char 1)
     (while (re-search-forward "^D" nil t)
       (setq delete-list
	     (cons (cons (shape-get-filename t) (1- (point)))
		   delete-list))))
    (if (null delete-list)
	(message "(No deletions requested)")
      (save-window-excursion
       (switch-to-buffer " *Deletions*")
       (erase-buffer)
       (setq fill-column 70)
       (let ((l (reverse delete-list)))
	 ;; Files should be in forward order for this loop.
	 (while l
	   (if (> (current-column) 59)
	       (insert ?\n)
	     (or (bobp)
		 (indent-to (* (/ (+ (current-column) 19) 20) 20) 1)))
	   (insert (car (car l)))
	   (setq l (cdr l))))
       (goto-char (point-min))
       (setq answer (yes-or-no-p "Delete these files? ")))
      (if answer
	  (let ((l delete-list)
		failures)
	    ;; Files better be in reverse order for this loop!
	    ;; That way as changes are made in the buffer
	    ;; they do not shift the lines still to be changed.
	    (while l
	      (goto-char (cdr (car l)))
	      (let ((buffer-read-only nil))
		(condition-case ()
		    (progn (shape-delete-file (concat default-directory
						      (car (car l))))
			   (delete-region (point)
					  (progn (forward-line 1) (point))))
		  
		  (error (delete-char 1)
			 (insert " ")
			 (setq failures (cons (car (car l)) failures)))))
	      (setq l (cdr l)))
	    (if failures
		(message "Deletions failed: %s"
			 (prin1-to-string failures))))))))


(defun shape-vcat()
  "Retrieve an old version and display it."
  (interactive)
  (message "Restoring %s ..." (shape-get-filename t))
  (setq vcat-buffer (create-file-buffer (shape-get-filename)))
  (call-process-with-error (concat shape-vcat-command " -q "
				   (shape-get-filename))
			   vcat-buffer)
  (let ((view-hook '(beginning-of-buffer)))
    (view-buffer vcat-buffer))
  (kill-buffer vcat-buffer))

(defun shape-vlog()
  "Display logentry for a particular version or entire history."
  (interactive)
  (if (file-directory-p (shape-get-filename))
      (error "Directories don't have any log-entries")
    (if (file-folded-p (shape-get-filename))
	(progn 
	  (setq history-filename (substring (shape-get-filename) 0 -3))
	  (setq msg-string 
		(concat "History log for " history-filename)))
      (if (file-AFS-p (shape-get-filename))
	  (progn (setq history-filename (shape-get-filename))
		 (setq msg-string (concat "Log entry for " history-filename)))
	(setq history-filename (shape-get-filename))
	(setq msg-string (concat "History log for " history-filename))))

    (setq vlog-buffer (create-file-buffer msg-string))
    (message (concat "Viewing " msg-string))
    (call-process-with-error (concat shape-vlog-command " "
				     history-filename)
			     vlog-buffer)
    (let ((view-hook '(beginning-of-buffer)))
      (view-buffer vlog-buffer))
    (kill-buffer vlog-buffer)))

(defun shape-vadm (vadm-input)
  "Perform vadm features."
  (interactive "svadm: ")
  (let ((buffer-read-only nil)
	(file (shape-get-filename)))
    (call-process-with-error (concat shape-vadm-command " -q " 
				     vadm-input " " file)
			     nil)
    (shape-redisplay file)))

(defun shape-vadm-lock ()
  "Perform vadm -lock, asks for intent message."
  (interactive)
  (let ((buffer-read-only nil)
	(file (shape-get-filename)))
    (if (y-or-n-p "Describe intended changes? ")
	(let ((descfile (make-temp-name "/tmp/vadm")))
	  (shape-get-description descfile)
	  (call-process-with-error (concat shape-vadm-command 
					   " -q -lock -stdin " 
					   file " < " descfile)
				   nil)
	  (kill-buffer (get-file-buffer descfile))
	  (delete-file descfile))
      (call-process-with-error (concat shape-vadm-command 
				       " -q -lock " file)
			       nil))
    (shape-redisplay file)))

(defun shape-vadm-promote()
  "Performs vadm -promote."
  (interactive)
  (let ((buffer-read-only nil)
	(file (shape-get-filename t))
	(file2 (shape-get-filename)))
    (if (not (file-AFS-p file))
	(message "Can't promote busy file or directory %s" file)
      (if (file-folded-p file)
	  (message "Can't promote folded history %s" file)
	(message "Promoting %s ..." file)
	(if (not (call-process-with-error (concat shape-vadm-command 
						  " -q -promote "
						  file2)
					  nil 'noerror))
	    (message "Could not promote %s" file)
	  ;;	(view-buffer (current-buffer))
	  (shape-redisplay file2)
	  (message "Done."))))))

(defun shape-vadm-unpromote()
  "Performs vadm -unpromote."
  (interactive)
  (let ((buffer-read-only nil)
	(file (shape-get-filename t))
	(file2 (shape-get-filename)))
    (if (not (file-AFS-p file))
	(message "Can't unpromote busy file or directory %s" file)
      (if (file-folded-p file)
	  (message "Can't unpromote folded history %s" file)
	(message "Unpromoting %s ..." file)
	(if (not (call-process-with-error (concat shape-vadm-command 
						  " -q -unpromote "
						  file2)
					  nil 'noerror))
	    (message "Could not unpromote %s" file)
	  (shape-redisplay file2)
	  (message "Done."))))))

(defun shape-vadm-change-mode()
  "Performs vadm -chmod."
  (interactive)
  (let ((buffer-read-only nil)
	(file (shape-get-filename t))
	(file2 (shape-get-filename)))
    (if (file-folded-p file2)
	(message "Can't chmod folded history")
      (setq input (read-string (concat "Change mode of " file " to: ")))
      (if (file-AFS-p file2)
	  (call-process-with-error (concat shape-vadm-command " -q -chmod "
					   input " " file2)
				   nil)
	(call-process-with-error (concat "/bin/chmod " input " " file2)
				 nil))
      (shape-redisplay file2)
      (message "Done."))))

(defun shape-vadm-change-author()
  "Performs vadm -chaut."
  (interactive)
  (let ((buffer-read-only nil)
	(file (shape-get-filename t))
	(file2 (shape-get-filename)))
    (setq input (read-string (concat "Change author of " file " to: ")))
    (call-process-with-error (concat shape-vadm-command " -q -chaut "
				     input " " file2)
			     nil)
    (shape-redisplay file2)
    (message "Done.")))


(defun shape-vadm-change-owner()
  "Performs vadm -chown."
  (interactive)
  (let ((buffer-read-only nil)
	(file (shape-get-filename t))
	(file2 (shape-get-filename)))
    (setq input (read-string (concat "Change owner of " file " to: ")))
    (call-process-with-error (concat shape-vadm-command " -q -chown " 
				     input " " file2)
			     nil)
    (shape-redisplay file2)
    (message "Done.")))


(defun shape-save ()
  "saves a file via the save command."
  (interactive)
  (save-excursion
    (let* ((buffer-read-only nil)
	   (file (shape-get-filename))
	   (file2 (shape-get-filename t))
	   (save-command (concat shape-save-command " -f -q " file))
	   (descfile nil))
      (if (or (file-AFS-p file) (file-DIR-p))
	  (message "This file is not a busy file or a directory")
	(if (y-or-n-p "Describe this document or changes? ")
	    (progn
	      (setq descfile (make-temp-name "/tmp/save"))
	      (call-process-with-error (concat shape-vl-command
					       " -last -intent "
					       file " > " descfile
					       " 2> /dev/null")
				       nil 'noerror)
	      (shape-get-description descfile)
	      (setq save-command (concat shape-save-command " -f -q -t "
					 descfile " " file))
	      (kill-buffer (get-file-buffer descfile))))
	(message "Saving file %s" file2)
	(call-process-with-error save-command t)
	(if descfile (delete-file descfile))
	(beginning-of-line)
	(kill-line 1)
	(message "Busy version of %s removed" file2)
	(shape-insert-new-version file2)
	(while (search-forward file2 nil t))))))

(defun shape-submit ()
  "submit a file via the submit command."
  (interactive)
  (save-excursion
    (let ((buffer-read-only nil)
	  (file (shape-get-filename))
	  (file2 (shape-get-filename t)))
      (if (or (file-AFS-p file) (file-DIR-p))
	  (message "This file is not a busy file or a directory")
	(message "Submitting file %s" file2)
	(call-process-with-error (concat shape-submit-command " -f -q " file)
				 t)
	(revert-buffer)))))


(defun shape-retrv()
  "Retrieves a version via the retrv command."
  (interactive)
  (save-excursion
    (let ((buffer-read-only nil))
	  (setq file (shape-get-filename t))
	  (setq file2 (substring file 0 (string-match "\\\[" file)))
      (if (not (file-AFS-p file))
	  (message "This file is not a saved file")
	(if (file-folded-p file)
	    (message "Can't retrieve a folded history.")
	  (if (file-exists-p file2)
	      (progn
		(if (y-or-n-p (concat "Writable busy version of "
				      file2
				      " exists! Overwrite it?"))
		    (progn
		      (shape-exec-retrv-command file)
		      (shape-redisplay file2)
		      (message "%s retrieved" file))))
	    (shape-exec-retrv-command file)
	    (if (file-exists-p file2)
		(progn
		  (while (search-backward (concat file2 "[") (point-min) t))
		  (shape-add-entry default-directory file2)
		  (message "%s retrieved." file))
	      (message "%s probably locked by somebody else" file2))))))))

(defun shape-exec-retrv-command (file)
  "This is only to be used by shape-retrv. 
It asks for an intent message and retrieves FILE."
    (if (y-or-n-p "Describe intended changes? ")
	(let ((descfile (make-temp-name "/tmp/save")))
	  (shape-get-description descfile)
	  (call-process-with-error (concat shape-retrv-command 
					   " -f -q -lock -i " 
					   descfile " " file)
				   nil)
	  (kill-buffer (get-file-buffer descfile))
	  (delete-file descfile))
      (call-process-with-error (concat shape-retrv-command 
				       " -f -q -lock " file)
			       nil)))


(defun shape-compare ()
  "compares two versions with diff and puts output into a view buffer."
  (interactive)
  (save-excursion
    (setq shape-buffer1 nil)
    (setq shape-buffer2 nil)
    (if (eq shape-compare-file1 nil)
	(progn
	  (if (or (file-DIR-p) (file-folded-p (shape-get-filename t)))
	      (message "Cant't compare directories or folded histories")
	    (setq shape-compare-file1 nil)
	    (setq shape-compare-file1 (shape-get-filename t))
	    (message "Compare %s with ? \(goto file2 and hit C again\)"
		     shape-compare-file1)
	    (shape-flag-file-compare "<")))
      
      (setq shape-compare-file2 (shape-get-filename t))
      (if (or (file-DIR-p) (file-folded-p shape-compare-file2))
	  (progn
	    (shape-unflag-file-compare)
	    (message "Cant't compare directories or folded histories"))
	(shape-flag-file-compare ">")
	(message "Comparing %s with %s" shape-compare-file1
		 shape-compare-file2)
	(setq diff-buffer (generate-new-buffer "diff"))
	(call-process shape-vdiff-command nil diff-buffer 'display
		      shape-compare-file1 shape-compare-file2)
	(shape-unflag-file-compare)
	(save-excursion
	  (if (prog2 (set-buffer diff-buffer)
		     (zerop (buffer-size)))
	      (message "%s and %s are identical"
		       shape-compare-file1 shape-compare-file2)
	    (let ((view-hook '(beginning-of-buffer)))
	      (goto-char (point-min))
	      (insert shape-compare-file1 " < > " shape-compare-file2 "\n")
	      (view-buffer diff-buffer)))))
      (setq shape-compare-file1 nil)
      (setq shape-compare-file2 nil)
      (kill-buffer diff-buffer))))

(defun shape-fold-directory ()
  "Fold whole directory."
  (message "Folding directory ...")
  (goto-char (point-min))
  (while (search-forward "[" nil t)
    (setq filename (shape-get-filename t t))
    (kill-line 1)
    (insert "*]")
    (newline)
    (setq filename2 (substring
		     filename 0 (string-match "\\\[" filename)))
    (setq filename2 (concat filename2 "\\\["))
    (delete-matching-lines filename2))
  (message "Done."))

(defun shape-fold ()
  "Compresses output; files with versions are displayed with <name>[*]."
  (interactive)
  (save-excursion
    (let ((buffer-read-only nil))
      (if (y-or-n-p "Fold whole directory? ")
	  (shape-fold-directory)
	(setq filename (shape-get-filename t))
	(if (file-AFS-p filename)
	    (progn
	      (setq filename2 (substring
			       filename 0 (string-match "\\\[" filename)))
	      (goto-char (point-min))
	      (search-forward (concat filename2 "["))
	      (beginning-of-line)
	      (search-forward "[" nil t)
	      (kill-line 1)
	      (insert "*]")
	      (newline)
	      (delete-matching-lines filename2)
	      (sit-for 0)
	      (message "Done."))
	  (message "No version: %s" filename))))))

(defun shape-unfold()
  "Expands folded entries."
  (interactive)
  (save-excursion
    (let ((buffer-read-only nil))
      (if (y-or-n-p "Unfold whole directory? ")
	  (progn
	    (message "Unfolding directory ...")
	    (let ((shape-fold-initially nil))
	      (revert-buffer))
	    (message "Done."))
	(if (equal (substring (shape-get-filename t)
			      -3 (length (shape-get-filename t)))
		   "[*]")
	    (let ((shape-fold-initially nil))
	      (setq filename (substring (shape-get-filename t) 0 -3))
	      (message "Unfolding %s ..." filename)
	      (beginning-of-line)
	      (kill-line 1)
	      (call-process-with-error (concat shape-vl-command " " 
					       shape-listing-switches
					       " -ss+ "
					       " "
					       filename " | grep '\\['")
				       t)
	      (shape-update-buffer)
	      (sit-for 0)
	      (message "Done.")
	      (shape-move-to-filename))
	  (message "File not folded."))))))
	      

(defun file-AFS-p(name)
  "decides whether a file is an AFS file or not (']' as last char)."
  (if (string-match "]" name) t nil))

(defun file-DIR-p()
  (beginning-of-line)
  (looking-at "	 d"))

(defun file-folded-p(file)
  (if (equal (substring file -3 (length file)) "[*]")
      t
    nil))

(defun shape-insert-new-version (file)
  "update buffer after save command."
  (interactive)
  (let ((buffer-read-only nil))
    (while (search-forward file nil t))
    (forward-line)
    (beginning-of-line)
    (call-process-with-error (concat shape-vl-command " "
				     shape-listing-switches " -last "
				     file)
			     t)
    (forward-line -1)
    (insert "  ")
    (move-to-column 28)
    (delete-char 7)
    (move-to-column 49)
    (delete-char 3)
    (shape-move-to-filename)))


(defun shape-get-description (descfile &optional initial-contents)
  "Read the description for a save, retrv, or lock command. DESCFILE is the 
file to edit, optional INITIAL-CONTENTS is inserted into the buffer before
editing."
  (save-excursion
    (find-file descfile)
    (if initial-contents (insert initial-contents))
    (local-set-key "\C-c\C-c" 'exit-recursive-edit)
    (message (substitute-command-keys "To stop type \\[exit-recursive-edit]"))
    (recursive-edit)
    (beginning-of-buffer)
    (write-file descfile)))


(defun shape-delete-file (file)
  (if (file-AFS-p file)
      (call-process-with-error (concat shape-vadm-command " -delete " file) 
			       nil)
    (delete-file file)))

(defun shape-flag-file-compare (mark)
  (let ((buffer-read-only nil))
    (save-excursion
      (beginning-of-line)
      (delete-char 1)
      (insert mark)
      (sit-for 0))))

(defun shape-unflag-file-compare ()
  (let ((buffer-read-only nil))
    (save-excursion
      (beginning-of-buffer)
      (re-search-forward "^[><]")
      (beginning-of-line)
      (delete-char 1)
      (insert " ")
      (re-search-forward "^[><]")
      (beginning-of-line)
      (delete-char 1)
      (insert " "))))
      
(defun shape-update-buffer()
  "Updates buffer after unfold."
  (interactive)
  (save-excursion
  (goto-char (point-min))
  (while (re-search-forward "^-" nil t)
    (beginning-of-line)
    (insert "  ")
    (move-to-column 28)
    (delete-char 7)
    (move-to-column 49)
    (delete-char 3))))

(defun shape-execute()
  "sets compile command to shape -k."
  (interactive)
  (save-excursion
    (setq filename (shape-get-filename t t))
    (setq shapefile nil)
    (setq promptstring nil)
    (setq basename (substring
		     filename 0 (string-match "\\\[" filename)))
    (if (or (equal basename "Shapefile")
	    (equal basename "shapefile")
	    (equal basename "Makefile")
	    (equal basename "makefile")
	    (equal filename "Shapefile")
	    (equal filename "shapefile")
	    (equal filename "Makefile")
	    (equal filename "makefile"))
	(setq shapefile filename)
      (setq shapefile nil))
    (if (file-folded-p filename)
	(setq shapefile nil)
      nil)
    (if (equal shapefile nil)
	(setq promptstring "shape -k ")
      (if (file-AFS-p filename)
	    (setq promptstring (concat "vcat " "\""
				       filename
				       "\"" " | shape -f - "))
	(setq promptstring (concat "shape -k -f " filename " "))))
    (setq input (read-string "shape: " promptstring))
    (if (equal input nil)
	(compile promptstring)
      (compile input))))

(defun shape-mail-bugs () 
  (interactive)
  (mail nil shape-bug-address)
  (goto-char (point-min))
  (next-line 1)
  (beginning-of-line)
  (insert "Index: <tool>/<source> <confid>\n")
  (goto-char (point-max))
  (insert shape-bug-description "\n")
  (mail-position-on-field "Subject")
  (message (substitute-command-keys "Type \\[mail-send] to send bug report.")))

(defun shape-mail-wishes ()
  (interactive)
  (mail nil shape-wish-address)
  (mail-position-on-field "Subject")
  (message (substitute-command-keys
	    "Type \\[mail-send] to send wish report.")))


(defun shape-execute-vl()
  "Executes vl reading parameters from the minibuffer."
  (interactive)
  (setq input (read-string "vl: "))
  (shell-command (concat shape-vl-command " " input)))
  
(defun shape-execute-save()
  "Executes save reading parameters from the minibuffer."
  (interactive)
  (setq input (read-string "save: "))
  (shell-command (concat shape-save-command " " input)))

(defun shape-execute-submit()
  "Executes submit reading parameters from the minibuffer."
  (interactive)
  (setq input (read-string "sbmt: "))
  (shell-command (concat shape-submit-command " " input)))

(defun shape-execute-retrv()
  "Executes retrv reading parameters from the minibuffer."
  (interactive)
  (setq input (read-string "retrv: "))
  (shell-command (concat shape-retrv-command " " input)))
  
(defun shape-execute-vadm()
  "Executes vadm reading parameters from the minibuffer."
  (interactive)
  (setq input (read-string "vadm: "))
  (shell-command (concat shape-vadm-command " " input)))
 
(defun shape-execute-vcat()
  "Executes vcat reading parameters from the minibuffer."
  (interactive)
  (setq input (read-string "vcat: "))
  (shell-command (concat shape-vcat-command " " input)))

(defun shape-execute-vlog()
  "Executes vlog reading parameters from the minibuffer."
  (interactive)
  (setq input (read-string "vlog: "))
  (shell-command (concat shape-vlog-command " " input)))

(defun shape-add-attribute()
  (interactive)
  (if (not (file-DIR-p))
      (progn
	(setq attr-filename (shape-get-filename))
	(setq attr-input (read-string "Enter attribute \(<name>=<value>\): "))
	(call-process-with-error (concat shape-vadm-command " -q -setuda "
					 attr-input " "attr-filename)
				 nil)
	(message "Done."))
    (message "Can't add an attribute to a directory")))

(defun shape-show-attribute()
  (interactive)
  (save-excursion
    (if (not (file-DIR-p))
	(progn
	  (setq attr-filename (shape-get-filename))
	  (setq msg-string (concat "Attributes of " attr-filename))
	  (setq vl-attrbuffer (create-file-buffer msg-string))
	  (if (y-or-n-p "Show attributes of all files? ")
	      (call-process-with-error (concat shape-vl-command " -Uux")
				       vl-attrbuffer)
	    (call-process-with-error (concat shape-vl-command " -Uux "
					     attr-filename)
				     vl-attrbuffer))
	  (message (concat "Viewing attributes of " attr-filename))
	  (set-buffer vl-attrbuffer)
	  (rename-buffer "Attributes")
	  (let ((view-hook '(beginning-of-buffer)))
	    (view-buffer vl-attrbuffer))
	  (kill-buffer vl-attrbuffer))
      (message "Directories don't have attributes \(yet?\)"))))

(defun shape-flag-backup-files ()
  "Flag all backup files (names ending with ~) for deletion."
  (interactive)
  (save-excursion
   (let ((buffer-read-only nil))
     (goto-char (point-min))
     (while (not (eobp))
       (and (not (looking-at "  d"))
	    (not (eolp))
	    (if (fboundp 'backup-file-name-p)
		(let ((fn (shape-get-filename t t)))
		  (if fn (backup-file-name-p fn)))
	      (end-of-line)
	      (forward-char -1)
	      (looking-at "~"))
	    (progn (beginning-of-line)
		   (delete-char 1)
		   (insert "D")))
       (forward-line 1)))))
