/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	afretr.c - retrieve interface 
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP
 *					   andy@db0tui62.BITNET)
 *
 *	$Header: afretr.c[1.20] Fri Jan 31 18:06:08 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *	af_bpfind -- find derived files by attributes
 *	af_find -- find source files by attributes
 *	af_getkey -- find single file by unique attributes
 *      af_predsucc -- get predecessor or successor of version
 *	af_initattrs -- initialize attribute buffer
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

extern int af_errno;

extern Af_hashent af_hashentry;

#ifdef MEMDEBUG
extern FILE *memprot;
#endif

/*================= collect_lists ====================================*/

LOCAL Af_revlist **collect_lists (pathsym, namesym, typesym, revlist, nlists)
     char       *pathsym, *namesym, *typesym;
     Af_revlist **revlist;
     int        *nlists; /* out */
{
  DIR           *dirp;
  struct direct *dirent;
  char          *afpath, *afname, *aftype, *dirpath;
  char          fullname[MAXPATHLEN], *realloc(), *af_ardir();
  bool          seconddir = FALSE;
  register int  listsiz, i;
  register Af_revlist *list;
  Af_revlist          *afTestList(), *afInitList();

  listsiz = AF_SEGLEN;
  *nlists = 0;

  /* open named directory  */
  if ((dirp = opendir (pathsym)) == (DIR *)0)
    FAIL ("collect_lists", "opendir", AF_ESYSERR, (Af_revlist **)0);

  dirpath = pathsym;

loop:
  /* lookup all files */
  while ((dirent = readdir (dirp)) != (struct direct *)0)
    {
      (void) sprintf (fullname, "%s/%s\0", dirpath, dirent->d_name);

      /* if entry belongs to binary pool */
      if (af_isbpfile (dirent->d_name))
	continue;

      /* regular files from the second directory will not be considered */
      if (seconddir && !(af_isarchive (dirent->d_name)))
	continue;
	
      afpath = af_entersym (af_afpath (fullname));
      if ((afname = af_entersym (af_afname (fullname))) == (char *)0)
	return ((Af_revlist **)0);
      aftype = af_entersym (af_aftype (fullname));

      /* if ((no name given || name equal) && (no type || type equal)) */
      if ( ((namesym[0] == '*') || (namesym == afname)) &&
	   ((typesym && (typesym[0] == '*')) || (typesym == aftype))    )
	{
	  if (*nlists == listsiz)
	    {
	      /* realloc memory for revlist */
	      if ((revlist = (Af_revlist **)realloc ((char *)revlist, (unsigned) (sizeof((Af_revlist *)0)*(listsiz + AF_SEGLEN)))) == (Af_revlist **)0)
		FAIL ("collect_lists","realloc", AF_ESYSERR, (Af_revlist **)0);
#ifdef MEMDEBUG
/*	      fprintf (memprot, "%x(revlist)-RE %d bytes\n", revlist,
		       sizeof((Af_revlist *)0)*(listsiz + AF_SEGLEN));
*/
#endif
	      listsiz += AF_SEGLEN;
	    }

	  /* see if list is already in "revlists" */
	  /* this loop may be quite time-consuming */
	  if ((list = afTestList (afpath, afname, aftype)) &&
	       (list->af_nrevs > 0))
	    {
	      i = 0;
	      while (i < *nlists)
		{
		  if ((revlist[i]->af_cattrs.af_syspath == afpath) &&
		      (revlist[i]->af_cattrs.af_globname == afname) &&
		      (revlist[i]->af_cattrs.af_globtype == aftype))
		    break;
		  i++;
		}
	      if (i == *nlists)
		{
		  if (afRefreshList (list, AF_ATTRS) == ERROR)
		    return ((Af_revlist **)0);
		  revlist[*nlists] = list;
		  (*nlists)++;
		}
	    }
	  else
	    {
	      if ((revlist[*nlists] =
		   afInitList (afpath, afname, aftype)) == (Af_revlist *)0)
		{
		  if (af_errno == AF_ENOATFSFILE)
		    continue;
		  else
		    return ((Af_revlist **)0);
		}
	      revlist[*nlists]->af_extent |= AF_LISTBUSY;
	      (*nlists)++;
	    }
	}
    }

  (void) closedir (dirp);

  if (!seconddir)
    {
      dirpath = af_ardir (pathsym);
      /* open AtFS subdirectory if it exists */
      if ((dirp = opendir (dirpath)) != (DIR *)0)
	{
	  seconddir = TRUE;
	  goto loop;
	}
    }
return (revlist);
}

/*====================================================================
 *    af_abufcmp -- compare attrbuf with version attributes
 *                  returns TRUE if attrs do not match
 *
 *====================================================================*/

LOCAL af_abufcmp (attrbuf, key)
     Af_attrs *attrbuf;
     Af_key   *key;
{
  register int match, j;

#ifdef STATISTICS
  _regsearchattrs (attrbuf);
#endif

  /* if (attribute is set && attributes does not match) -- return ERROR */
  
  /*** generation number ***/
  if ((attrbuf->af_gen != AF_NOVNUM) && (attrbuf->af_gen != VATTR(key).af_gen))
    return (ERROR);
  
  /*** revision number ***/
  if ((attrbuf->af_rev != AF_NOVNUM) && (attrbuf->af_rev != VATTR(key).af_rev))
    return (ERROR);

  /*** state ***/
  if ((attrbuf->af_state != AF_NOSTATE) &&
      (attrbuf->af_state != VATTR(key).af_state))
    return (ERROR);

  /*** owner ***/
  if ( (attrbuf->af_owner.af_username[0]) &&
       (strcmp (attrbuf->af_owner.af_username, CATTR(key).af_ownname)) )
    return (ERROR);
  if ( (attrbuf->af_owner.af_userhost[0]) &&
       (strcmp (attrbuf->af_owner.af_userhost, CATTR(key).af_ownhost)) )
    return (ERROR);
  if ( (attrbuf->af_owner.af_userdomain[0]) &&
       (strcmp (attrbuf->af_owner.af_userdomain, CATTR(key).af_owndomain)) )
    return (ERROR);
  
  /*** author ***/
  if ( (attrbuf->af_author.af_username[0]) &&
       (strcmp (attrbuf->af_author.af_username, VATTR(key).af_auname)) )
    return (ERROR);
  if ( (attrbuf->af_author.af_userhost[0]) &&
       (strcmp (attrbuf->af_author.af_userhost, VATTR(key).af_auhost)) )
    return (ERROR);
  if ( (attrbuf->af_author.af_userdomain[0]) &&
       (strcmp (attrbuf->af_author.af_userdomain, VATTR(key).af_audomain)) )
    return (ERROR);
  
  /*** size ***/
  if ((attrbuf->af_size != AF_NOSIZE) &&
      (attrbuf->af_size != VATTR(key).af_fsize) )
    return (ERROR);

  /*** mode ***/
  if ((attrbuf->af_mode != AF_NOMODE) &&
      ((attrbuf->af_mode & VATTR(key).af_mode) != attrbuf->af_mode))
    return (ERROR);

  /*** locker ***/
  if ( (attrbuf->af_locker.af_username[0] == '\01') &&
       (attrbuf->af_locker.af_username[1] == '$') &&
       VATTR(key).af_lckname)
    return (ERROR);

  if ( (attrbuf->af_locker.af_username[0]) &&
       (strcmp (attrbuf->af_locker.af_username,
		NOTNIL(VATTR(key).af_lckname))) )
    return (ERROR);
  if ( (attrbuf->af_locker.af_userhost[0]) &&
       (strcmp (attrbuf->af_locker.af_userhost,
		NOTNIL(VATTR(key).af_lckhost))) )
    return (ERROR);
  if ( (attrbuf->af_locker.af_userdomain[0]) &&
       (strcmp (attrbuf->af_locker.af_userdomain,
		NOTNIL(VATTR(key).af_lckdomain))) )
    return (ERROR);
  
  /*** date of last modification ***/
  if ((attrbuf->af_mtime != AF_NOTIME) &&
      (attrbuf->af_mtime != VATTR(key).af_mtime) )
    return (ERROR);

  /*** date of last access ***/
  if ((attrbuf->af_atime != AF_NOTIME) &&
      (attrbuf->af_atime != VATTR(key).af_atime) )
    return (ERROR);

  /*** date of last status change ***/
  if ((attrbuf->af_ctime != AF_NOTIME) &&
      (attrbuf->af_ctime != VATTR(key).af_ctime) )
    return (ERROR);

  /*** saving date ***/
  if ((attrbuf->af_stime != AF_NOTIME) &&
      (attrbuf->af_stime != VATTR(key).af_stime) )
    return (ERROR);
  
  /*** date of last lock change ***/
  if ((attrbuf->af_stime != AF_NOTIME) &&
      (attrbuf->af_stime != VATTR(key).af_stime) )
    return (ERROR);
  
  /*** user defined attributes ***/
  if (attrbuf->af_udattrs[0] != (char *)0)
    {
      /* if list of user defined attributes is not empty or there */
                                                /* are attributes */
      match = TRUE;
      if ((attrbuf->af_udattrs[0][0] != '\0') || (VATTR(key).af_udanum != 0))
	{
	  /* test all given entries */
	  j=0;
	  while ((attrbuf->af_udattrs[j] != (char *)0) 
		 && (match = !afMatchUda (key, attrbuf->af_udattrs[j])))
	    j++;
	} /* if */
      if (match == FALSE)
	return (ERROR);
    } /* if */
return (AF_OK);
}


/*====================================================================
 *    af_bpfind
 *
 *====================================================================*/

EXPORT af_bpfind (attrbuf, set)
     Af_attrs *attrbuf;
     Af_set   *set;     /* out */
{
  register Af_revlist *bplist;
  Af_revlist   *afInitBpList();
  register int i, maxindex;
  char         *pathsym, *malloc();
  Af_key       key;

  /* init set */
  set->af_nkeys = 0;
  set->af_setlen = 0;
  set->af_klist = (Af_key *)0;

  /* build pathname (default is current directory) */
  pathsym = af_entersym (af_uniqpath (attrbuf->af_syspath));

  if ((bplist = afInitBpList (pathsym)) == (Af_revlist *)0)
    if (af_errno == AF_ENOATFSDIR)
      return (0);
    else
      return (ERROR);

  /* alloacte memory for set */
  if ((set->af_klist = (Af_key *)malloc ((unsigned) (sizeof (Af_key) * bplist->af_nrevs))) == (Af_key *)0)
    FAIL ("bpfind", "malloc", AF_ESYSERR, ERROR);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(set-bpfind)-AL %d bytes\n", set->af_klist,
	       sizeof (Af_key) * bplist->af_nrevs);
#endif
  set->af_setlen = bplist->af_nrevs;

  /* add all desired bpfiles to set */
  maxindex = bplist->af_nrevs-1;
  for (i=0; i <= maxindex; i++)
    {
      /* skip invalid bpfiles */
      if (!(bplist->af_list[i].af_class & AF_VALID))
	{
	  if (++maxindex == bplist->af_listlen)
	    FAIL ("bpfind", "bad revision count", AF_EINCONSIST, ERROR);
	  continue;
	}
      if (((attrbuf->af_name[0] != '*') &&
	   (strcmp (attrbuf->af_name, bplist->af_list[i].af_name))) ||
	  ((attrbuf->af_type[0] != '*') &&
	   (strcmp (attrbuf->af_type, NOTNIL(bplist->af_list[i].af_type)))))
	continue;

      key.af_ldes = bplist;
      key.af_lpos = i;
      if (af_abufcmp (attrbuf, &key))
	continue;
      
      /* else add bpfile to set */
      set->af_klist[set->af_nkeys].af_ldes = bplist;
      set->af_klist[set->af_nkeys].af_lpos = i;
      set->af_nkeys++;
      bplist->af_access++;
#ifdef MEMDEBUG
  fprintf (memprot, "Access: %x,%d +1 (bpfind)\ttotal: %d\n", 
	   bplist, i, bplist->af_access);
#endif
      bplist->af_list[i].af_nrefs++;
    }
  /* if set is empty */
  if (set->af_nkeys == 0)
    {
      free ((char *)set->af_klist);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(set-bpfind)-FR\n", set->af_klist);
#endif
      set->af_setlen = 0;
    }

  return (set->af_nkeys);
}


/*====================================================================
 *    af_find
 *
 *====================================================================*/

EXPORT af_find (attrbuf, set)
     Af_attrs *attrbuf;
     Af_set   *set;     /* out */
{
  char          *pathsym, *namesym, *typesym, *malloc();
  register int  i, maxindex;
  int           nlists;
  Af_revlist    **revlist, *afInitList();
  Af_key        key;

  /* alloc memory for revlist */
  if ((revlist = (Af_revlist **)malloc ((unsigned) (sizeof((Af_revlist *)0)*AF_SEGLEN))) == (Af_revlist **)0)
    FAIL ("find", "malloc", AF_ESYSERR, ERROR);
#ifdef MEMDEBUG
/*  fprintf (memprot, "%x(revlist)-AL %d bytes\n", revlist,
	   sizeof((Af_revlist *)0)*AF_SEGLEN);
*/
#endif

  /* init set */
  set->af_nkeys = 0;
  set->af_setlen = 0;
  set->af_klist = (Af_key *)0;

  /* build pathname (default is current directory) */
  pathsym = af_entersym (af_uniqpath (attrbuf->af_syspath));
  namesym = af_entersym (attrbuf->af_name);
  typesym = af_entersym (attrbuf->af_type);

  if ((attrbuf->af_name[0] == '*') || (attrbuf->af_type[0] == '*'))
    {
      if ((revlist = collect_lists (pathsym, namesym, typesym, revlist,
				    &nlists)) == (Af_revlist **)0)
	return (ERROR);
    }
  else
    {
      if (!attrbuf->af_name[0])
	{
	  free ((char *)revlist);
#ifdef MEMDEBUG
/*	  fprintf (memprot, "%x(revlist)-FR\n", revlist);
*/
#endif
	  return (set->af_nkeys); /* no Af-files found */
	}
      if ((revlist[0] = afInitList (pathsym, namesym, typesym))
	  == (Af_revlist *)0)
	{
	  free ((char *)revlist);
#ifdef MEMDEBUG
/*	  fprintf (memprot, "%x(revlist)-FR\n", revlist);
*/
#endif
	  if (af_errno == AF_ENOATFSFILE)
	    nlists = 0;
	  else
	    return (ERROR);
	}
      else
	nlists = 1;
    }

  if (nlists == 0)
    return (set->af_nkeys); /* no Af-files found */

  /* alloacte memory for set */
  if ((set->af_klist = (Af_key *)malloc ((unsigned) (sizeof (Af_key) * AF_SEGLEN))) == (Af_key *)0)
    FAIL ("find", "malloc", AF_ESYSERR, ERROR);
#ifdef MEMDEBUG
  fprintf (memprot, "%x(set-find)-AL %d bytes\n", set->af_klist,
	   sizeof (Af_key) * AF_SEGLEN);
#endif
  set->af_setlen = AF_SEGLEN;

  /* lookup all revisions in all lists */
  /* this part is implemented quite dull up to now */
  /*     -- the number of "if"'s should be reduced */
  for (;nlists > 0; nlists--)
    {
      maxindex = revlist[nlists-1]->af_nrevs-1;
      for (i = 0; i <= maxindex; i++)
	{
	  /* skip holes in the list */
	  if (!(revlist[nlists-1]->af_list[i].af_class & AF_VALID))
	    {
	      if (++maxindex == revlist[nlists-1]->af_listlen)
		FAIL ("find", "bad revision count", AF_EINCONSIST, ERROR);
	      continue;
	    }

	  /* test all attributes -- returnes true if attrs do not match */
	  key.af_ldes = revlist[nlists-1];
	  key.af_lpos = i;
	  if (af_abufcmp (attrbuf, &key))
	    continue;

	  /********************************************/
	  /********** put AF-file into set ************/
	  /********************************************/

	  /* if set is full, enlarge it */
	  if (set->af_nkeys == set->af_setlen)
	    {
	      if ((set->af_klist = 
		   (Af_key *)realloc ((char *)set->af_klist, (unsigned) (sizeof(Af_key) * (set->af_setlen + AF_SEGLEN)))) == (Af_key *)0)
		FAIL ("find", "realloc", AF_ESYSERR, ERROR);
#ifdef MEMDEBUG
	      fprintf (memprot, "%x(set-find)-RE %d bytes\n", set->af_klist,
		       sizeof (Af_key) * (set->af_setlen + AF_SEGLEN));
#endif
	      set->af_setlen += AF_SEGLEN;
	    }

	  /* add revision to key-set */
	  set->af_klist[set->af_nkeys].af_ldes = revlist[nlists-1];
	  set->af_klist[set->af_nkeys].af_lpos = i;
	  set->af_nkeys++;
	  revlist[nlists-1]->af_access++;
	  revlist[nlists-1]->af_list[i].af_nrefs++;
#ifdef MEMDEBUG
  fprintf (memprot, "Access: %x,%d +1 (find)\ttotal: %d\n", 
	   revlist[nlists-1], i, revlist[nlists-1]->af_access);
#endif
	} /* for all revisions in archive */
      revlist[nlists-1]->af_extent &= ~AF_LISTBUSY;
    } /* for all archives */

  /* if set is empty */
  if (set->af_nkeys == 0)
    {
      free ((char *)set->af_klist);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(set-find)-FR\n", set->af_klist);
#endif
      set->af_setlen = 0;
    }

  free ((char *)revlist);
#ifdef MEMDEBUG
/*  fprintf (memprot, "%x(revlist)-FR\n", revlist);
*/
#endif
  return (set->af_nkeys);
}

/*====================================================================
 *    af_getkey
 *
 *====================================================================*/

EXPORT af_getkey (syspath, name, type, gen, rev, key)
     char   *syspath, *name, *type;
     int    gen, rev;
     Af_key *key;
{
  if (!name || (name[0] == '\0'))
    FAIL ("getkey", "no filename given", AF_EMISC, ERROR);

  return (afGetAso (syspath, name, type, gen, rev, key));
}

/*====================================================================
 *    af_predsucc
 *
 *====================================================================*/

EXPORT af_predsucc (inkey, mode, outkey)
     Af_key *inkey, *outkey;
     int    mode;
{
  if (afAccessAso (inkey, AF_ATTRS))
    SFAIL ("predsucc", "", AF_EINVKEY, ERROR);

  switch (mode)
    {
    case AF_LOGICAL_SUCC:
      SFAIL ("predsucc", "not yet implemented", AF_EMISC, ERROR);
    case AF_LOGICAL_PRED:
      SFAIL ("predsucc", "not yet implemented", AF_EMISC, ERROR);
    case AF_PHYSICAL_SUCC:
      if (VATTR(inkey).af_succgen == AF_NOVNUM)
	return (0);
      if (af_buildkey (inkey->af_ldes, VATTR(inkey).af_succgen, VATTR(inkey).af_succrev, outkey) == ERROR)
	SFAIL ("predsucc", "successor not found", AF_EINTERNAL, ERROR);
      return (1);
    case AF_PHYSICAL_PRED:
      if (VATTR(inkey).af_predgen == AF_NOVNUM)
	return (0);
      if (af_buildkey (inkey->af_ldes, VATTR(inkey).af_predgen, VATTR(inkey).af_predrev, outkey) == ERROR)
	SFAIL ("predsucc", "predecessor not found", AF_EINTERNAL, ERROR);
      return (1);
    }
  SFAIL ("predsucc", "invalid mode argument", AF_EMISC, ERROR);
}

/*====================================================================
 *    af_initattrs
 *
 *====================================================================*/

EXPORT af_initattrs (attrs)
     Af_attrs *attrs;
{
  attrs->af_host[0] = '\0';
  attrs->af_syspath[0] = '\0';
  attrs->af_name[0] = '*';
  attrs->af_name[1] = '\0';
  attrs->af_type[0] = '*';
  attrs->af_type[1] = '\0';
  attrs->af_gen = AF_NOVNUM;
  attrs->af_rev = AF_NOVNUM;
  attrs->af_state = AF_NOSTATE;
  attrs->af_owner.af_username[0] = '\0';
  attrs->af_owner.af_userhost[0] = '\0';
  attrs->af_owner.af_userdomain[0] = '\0';
  attrs->af_author.af_username[0] = '\0';
  attrs->af_author.af_userhost[0] = '\0';
  attrs->af_author.af_userdomain[0] = '\0';
  attrs->af_size = AF_NOSIZE;
  attrs->af_mode = AF_NOMODE;
  attrs->af_locker.af_username[0] = '\0';
  attrs->af_locker.af_userhost[0] = '\0';
  attrs->af_locker.af_userdomain[0] = '\0';
  attrs->af_mtime = AF_NOTIME;
  attrs->af_atime = AF_NOTIME;
  attrs->af_ctime = AF_NOTIME;
  attrs->af_stime = AF_NOTIME;
  attrs->af_ltime = AF_NOTIME;
  attrs->af_udattrs[0] = (char *)0;
}

