/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	aferror.c -- Error handling functions
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: aferror.c[1.3] Fri Jan 31 18:04:51 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *
 *      af_errno -- global variable holding the actual error code
 *	af_serr -- report error without writing error protocol
 *	af_err -- write error protocol
 *	af_wng -- print out warning message
 *      af_perror -- print AtFS-error
 */

#include <stdio.h>

#ifdef SYSLOG
#include <syslog.h>
#endif

#include "afsys.h"
#include "atfs.h"

char        *getlogin();
static char *logname;

/*=========================================================================
 *     af_serr -- report error without writing error protocol
 *
 *=========================================================================*/

extern int errno;

EXPORT int af_errno, af_nodiag = FALSE;

EXPORT void af_serr (routine, called, errcd)
     char  *routine;
     char  *called;
     int   errcd;
{
  af_nodiag = TRUE;
  af_err (routine, called, errcd);
  af_nodiag = FALSE;
}    

/*=========================================================================
 *     af_err -- write error protocol
 *
 *=========================================================================*/

static char diagstr[265]; /* for diagnistics of AF_EMISC */

static char *errors[] = 
  {
    "", "", "",
    "permission denied",				/*  3 */
    "archive file has changed since last read",		/*  4 */
    "archive file is locked for writing",		/*  5 */
    "no additional space in binary pool",		/*  6 */
    "specified revision must not be a busy version",	/*  7 */
    "specified revision is a derived object",		/*  8 */
    "invalid key",					/*  9 */
    "invalid set",					/* 10 */
    "invalid user",					/* 11 */
    "bad version number",				/* 12 */
    "invalid location of archive",			/* 13 */
    "application exceeds memory limit",                 /* 14 */
    "miscellaneous errors",				/* 15 */
    "invalid mode",					/* 16 */
    "AtFS subdirectory missing or not writable",	/* 17 */
    "key does not exist in set",			/* 18 */
    "invalid position in set",				/* 19 */
    "specified revision does not exist",		/* 20 */
    "specified object is no busy version",		/* 21 */
    "specified object is no derived object",		/* 22 */
    "version is not locked or locked by someone else",	/* 23 */
    "specified object is no regular file",		/* 24 */
    "specified object has no versions",			/* 25 */
    "user defined attribute does not exist",		/* 26 */
    "saved versions cannot be modified",		/* 27 */
    "invalid state transition",				/* 28 */
    "string too long",					/* 29 */
    "too many user defined attributes",			/* 30 */
    "wrong state",					/* 31 */
    "error during delta operation",			/* 32 */
    "Archive file inconsistent",			/* 33 */
    "internal error",					/* 34 */
    "no AtFS file",					/* 35 */
    "cannot build archive name (filename too long)",    /* 36 */
  };


EXPORT void af_err (routine, called, errcd)
     char  *routine;
     char  *called;
     int   errcd;
{
#ifndef SYSLOG
  FILE *errfile;
  char *af_asctime();
#endif

  if (af_nodiag)
    {
      af_errno = errcd;
      if (af_errno == AF_EMISC)
	(void) strcpy (diagstr, called);
      return; /* do nothing */
    }

#ifdef SYSLOG
  if (!openlog ("AtFS", LOG_PID, LOG_LOCAL1))
#else
  if ((errfile = fopen (AF_ERRLOG, "a")) == (FILE *)0)
#endif
    {
      fprintf (stderr, "AtFS: cannot open Error-logfile\n");
      return;
    }
  (void) chmod (AF_ERRLOG, 0666);

  if (!logname) logname = getlogin();

#ifdef SYSLOG
  switch (errcd)
    {
    case AF_ESYSERR: syslog (LOG_ERR, "%s called af_%s: %s error in %s (%m)", 
			     logname, routine, errors [abs(errcd)], called);
                     break;
    case AF_EINCONSIST:
    case AF_ENOATFSFILE:
    case AF_EINTERNAL: syslog (LOG_ERR, "%s called af_%s: %s (%s)", logname,
			       routine, errors [abs(errcd)], called);
                      break;
    case AF_EMISC: syslog (LOG_ERR, "%s called af_%s: %s ", logname,
			   routine, called);
                   (void) strcpy (diagstr, called);
                   break;
    default: syslog (LOG_ERR, "%s called af_%s: %s", logname, 
		     routine, errors [abs(errcd)]);
    }
#else
  fprintf (errfile, "%s pid[%d] %s",af_gethostname(),getpid (), af_asctime ());
  switch (errcd)
    {
    case AF_ESYSERR: fprintf (errfile, "\t%s called af_%s: %s error in %s\n", 
			     logname, routine, errors [abs(errcd)], called);
                     break;
    case AF_EINCONSIST:
    case AF_ENOATFSFILE:
    case AF_EINTERNAL: fprintf (errfile, "\t%s called af_%s: %s (%s)\n", logname, routine, errors [abs(errcd)], called);
                       break;
    case AF_EMISC: fprintf (errfile, "\t%s called af_%s: %s\n", logname, routine, called);
                   (void) strcpy (diagstr, called);
                   break;
    default: fprintf (errfile, "\t%s called af_%s: %s\n", logname, routine, errors [abs(errcd)]);
    }
#endif

#ifdef SYSLOG
  closelog ();
#else
  (void) fclose (errfile);
#endif

  af_errno = errcd;
  return;
}

/*=========================================================================
 *     af_wng -- write warning to error protocol
 *
 *=========================================================================*/

EXPORT void af_wng (routine, comment)
     char  *routine, *comment;
{
#ifndef SYSLOG
  FILE *errfile;
  char *af_asctime();
#endif

#ifdef SYSLOG
  if (!openlog ("AtFS", LOG_PID, LOG_LOCAL1))
#else
  if ((errfile = fopen (AF_ERRLOG, "a")) == (FILE *)0)
#endif
    {
      fprintf (stderr, "AtFS: cannot open Error-logfile\n");
      return;
    }
  (void) chmod (AF_ERRLOG, 0666);

  if (!logname) logname = getlogin();

#ifdef SYSLOG
  syslog (LOG_WARNING, "%s called af_%s: %s", logname, routine, comment);
#else
  fprintf (errfile, "%s pid[%d] %s", af_gethostname(), getpid (), af_asctime ());
  fprintf (errfile, "\t%s called af_%s: %s\n", logname, routine, comment);
#endif

#ifdef SYSLOG
  closelog ();
#else
  (void) fclose (errfile);
#endif
  return;
}


/*=========================================================================
 *     af_perror -- print AtFS-error message
 *
 *=========================================================================*/

EXPORT void af_perror (string)
     char  *string;
{
  switch (af_errno)
    {
    case AF_ESYSERR: perror (string);
                     break;
    case AF_EMISC: fprintf (stderr, "%s: %s\n", string, diagstr);
                  break;
    default: fprintf (stderr, "%s: %s\n", string, errors [abs(af_errno)]);
    }
}
