/*LINTLIBRARY*/
/*$__copyright$ */
/*
 *	Shape/AtFS
 *
 *	afattrs.c - reading and writing attributes
 *
 *	by Andreas Lampen, TU-Berlin (Andreas.Lampen@cs.tu-berlin.de)
 *
 *	$Header: afattrs.c[1.2] Fri Apr 24 17:56:19 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *	af_retattr -- return string attribute
 *	af_retnumattr -- return numeric attribute
 *	af_retuserattr -- return user attribute
 *	af_rettimeattr -- return time attribute
 *	af_setattr -- set user defined attribute
 *      af_allattrs -- return all attributes
 */

#include "atfs.h"

char *malloc();

/*===========================================================
 *  af_retattr -- return attribute in string representation
 *===========================================================*/

EXPORT char *af_retattr (aso, attrName)
     Af_key *aso;
     char   *attrName;
{
  static char attrBuf[MAXPATHLEN], *udattr, *entry, *valptr;
;

  if (afAccessAso (aso, AF_ATTRS))
    SFAIL ("retattr", "", AF_EINVKEY, (char *)0);

  if (!attrName)
    SFAIL ("retattr", "no attribute name given", AF_EMISC, NULL);

  if (!strncmp (attrName, "af_", 3))
    {
      /* presumably a standard attribute name */
      char *aNamPtr = &attrName[3];
      attrBuf[0] = '\0';

      if (!strcmp (&AF_ATTHOST[3], aNamPtr)) {
	if (CATTR(aso).af_host)
	  (void) strcpy (attrBuf, CATTR(aso).af_host);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTSPATH[3], aNamPtr)) {
	if (CATTR(aso).af_syspath)
	  (void) strcpy (attrBuf, CATTR(aso).af_syspath);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTUNIXNAME[3], aNamPtr)) {
	(void) strcpy (attrBuf, VATTR(aso).af_name);
	if (VATTR(aso).af_type) {
	  (void) strcat (attrBuf, ".");
	  (void) strcat (attrBuf, VATTR(aso).af_type);
	}
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTNAME[3], aNamPtr)) {
	if (VATTR(aso).af_name)
	  (void) strcpy (attrBuf, VATTR(aso).af_name);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTTYPE[3], aNamPtr)) {
	if (VATTR(aso).af_type)
	  (void) strcpy (attrBuf, VATTR(aso).af_type);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTGEN[3], aNamPtr)) {
	if (VATTR(aso).af_gen == AF_BUSYVERS)
	  strcpy (attrBuf, "busy");
	else
	  sprintf (attrBuf, "%d", VATTR(aso).af_gen);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTREV[3], aNamPtr)) {
	if (VATTR(aso).af_rev == AF_BUSYVERS)
	  strcpy (attrBuf, "busy");
	else
	  sprintf (attrBuf, "%d", VATTR(aso).af_rev);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTVERSION[3], aNamPtr)) {
	if (VATTR(aso).af_gen == AF_BUSYVERS)
	  strcpy (attrBuf, "busy");
	else
	  sprintf (attrBuf, "%d.%d", VATTR(aso).af_gen, VATTR(aso).af_rev);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTHUMAN[3], aNamPtr)) {
	if (VATTR(aso).af_type)
	  sprintf (attrBuf, "%s.%s[%d.%d]", VATTR(aso).af_name,
		   VATTR(aso).af_type, VATTR(aso).af_gen, VATTR(aso).af_rev);
	else
	  sprintf (attrBuf, "%s[%d.%d]", VATTR(aso).af_name,
		   VATTR(aso).af_gen, VATTR(aso).af_rev);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTSTATE[3], aNamPtr)) {
	switch (VATTR(aso).af_state)
	  {
	  case AF_BUSY:
	    strcpy (attrBuf, "busy"); break;
	  case AF_SAVED:
	    strcpy (attrBuf, "saved"); break;
	  case AF_PROPOSED:
	    strcpy (attrBuf, "proposed"); break;
	  case AF_PUBLISHED:
	    strcpy (attrBuf, "published"); break;
	  case AF_ACCESSED:
	    strcpy (attrBuf, "accessed"); break;
	  case AF_FROZEN:
	    strcpy (attrBuf, "frozen"); break;
	  }
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTOWNER[3], aNamPtr)) {
	sprintf (attrBuf, "%s@%s", CATTR(aso).af_ownname,
		                   CATTR(aso).af_owndomain);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTAUTHOR[3], aNamPtr)) {
	sprintf (attrBuf, "%s@%s", VATTR(aso).af_auname, 
		                   VATTR(aso).af_audomain);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTDSIZE[3], aNamPtr)) {
	sprintf (attrBuf, "%u", VATTR(aso).af_dsize);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTSIZE[3], aNamPtr)) {
	sprintf (attrBuf, "%u", VATTR(aso).af_fsize);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTMODE[3], aNamPtr)) {
	sprintf (attrBuf, "%o", VATTR(aso).af_mode);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTLOCKER[3], aNamPtr)) {
	if (VATTR(aso).af_lckname)
	  sprintf (attrBuf, "%s@%s", VATTR(aso).af_lckname, 
		                     VATTR(aso).af_lckdomain);
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTMTIME[3], aNamPtr)) {
	strcpy (attrBuf, asctime (localtime (&VATTR(aso).af_mtime)));
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTATIME[3], aNamPtr)) {
	strcpy (attrBuf, asctime (localtime (&VATTR(aso).af_atime)));
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTCTIME[3], aNamPtr)) {
	strcpy (attrBuf, asctime (localtime (&VATTR(aso).af_ctime)));
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTSTIME[3], aNamPtr)) {
	if (VATTR(aso).af_stime != AF_NOTIME)
	  strcpy (attrBuf, asctime (localtime (&VATTR(aso).af_stime)));
	return (attrBuf);
      }

      if (!strcmp (&AF_ATTLTIME[3], aNamPtr)) {
	if (VATTR(aso).af_ltime != AF_NOTIME)
	  strcpy (attrBuf, asctime (localtime (&VATTR(aso).af_ltime)));
	return (attrBuf);
      }
    }

  /* assume user defined attribute */

  if ((entry = afLookupUda (aso, attrName)) == (char *)0)
    return ((char *)0);

  if ((valptr = index (entry, AF_UDANAMDEL)) != (char *)0)
    {
      if ((udattr =
	   malloc ((unsigned) strlen(valptr) + sizeof(char))) == (char *)0)
	FAIL ("rudattr", "malloc", AF_ESYSERR, (char *)0);
      /* replace delimiters by '\n' */
      (void) strcpy (udattr, valptr+1);
      valptr = udattr;
      while ((valptr = index (valptr, AF_UDAVALDEL)) != (char *)0)
	*valptr = '\n';
      return (udattr);
    }
  return (attrBuf);
}

/*=============================================
 *  af_retnumattr -- return numeric attribute
 *=============================================*/

EXPORT int af_retnumattr (aso, attrName)
     Af_key *aso;
     char   *attrName;
{
  if (afAccessAso (aso, AF_ATTRS))
    SFAIL ("retnumattr", "", AF_EINVKEY, ERROR);

  if (!attrName)
    SFAIL ("retnumattr", "no attribute name given", AF_EMISC, NULL);

  if (!strcmp (AF_ATTGEN, attrName))
    return (VATTR(aso).af_gen);

  if (!strcmp (AF_ATTREV, attrName))
    return (VATTR(aso).af_rev);

  if (!strcmp (AF_ATTSTATE, attrName))
    return (VATTR(aso).af_state);

  if (!strcmp (AF_ATTDSIZE, attrName)) {
    if (VATTR(aso).af_repr != AF_DELTA)
      return (ERROR);
    return (VATTR(aso).af_dsize);
  }

  if (!strcmp (AF_ATTSIZE, attrName))
    return (VATTR(aso).af_fsize);

  if (!strcmp (AF_ATTMODE, attrName))
    return (VATTR(aso).af_mode);

  return (ERROR);
}

/*===========================================
 *  af_retuserattr -- return user attribute
 *===========================================*/

EXPORT Af_user *af_retuserattr (aso, attrName)
     Af_key *aso;
     char   *attrName;
{
  static Af_user userBuf;

  if (afAccessAso (aso, AF_ATTRS))
    SFAIL ("retuserattr", "", AF_EINVKEY, NULL);

  if (!attrName)
    SFAIL ("retuserattr", "empty attribute name", AF_EMISC, NULL);

  if (!strcmp (AF_ATTOWNER, attrName)) {
    (void) strcpy (userBuf.af_username, CATTR(aso).af_ownname);
    (void) strcpy (userBuf.af_userhost, CATTR(aso).af_ownhost);
    (void) strcpy (userBuf.af_userdomain, CATTR(aso).af_owndomain);
    return (&userBuf);
  }

  if (!strcmp (AF_ATTAUTHOR, attrName)) {
    (void) strcpy (userBuf.af_username, VATTR(aso).af_auname);
    (void) strcpy (userBuf.af_userhost, VATTR(aso).af_auhost);
    (void) strcpy (userBuf.af_userdomain, VATTR(aso).af_audomain);
    return (&userBuf);
  }

  if (!strcmp (AF_ATTLOCKER, attrName)) {
    if (VATTR(aso).af_lckname != (char *)0) {
      (void) strcpy (userBuf.af_username, VATTR(aso).af_lckname);
      (void) strcpy (userBuf.af_userhost, VATTR(aso).af_lckhost);
      (void) strcpy (userBuf.af_userdomain, VATTR(aso).af_lckdomain);
    }
    else {
      userBuf.af_username[0] = '\0';
      userBuf.af_userhost[0] = '\0';
      userBuf.af_userdomain[0] = '\0';
    }
    return (&userBuf);
  }
  return (NULL);
}

/*===========================================
 *  af_rettimeattr -- return time attribute
 *===========================================*/

EXPORT time_t af_rettimeattr (aso, attrName)
     Af_key *aso;
     char   *attrName;
{
  if (afAccessAso (aso, AF_ATTRS))
    SFAIL ("rettimeattr", "", AF_EINVKEY, AF_NOTIME);

  if (!attrName)
    SFAIL ("rettimeattr", "empty attribute name", AF_EMISC, NULL);

  if (!strcmp (AF_ATTMTIME, attrName))
    return (VATTR(aso).af_mtime);

  if (!strcmp (AF_ATTATIME, attrName))
    return (VATTR(aso).af_atime);

  if (!strcmp (AF_ATTCTIME, attrName))
    return (VATTR(aso).af_ctime);

  if (!strcmp (AF_ATTSTIME, attrName))
    return (VATTR(aso).af_stime);

  if (!strcmp (AF_ATTLTIME, attrName))
    return (VATTR(aso).af_ltime);

  return (AF_NOTIME);
}

/*===============================================
 *  af_setattr -- return user defined attribute
 *===============================================*/

EXPORT int af_setattr (key, mode, attr)
     Af_key *key;
     int    mode;
     char   *attr;
{
  register char *udaptr, *tmpuda, *valptr;
  register int  tail;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("setattr", "", AF_EINVKEY, ERROR);

  if (!attr)
    SFAIL ("setattr", "empty attribute name", AF_EMISC, NULL);

  if ((VATTR(key).af_state == AF_FROZEN) && (mode == AF_REMOVE))
    SFAIL ("setattr", "", AF_EWRONGSTATE, ERROR);

  /* look for delimiter character in attribute string */
  if (!attr || (index (attr, AF_UDAVALDEL) != (char *)0))
    SFAIL ("setattr", "illegal format of attribute string", AF_EMISC, ERROR);

  if (af_checkperm (key, AF_WORLD) == ERROR)
    return (ERROR);

  /* search entry */
  udaptr = afLookupUda (key, attr); 

  switch (mode)
    {
    case AF_ADD: if (udaptr != (char *)0)
                   {
		     /* build new entry and replace old one */
		     if (!(valptr = index (attr, AF_UDANAMDEL)))
		       {
			 /* no new value -- nothing to do */
			 break;
		       }
		     valptr += sizeof (char);
		     if ((tmpuda = malloc ((unsigned) ((strlen (udaptr) + strlen (valptr) +2) * sizeof (char)))) == (char *)0)
		       FAIL ("setattr", "malloc", AF_ESYSERR, ERROR);

		     (void) strcpy (tmpuda, udaptr);
		     tail = strlen (tmpuda);
		     tmpuda[tail] = AF_UDAVALDEL;
		     tmpuda[tail+1] = '\0';
		     (void) strcat (tmpuda, valptr);
		     (void) afReplUda (key, tmpuda);
		     free (tmpuda);
 		   }    
                 else
		   {
		     /* add new entry */
		     if (VATTR(key).af_udanum == AF_MAXUDAS)
		       SFAIL ("setattr", "", AF_EUDASNUM, ERROR);
		     (void) afEnterUda (key, attr);
		   }
                 break;

    case AF_REMOVE: if (udaptr == (char *)0)
                      SFAIL ("setattr", "", AF_ENOUDA, ERROR);
		    (void) afDelUda (key, udaptr);
                    break;

    case AF_REPLACE: if (udaptr == (char *)0)
                       SFAIL ("setattr", "", AF_ENOUDA, ERROR);
                     (void) afReplUda (key, attr);
                     break;
    default: SFAIL ("setattr", "", AF_EMODE, ERROR);
    }

  /* save changes */
  if (afUpdateAso (key, AF_CHANGE) == ERROR)
    return (ERROR);
  return (AF_OK);
}

/*=========================================
 *  af_allattrs -- return all attributes
 *=========================================*/

EXPORT int af_allattrs (key, attrbuf)
     Af_key   *key;
     Af_attrs *attrbuf;
{
  register int  i, size;
  register char *udattrs, *valptr;
  char          *udalist[AF_MAXUDAS+1];

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("allattrs", "", AF_EINVKEY, ERROR);

  (void) strcpy (attrbuf->af_host, CATTR(key).af_host);
  (void) strcpy (attrbuf->af_name, VATTR(key).af_name);
  (void) strcpy (attrbuf->af_type, NOTNIL(VATTR(key).af_type));
  (void) strcpy (attrbuf->af_syspath, CATTR(key).af_syspath);

  attrbuf->af_gen = VATTR(key).af_gen;
  attrbuf->af_rev = VATTR(key).af_rev;
  attrbuf->af_state = (int) VATTR(key).af_state;
  (void) strcpy (attrbuf->af_owner.af_username, CATTR(key).af_ownname);
  (void) strcpy (attrbuf->af_owner.af_userhost, CATTR(key).af_ownhost);
  (void) strcpy (attrbuf->af_owner.af_userdomain, CATTR(key).af_owndomain);
  (void) strcpy (attrbuf->af_author.af_username, VATTR(key).af_auname);
  (void) strcpy (attrbuf->af_author.af_userhost, VATTR(key).af_auhost);
  (void) strcpy (attrbuf->af_author.af_userdomain, VATTR(key).af_audomain);
  attrbuf->af_size = VATTR(key).af_fsize;
  attrbuf->af_mode = VATTR(key).af_mode;
  (void) strcpy (attrbuf->af_locker.af_username, NOTNIL(VATTR(key).af_lckname));
  (void) strcpy (attrbuf->af_locker.af_userhost, NOTNIL(VATTR(key).af_lckhost));
  (void) strcpy (attrbuf->af_locker.af_userdomain, NOTNIL(VATTR(key).af_lckdomain));
  attrbuf->af_mtime = VATTR(key).af_mtime;
  attrbuf->af_atime = VATTR(key).af_atime;
  attrbuf->af_ctime = VATTR(key).af_ctime;
  attrbuf->af_stime = VATTR(key).af_stime;
  attrbuf->af_ltime = VATTR(key).af_ltime;

  /* copy user defined attributes */
  if (VATTR(key).af_udanum > 0)
    {
      (void) afListUdas (key, udalist);
      size = 0;
      i=0;
      while (udalist[i])
	{
	  size += strlen (udalist[i]) + sizeof (char);
	  i++;
	}
      if ((udattrs = malloc ((unsigned) size)) == (char *)0)
	FAIL ("allattrs", "malloc", AF_ESYSERR, ERROR);
      i=0;
      while (udalist[i] != (char *)0)
	{
	  (void) strcpy (udattrs, udalist[i]);
	  /* replace delimiters by '\n' */
	  if (valptr = index (udattrs, AF_UDANAMDEL))
	    {
	      while ((valptr = index (valptr, AF_UDAVALDEL)) != (char *)0)
		valptr[0] = '\n';
	    }
	  attrbuf->af_udattrs[i] = udattrs;
	  udattrs = udattrs + (strlen(udalist[i])+1);
	  i++;
	}
      attrbuf->af_udattrs[i] = (char *)0; /* finish list */
    }
  else
    attrbuf->af_udattrs[0] = (char *)0;

  return (AF_OK);
}
