/**************************************************************
 * Send any comments or questions to: OTSO-Bug@tel.vtt.fi
 *
 * Name: /home/users/otso/official/otso/include/SCCS/s.isoaddrs.hxx
 * Vers: 5.4    Time: 92/08/03, 15:42:08
 **************************************************************/

/***************************************************************
* 
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose and without fee is hereby granted, provided
* that this notice and the reference to this notice appearing in each software
* module be retained unaltered, and that the name of any contributors shall not
* be used in advertising or publicity pertaining to distribution of the software
* without specific written prior permission.  No contributor makes any
* representations about the suitability of this software for any purpose.
* It is provided "as is" without any express or limited warranty.
*
*			NO WARRANTY
*
* ALL CONTRIBUTORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
* INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS.  IN NO
* EVENT SHALL ANY CONTRIBUTOR BE LIABLE FOR ANY SPECIAL, PUNITIVE, INDIRECT OR
* CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
* DATA, OR PROFITS, WHETHER IN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
* TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH, THE USE OR PERFORMANCE
* OF THIS SOFTWARE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THIS
* SOFTWARE IS WITH YOU.  SHOULD THIS SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE
* COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
*
* 
* 
***************************************************************/


#ifndef ISOADDRS_HXX
#define ISOADDRS_HXX	1

/******************************************
* FILE
*	isoaddrs.hxx
*******************************************/

#ifndef _TYPES_
# include <sys/types.h>
#endif

class NETaddr;		/* Physical Network Address */
class NSAPaddr;		/* NSAP = nsel + 4 possible NETaddr's */
class TSAPaddr;		/* TSAP = tsel + NSAP */
class SSAPaddr;		/* SSAP = ssel + TSAP */
class PSAPaddr;		/* PSAP = psel + SSAP */


/*  */

/*******************************************************************
The header file isoaddrs.hxx contains several classes of "addresses".
They are:
.nf
   NETaddr NSAPaddr TSAPaddr SSAPaddr and PSAPaddr (OSIaddr = PSAPaddr)
.fi
They represent addresses / service access points as defined in various
OSI documents (e.g ISO 8348).  They are used for in the network,
transport, session, and presentation layers.

----------------------

A NETaddr is the base class that can be used to represent an OSI
"physical" network address.   

See ISO Standard 8348 (Network Service Definitions) about the usage/values 
of afi, idi, and dsp.  The afi, idi, dsp defined below are stored as ascii
strings and not byte encodings as defined in ISO 8348.  It is a local protocol
matter to convert the ascii string into the correct byte encoding.

(i.e. afi=08 is stored as two ascii bytes: 0x3038
            and not as a two byte integer: 0x0012)
*********************************************************************/

class NETaddr : public Object {

public:
    Bytes	afi;		// OSI Authority and Format Identifier
    Bytes	idi;		// OSI Initial Domain Identifier
    Bytes	dsp;		// OSI Domain Specific Part

    NETaddr&	copy();			// make a copy of the Object

    virtual void ask (Istream&); // cin >> PSAPaddr
    virtual void print (Ostream& os) ;

    virtual void to(ODump&);		// Channel encoder.
    virtual void from(IDump&);		// Channel decoder.
    virtual String className () const { return "NETaddr"; }

    NETaddr& operator= (NETaddr&);	// assignment
    boolean operator== (NETaddr&);	// equivalence
    boolean operator== (NSAPaddr&);	// Equivalent if any NETaddr in
					// NSAPaddr matches.
    boolean operator!= (NETaddr& x) 
	{return !(*this == x);}	// non-equivalence
    boolean operator!= (NSAPaddr& x) 
	{return !(*this == x);}	// non-equivalence

    NETaddr () {}
    NETaddr::NETaddr (char *s1, char* s2, char* s3) 
            : afi(s1), idi(s2), dsp(s3) { }
    NETaddr (NETaddr& nar) { *this = nar; }
};


/*  */


/*******************************************************************
Network Sevice Access Point (NSAP) :

An NSAPaddr is a base class that can be used to represent an OSI
Network Service access point (NSAP).

An "NSAPaddr" consists of two parts:
.nf
	- na_selector : arbitrary format string that selects the layer 
		above the network layer.  However, it is quite common
		for other OSI vendors to use a 2 byte GOSIP identifier.
	- na_addrs array : A choice of up to 4 possible physical network
		addresses (see NETaddr above).
.fi
********************************************************************/

#define NTADDR	4	/* Number of possible network addresses */

class NSAPaddr : public Object {

public:

    Bytes	na_selector; 		// network selector string

    NETaddr*	na_addrs [NTADDR];	// Array of physical netwk addresses
    int		na_naddr;		// Number of network addresses

    NSAPaddr&	copy();			// make a copy of the Object

    virtual void reset();		// make object empty

    virtual void ask (Istream&);	//  cin >> NSAPaddr
    virtual void print (Ostream&);	// cout << NSAPaddr

    virtual void to(ODump&);		// Channel encoder.
    virtual void from(IDump&);		// Channel decoder.
    virtual String className () const { return "NSAPaddr"; }

    NSAPaddr& operator= (NSAPaddr&);	// assignment
    boolean operator== (NSAPaddr&);	// Equivalent if any NETaddr
					// in NSAPaddr matches any NETaddr
					// in other NSAPaddr and 
					// na_selectors match.
    boolean operator== (NETaddr& x)	
	{ return (x == *this); }	// Equivalent if any NETaddr
					// in NSAPaddr matches.
    boolean operator!= (NETaddr& x)	
	{ return (x != *this); }	// non-equivalence
    boolean operator!= (NSAPaddr& x)	
	{ return (!(*this == x)); }	// non-equivalence

    void NSAPinit ();
    NSAPaddr ();
    NSAPaddr (char*);		// Construct an NSAPaddr from a ascii string of
				// form : "nsel/NS+afi+idi+dsp|NS+afi+idi+dsp"
    NSAPaddr (NSAPaddr&);
    NSAPaddr (NETaddr&);
};

/*  */

/*******************************************************************
Transport Service Access Point:

"TSAPaddr" is a derived class that can be used to represent an OSI
Transport Service access point (TSAP).

"TSAPaddr" consists of a NSAPaddr (via inheritance) and a "ta_selector" 
to select the layer on top of the TSAP (i.e. session layer).
The ta_selector can be a 2-byte GOSIP selector (e.g. '0001'H is the ISO
session layer), or a longer implementation dependent string.
********************************************************************/
class TSAPaddr : public NSAPaddr {

public:

    Bytes	ta_selector;	// transport selector string.

    TSAPaddr&	copy();			// make a copy of the Object

    virtual void reset();		// make object empty
    virtual void ask (Istream&);	// cin >> PSAPaddr
    virtual void print (Ostream&);	// cout << TSAPaddr
    virtual String className () const { return "TSAPaddr"; }

    TSAPaddr& operator= (TSAPaddr&);	// assignment
    TSAPaddr& operator= (NSAPaddr&);	// assignment
    TSAPaddr& operator= (NETaddr&);	// assignment
    boolean  operator== (TSAPaddr&);	// Equivalent if ta_selector's and
					// NSAPaddr's match.
    boolean operator== (NSAPaddr& x)
	 { return (x == (NSAPaddr&)*this); } // Equivalent if NSAPaddr's match.
    boolean operator== (NETaddr&  x) 
	{ return (x == (NSAPaddr&)*this); }  // Equivalent if any NETaddr in
					     // matches.
    boolean operator!= (TSAPaddr& x) 
	{ return !(*this == x); }	// non-equivalence
    boolean operator!= (NSAPaddr& x)
	{ return !(*this == x); }	// non-equivalence
    boolean operator!= (NETaddr&  x)
	{ return !(*this == x); }	// non-equivalence


    TSAPaddr (){}
    TSAPaddr (char*);		// Construct an NSAPaddr from a ascii string of
				// form : Form: "tsel/nsel/NS+afi+idi+dsp"
    TSAPaddr (NSAPaddr&);
    TSAPaddr (TSAPaddr& tmp) { *this = tmp; }
};

/*  */

/*******************************************************************
Session Service Access Point:

"SSAPaddr" is a derived class that can be used to represent an OSI
Session Service access point (SSAP).

"SSAPaddr" consists of a TSAPaddr (via inheritance) and a "sa_selector" 
to select the layer on top of the SSAP (i.e. session layer).
The sa_selector can be a 2-byte GOSIP selector (e.g. '0001'H is the ISO
presentaion layer), or a longer implementation dependent string.
********************************************************************/
class SSAPaddr : public TSAPaddr {

public:

    Bytes	sa_selector;	// session selector string

    SSAPaddr&	copy();			// make a copy of the Object

    virtual void reset();		// make object empty

    virtual void ask (Istream&);	// cin >> PSAPaddr
    virtual void print (Ostream&);	// cout << SSAPaddr

    virtual String className() const { return "SSAPaddr"; }

    SSAPaddr& operator= (SSAPaddr&);	// assignment
    SSAPaddr& operator= (TSAPaddr&);	// assignment
    SSAPaddr& operator= (NSAPaddr&);	// assignment
    boolean  operator== (SSAPaddr&);	// Equivalent if sa_selector, and
					// TSAPaddr's match.
    boolean operator== (TSAPaddr& x) 
	{ return (x == (TSAPaddr&)*this); }	// equivalence
    boolean operator== (NSAPaddr& x) 
	{ return (x == (NSAPaddr&)*this); }	// equivalence
    boolean operator== (NETaddr&  x) 
	{ return (x == (NSAPaddr&)*this); }	// equivalence
    boolean operator!= (SSAPaddr& x) 
	{ return !(*this == x); }	// non-equivalence
    boolean operator!= (TSAPaddr& x) 
	{ return !(*this == x); }	// non-equivalence
    boolean operator!= (NSAPaddr& x) 
	{ return !(*this == x); }	// non-equivalence
    boolean operator!= (NETaddr&  x) 
	{ return !(*this == x); }	// non-equivalence

    SSAPaddr () {}
    SSAPaddr (char*);		// Construct an SSAPaddr from a ascii string of
				// form : "ssel/tsel/nsel/NS+afi+idi+dsp"
    SSAPaddr (NSAPaddr& tmp) { *this = tmp; }	// !WARNING move!
    SSAPaddr (TSAPaddr& tmp) { *this = tmp; }	// !WARNING move!
    SSAPaddr (SSAPaddr& tmp) { *this = tmp; } 	// !WARNING move ! 
};

/*  */

/*******************************************************************
Presentation Layer Service Access Point:

"PSAPaddr" is a derived class that can be used to represent an OSI
Presentation Service access point (PSAP).

"PSAPaddr" consists of a SSAPaddr (via inheritance) and a "pa_selector" 
to select the layer on top of the PSAP (i.e. application layer (FTAM, DTAM,
X500, ...).

The pa_selector can be a 2-byte GOSIP selector (e.g. '0101'H is the GOSIP
selector for an ISO X.500 application layer), or a longer implementation 
dependent string (e.g. "pres").

PSAPaddr, SSAPaddr, TSAPaddr, NSAPaddr, and NETaddr can all constructed 
from simple ASCII strings according to the following format :

.SH STRING ENCODING OF PRESENTATION ADDRESSES
.nf

The BNF grammar for the strings looks like:

<presentation_address>	::= [<psel> "/" ] session_address
<session_address>	::= [<ssel> "/" ] transport_address
<transport_address>	::= [<tsel> "/" ] network_address
<network_address>	::= [<nsel> "/" ] <phys_net_address_list>
<phys_net_address_list> ::=
	<phys_net_address> "|" <phys_net_address_list>
	| <phys_net_address>

<psel> ::= <selector>
<ssel> ::= <selector>
<tsel> ::= <selector>
<nsel> ::= <selector>
     
<selector> ::= <string> | <hexstring> 

<phys_net_address> ::=  "NS" + <afi> "+" <idi> [ "+" <dsp> ]
     
<afi> ::= <digitstring>
<idi> ::= <digitstring>
<dsp> ::= <digitstring>			-- OSI dsp
          | <portnum> "@" <quadstring>	-- fake TCPIP dsp (e.g 102@tel.vtt.fi)
     
quadstring ::=
	digitstring "." digitstring "." digitstring [ "." digitstring ]
	| otherstring "." otherstring "." otherstring [ "." otherstring ]
	-- e.g. 130.188.12.1  OR  tel3.tel.vtt.fi

<portnum> ::= <digitstring>	-- e.g. 102

<digit>    ::= [09]
<other>    ::= [09azAZ+.]
<hexdigit> ::= [09af]
     
<digitstring> ::= <digit> <digitstring> | <digit>
<hexstring>   ::= <hexdigit> <hexstring> | <hexdigit>

     
.SH EXAMPLES
.nf
PSAPaddr p1("101H/1H/01H/001H/NS+0+0+0001");	-- psel/ssel/tsel/nsel/naddr
SSAPaddr s1("01H/01H/01H/NS+0+0+0001");		--      ssel/tsel/nsel/naddr
TSAPaddr t1("01H/01H/NS+0+0+0001");		--           tsel/nsel/naddr
NSAPaddr n1("01H/NS+0+0+0001");			--                nsel/naddr
NETaddr n1("NS+0+0+0001|NS+54+00728722+102@tel3.tel.vtt.fi");	-- phys network
.fi
********************************************************************/
class PSAPaddr : public SSAPaddr
{

public:
    Bytes	pa_selector;	// presentation selector string

    PSAPaddr&	copy();			// make a copy of the Object

    virtual void reset();		// make object empty

    virtual void ask (Istream&);	// cin >> PSAPaddr
    virtual void print (Ostream&);	// cout << PSAPaddr

    virtual String className() const { return "PSAPaddr"; }

    PSAPaddr& operator=(PSAPaddr&);	// assignment
    PSAPaddr& operator=(SSAPaddr&);	// assignment
    PSAPaddr& operator=(TSAPaddr&);	// assignment
    PSAPaddr& operator=(NSAPaddr&);	// assignment
    boolean  operator==(PSAPaddr&);	// Equivalent if pa_selector
					// and SSAPaddr's match.
    boolean operator==(SSAPaddr& x)
	{ return (x == (SSAPaddr&)*this); }	// equivalence
    boolean operator==(TSAPaddr& x)
	{ return (x == (TSAPaddr&)*this); }	// equivalence
    boolean operator==(NSAPaddr& x)
	{ return (x == (NSAPaddr&)*this); }	// equivalence
    boolean operator==(NETaddr&  x)
	{ return (x == (NSAPaddr&)*this); }	// equivalence
    boolean operator!=(PSAPaddr& x)
	{ return !(*this == x); }	// non-equivalence
    boolean operator!=(SSAPaddr& x)
	{ return !(*this == x); }	// non-equivalence
    boolean operator!=(TSAPaddr& x)
	{ return !(*this == x); }	// non-equivalence
    boolean operator!=(NSAPaddr& x)
	{ return !(*this == x); }	// non-equivalence
    boolean operator!=(NETaddr&  x)
	{ return !(*this == x); }	// non-equivalence

    PSAPaddr () {}
    PSAPaddr (char*);		// Construct a PSAPaddr from an ascii string
				// of form: "psel/ssel/tsel/nsel/NS+afi+idi+dsp"
    PSAPaddr (NSAPaddr& tmp) { *this = tmp; }	/* !WARNING move! */
    PSAPaddr (TSAPaddr& tmp) { *this = tmp; }	/* !WARNING move! */
    PSAPaddr (SSAPaddr& tmp) { *this = tmp; }	/* !WARNING move! */
    PSAPaddr (PSAPaddr& tmp) { *this = tmp; }	/* !WARNING move! */
};

typedef PSAPaddr OSIaddr;	// alias


#endif	/* ISOADDRS wrapper */
