/**************************************************************
 * Send any comments or questions to: OTSO-Bug@tel.vtt.fi
 *
 * Name: /home/users/otso/official/otso/dvops/SCCS/s.sampled.cxx
 * Vers: 5.1    Time: 92/08/04, 15:11:22
 **************************************************************/

#ifdef SCCS_ID
/* for Unix 'what' command */
static char sccs_id[] = "@(#)sampled.cxx	5.1 92/08/04";
#endif

/***************************************************************
* Copyright (c) 1992      Technical Research Centre of Finland (VTT)
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose and without fee is hereby granted, provided
* that this notice and the reference to this notice appearing in each software
* module be retained unaltered, and that the name of any contributors shall not
* be used in advertising or publicity pertaining to distribution of the software
* without specific written prior permission.  No contributor makes any
* representations about the suitability of this software for any purpose.
* It is provided "as is" without any express or limited warranty.
*
*			NO WARRANTY
*
* ALL CONTRIBUTORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
* INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS.  IN NO
* EVENT SHALL ANY CONTRIBUTOR BE LIABLE FOR ANY SPECIAL, PUNITIVE, INDIRECT OR
* CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
* DATA, OR PROFITS, WHETHER IN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
* TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH, THE USE OR PERFORMANCE
* OF THIS SOFTWARE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THIS
* SOFTWARE IS WITH YOU.  SHOULD THIS SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE
* COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
*
* As used above, "contributor" includes, but is not limited to :
*        The Technical Research Centre of Finland
***************************************************************/


//sampled.cxx	- Juha 18.3.92

/**********************************************************************
This file is the combination of 
 - histogram.cxx
 - student.cxx
 - samplnum.cxx
 - samplcon.cxx
 - sampcohg.cxx
 - sampldis.cxx
 - sampdihg.cxx
 - samplrat.cxx
**********************************************************************/

//#include "channel.hxx"
//#include "statobj.hxx"
//#include "sampled.hxx"
//#include "time.hxx"
#include "OTSO.hxx"

#if SIMULATING			/* defined only after #including OTSO.hxx */

#include <values.h>
#include <math.h>

/************* student.cxx *******************************************/

// t-distribution: given p-value and degrees of freedom, return t-value
// adapted from Peizer & Pratt JASA, vol63, p1416

double student(double p, int df) 
{
  double t;
  int positive = p >= 0.5;
  p = (positive)? 1.0 - p : p;
  if (p <= 0.0 || df == 0)
    t = HUGE;
  else if (p == 0.5)
    t = 0.0;
  else if (df == 1)
    t = 1.0 / tan((p + p) * 1.57079633);
  else if (df == 2)
    t = sqrt(1.0 / ((p + p) * (1.0 - p)) - 2.0);
  else
  {	
    double ddf = df;
    double a = sqrt(log(1.0 / (p * p)));
    double aa = a * a;
    a = a - ((2.515517 + (0.802853 * a) + (0.010328 * aa)) /
             (1.0 + (1.432788 * a) + (0.189269 * aa) +
              (0.001308 * aa * a)));
    t = ddf - 0.666666667 + 1.0 / (10.0 * ddf);
    t = sqrt(ddf * (exp(a * a * (ddf - 0.833333333) / (t * t)) - 1.0));
  }
  return (positive)? t : -t;
}

/********** histogram.cxx ********************/

Histogram::Histogram(double lo_, double hi_, double width_)
{
	if (lo_ > hi_) {
		dout << "Histogram::ctor(); low is bigger than high\n";
		exit (1);
	}

	if ((hi_ - lo_) / width_ > 65535) {
		dout << "Histogram::ctor(); max index would exceed 65535\n";
		exit (1);
	}

	lo = lo_;
	hi = hi_;
	wi = width_;
}

sint32 Histogram::index(double v)
{
	if (v < lo || v > hi) return -1;

	return ((v - lo) / wi);
}


/********** samplnum.cxx ********************/

SampledNum::SampledNum(double v)
{
	val = v;
	reset();
}

void SampledNum::reset()
{
	x = x2 = 0.0;
	minValue = MAXFLOAT;
	maxValue = -MAXFLOAT;
}

void SampledNum::print(Ostream& os)
{
  os << form("%7.2f [%7.2f, %7.2f] (%7.2f)\n", mean(), min(), max(), stdDev());
}


/********** samplcon.cxx ********************/

Time SampledCont::sampleInterval()
{
	checkCount();
	return stopCnt - startCnt;
}

void SampledCont::updIgral()
{
	stopIVal = time();

	ival = stopIVal - startIVal;
	tmp  = (double) ival;

	x  += (val * tmp);
	x2 += (val * val * tmp);

	updDerived(ival);

	startIVal = stopIVal;
}

SampledCont::SampledCont(double v)
	: SampledNum(v)
{
	reset();
}

void SampledCont::reset()
{
	startCnt.reset();
	stopCnt.reset();

	SampledNum::reset();
}

void SampledCont::operator+=(double d)
{
  //printf("@@@    Entering SampledCont::op+=()\n");
	if (startCnt.isSet() && !stopCnt.isSet()) {
		updIgral();
		if (minValue > d) minValue = d;
		if (maxValue < d) maxValue = d;
	}

	val = d;
  //printf("@@@Exiting SampledCont::op+=()\n");
}

double SampledCont::mean()
{
	return x / (double) sampleInterval();
}

double SampledCont::var()
{
	ival = (double) sampleInterval();
	return (x2 - x*x / ival) / ival;
}

double SampledCont::stdDev()
{
	double	v;

	if ((v = var()) < 0.0) {
		dout << "SampledCont::stdDev(); internal error\n";
		exit (1);
	} else
		return (double) sqrt(v);
}

void SampledCont::start()
{
	if (startCnt.isSet()) {
		dout << "SampledCont::start(); another start?\n";
		exit (1);
	}

	startCnt = startIVal = time();
}

void SampledCont::stop()
{
	if (!startCnt.isSet() || stopCnt.isSet()) {
		dout << "SampledCont::stop(); cannot stop\n";
		exit (1);
	}

	updIgral();
	stopCnt = stopIVal;
}

void SampledCont::checkCount()
{
	if (!hasBeenCounted()) {
		dout << "SampledCont::checkCount(); no interval!\n";
		exit (1);
	}
}

SampledCont::~SampledCont() {}

/********** sampcohg.cxx ********************/

SampledContHg::SampledContHg(double lo, double hi, double width)
	: Histogram(lo, hi, width)
{
	shares	= new Time[maxIndex() + 1];
	reset();
}

SampledContHg::SampledContHg(const SampledContHg& sch)
	: SampledCont(sch), Histogram(sch.low(), sch.high(), sch.width())
{
	shares	= new Time[maxIndex() + 1];
	for (uint32 i = 0; i <= maxIndex(); i++)
		shares[i] = sch.shares[i];
}

Time SampledContHg::spentBySimilar(double v)
{
	checkCount();

	sint32	ix = index(v);

	if (ix != -1)
		return shares[ix];
	else
		return (uint32) 0;
}

void SampledContHg::reset()
{
	SampledCont::reset();

	for (uint32 i = 0; i <= maxIndex(); i++)
		shares[i] = (uint32) 0;
}

void SampledContHg::updDerived(Time iv)
{
	sint32	ix = index(value());

	if (ix != -1) shares[ix] = shares[ix] + iv;
}

double SampledContHg::perCentOfSimilar(double v)
{
	return 100.0 * (double) spentBySimilar(v) / (double) sampleInterval();
}


/********** sampldis.cxx ********************/

SampledDisc::SampledDisc()
{
	reset();
}

void SampledDisc::reset()
{
	n = 0;
	SampledNum::reset();
}

void SampledDisc::operator+=(double d)
{
	val = d;

	n++;
	x  += d;
	x2 += (d * d);
	if (minValue > d) minValue = d;
	if (maxValue < d) maxValue = d;
}

double SampledDisc::mean()
{
	return (n > 0) ? x / n : 0.0;
}

double SampledDisc::var()
{
	if (n > 1) {
		return (x2 - x*x / n) / (n - 1);
	} else
		return 0.0;
}

double SampledDisc::stdDev()
{
	double	v;

	if (n <= 0 || (v = var()) < 0.0) {
		dout << "SampledDisc::stdDev(); internal error\n";
		exit (1);
	} else
		return (double) sqrt(v);
}

double SampledDisc::confForMean(double p_value)
{
	if (size() < 2) return HUGE;

	double t = student((1.0 + p_value) * 0.5, (int) size() - 1);

	if (t == HUGE)
		return t;
	else
		return t * stdDev() / sqrt(double( size() ));
}

void SampledDisc::print(Ostream& os)
{
	if (!size()) {
		os << "no samples\n";
		return;
	}

	SampledNum::print(os);
}


/********** sampdihg.cxx ********************/

SampledDiscHg::SampledDiscHg(double lo, double hi, double width)
	: Histogram(lo, hi, width)
{
	shares	= new uint32[maxIndex() + 1];
	reset();
}

uint32 SampledDiscHg::noOfSimilar(double v)
{
	sint32	ix = index(v);

	if (ix != -1)
		return shares[ix];
	else
		return 0;
}

void SampledDiscHg::reset()
{
	SampledDisc::reset();

	for (uint32 i = 0; i <= maxIndex(); i++)
		shares[i] = 0;

	accCount = 0;
}

void SampledDiscHg::operator+=(double v)
{
	sint32	ix = index(v);

	if (ix != -1) {
		++accCount;
		++shares[ix];
	}

	SampledDisc::operator+=(v);
}

double SampledDiscHg::perCentOfSimilar(double v)
{
	return (accCount) ? 100.0 * (double) noOfSimilar(v) / accCount
			  : 0.0;

}


/********** samplrat.cxx ********************/

SampledRatio::SampledRatio()
{
	reset();
}

void SampledRatio::reset()
{
	ntor.reset();
	dtor.reset();
	xy = 0.0;
}

double SampledRatio::mean()
{
	if (size() > 0)
		return ntor.mean() / dtor.mean();
	else
		return 0.0;
}

double SampledRatio::var()
{
	double	m = mean();

	return ntor.var() - 2 * m * covar() + m * m * dtor.var();
}

double SampledRatio::confForMean(double p_value)
{
	if (size() < 2) return HUGE;

	double t = student((1.0 + p_value) * 0.5, (int) size() - 1);

	if (t == HUGE)
		return t;
	else
		return t * stdDev() / (dtor.mean() * sqrt(size()));
}

void SampledRatio::ratioSample(double n, double d)
{
	ntor += n;
	dtor += d;

	xy += (n * d);
}

double SampledRatio::covar()
{
	uint32	n = size();

	if (n > 1)
		return (xy - n * ntor.mean() * dtor.mean()) / (n - 1);
	else
		return 0.0;
}


#endif 	/*SIMULATING*/
