/**************************************************************
 * Send any comments or questions to: OTSO-Bug@tel.vtt.fi
 *
 * Name: /home/users/otso/official/otso/dvops/SCCS/s.cframe.c
 * Vers: 5.3    Time: 92/08/13, 11:53:59
 **************************************************************/

#ifdef SCCS_ID
/* for Unix 'what' command */
static char sccs_id[] = "@(#)cframe.c	5.3 92/08/13";
#endif

/***************************************************************
* Copyright (c) 1992      Technical Research Centre of Finland (VTT)
*
* Permission to use, copy, modify, and distribute this software and its
* documentation for any purpose and without fee is hereby granted, provided
* that this notice and the reference to this notice appearing in each software
* module be retained unaltered, and that the name of any contributors shall not
* be used in advertising or publicity pertaining to distribution of the software
* without specific written prior permission.  No contributor makes any
* representations about the suitability of this software for any purpose.
* It is provided "as is" without any express or limited warranty.
*
*			NO WARRANTY
*
* ALL CONTRIBUTORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
* INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTIBILITY AND FITNESS.  IN NO
* EVENT SHALL ANY CONTRIBUTOR BE LIABLE FOR ANY SPECIAL, PUNITIVE, INDIRECT OR
* CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
* DATA, OR PROFITS, WHETHER IN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
* TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH, THE USE OR PERFORMANCE
* OF THIS SOFTWARE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THIS
* SOFTWARE IS WITH YOU.  SHOULD THIS SOFTWARE PROVE DEFECTIVE, YOU ASSUME THE
* COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
*
* As used above, "contributor" includes, but is not limited to :
*        The Technical Research Centre of Finland
***************************************************************/


 /********************************************************************
 *********************************************************************
 * NOTE:
 * THIS code *should* be shared with CVOPS.
 * This was taken from the OLD DVOPS (when we moved to 
 * the SUN SPARC STATION and GNU G++ 2.0):
 *  Name:	/files/dvops/dvops/SCCS/s.frame.c
 *  Version:	1.1
 * which was taken from CVOPS:
 *  Name:	/files/cvops/code/cvops/SCCS/s.frame.c
 *  Version:	1.4
 *********************************************************************
 **********************************************************************/

/*************************************************************
 *
 *    FRAME.C      VERSION 3.15           8.8.89
 *
 *    ORIGINAL AUTHORS: J.Malka
 *
 *    MODULE NAME: FRAME
 *
 *
 *    PURPOSE:
 *    Contains functions for handling frames.
 *
 *    GLOBAL FUNCTIONS:
 *     DATA_BLOCK *newDataBlock ()
 *     FRAME *createFrame()
 *     void deleteFrame()
 *     void putFirstByte()
 *     void putnFirstBytes()
 *     BYTE_TYPE getFirstByte()
 *     int getnFirstBytes()
 *     BYTE_TYPE readByte()
 *     void writeByte()
 *     INT32 frameLength ()
 *     void combineFrame ()
 *     FRAME *splitFrame ()
 *     void deleteBytes()
 *     int toHex ()
 *     int nextByte ()
 *     FRAME *inputFrame()
 *     void showAscii()
 *     void outputFrame()
 *     void outputFrameBin()
 *     void copyBytesToArray()
 *     FRAME *copyFrame ()                    
 *     DATA_BLOCK *getFirstDataBlock()
 *     
 *
 *    STATIC FUNCTIONS:
 *     void copyDataBlocks ()
 *
 *    MODIFICATIONS TO VERSION 3.0 :
 *    Changed compilations dependent on NO_IO_SYSTEM or NO_DEBUG to
 *    be dependent on NO_TERM_INPUT and NO_TERM_OUTPUT
 *    Included enviro.def. 23.6.87 EO
 *    Fixed block overflow with small block sizes in inputFrame. 29.6.87 JeL
 *    Input and output with trace functions. 23.10.87 KV
 *    Old outputFrame renamed outputFrameBin. 30.11.87 EO
 *    New outputFrame, prints in hexa format. 30.11.87 EO
 *    Added inputFrameHex(), nextHex() and toHex(). 30.12.87 EO
 *    Corrected splitframe. 23.1.88 SM
 *    Added 'hidden' PREFIX and SUFFIX to data blocks. 14.3.88 JeL
 *    Merged inputFrame and inputFrameHex, nextHex -> nextByte. 15.3.88 JeL
 *    Modified copyFrame and added static function copyDataBlocks. 22.3.88 SM
 *    outputFrame dumps 16 chars per line (no autowrap!). 23.3.88 JeL
 *    Modified copyFrame. 25.3.88 SM  
 *    Added getFirstDataBlock. 13.5.88 SM
 *    Changed _movebytes with memcpy. 21.6.88 SM
 *    Added copyFrameToString(). 31.1.89 EO
 *    Some NO_DEBUG flags added. 8.8.89 JKu
 *    Removed finally inputFrameHex. 28.8.89 JeL
 *    (POP1) 21.12.89: Defines LIFO and FIFO added and several functions
 *                     changed to enable FIFO handling of data blocks
 *                     (search FIFO and LIFO)
 *    (POP2) 07.01.90: Including files put behind switch STAND_ALONE_ASN.
 *
 ************************************************************/


/*   ------   OTHER MODULES USED   ----------------------   */

/*
 *             UTILS
 *             MEMORY
 */

/*   ------   PROGRAM WIDE DEFINITIONS ------------------   */

#define ARGS(s)  ()
extern char  *malloc ARGS((unsigned));

#include "frame.typ"

#define DVOPS 1

/*   ------   LOCAL DEFINITIONS   -----------------------   */

#if QP
#define PREFIX 24
#define SUFFIX 2
#define COPY_PREFIX 5
#else
#define COPY_PREFIX 0
#define PREFIX 0
#define SUFFIX 0
#endif

#define DB_SIZE 128       /* default size of the datablocks*/
#define MAX_DB_SIZE 1024  /* maximum size of datablocks (used in copyFrame) */
#define LINE_LENGTH 50    /*in the output*/

#define LIFO	0 /* by Pekka Ollikainen 21.12.1989 (POP1) */
#define FIFO	1 /* by Pekka Ollikainen 21.12.1989 (POP1) */


/* ====   MEMORY FUNCTIONS USED BY FRAME (CVOPS MEMORY.C NOT AVAILABLE) === */

char* walloc (nbytes)
unsigned nbytes;
{
  return malloc(nbytes); /* SPARC, in C++: new char[nbytes]; */
}

char* wzalloc (nbytes)
unsigned nbytes;
{
  char* p = walloc (nbytes);
#if 1
  memset(p, 0, nbytes);
#else
  while (nbytes)
    p[--nbytes] = 0;
#endif
  return p;
}

void wfree (ptr)
void* ptr;
{
  free(ptr); /*SPARC, in C++: delete ptr; */
}

/********************************************************************/

/*   ======   LOCAL VARIABLES AND ALL FUNCTIONS   =======   */

/************************************************************
 *
 *   GLOBAL FUNCTION: newDataBlock
 *
 *   FUNCTION:
 *   Creates a new datablock and returns a pointer to it. 
 *
 *   PARAMETERS:  int size: size of the data part. (input)
 *                  
 *************************************************************/


DATA_BLOCK *newDataBlock(size,mode)
int size,mode;
{
  DATA_BLOCK *db;

  db = (DATA_BLOCK *)walloc(sizeof(DATA_BLOCK));  
  db->next = NULL;
  db->size = size + PREFIX + SUFFIX;
  db->data = (BYTE_TYPE *)walloc(db->size);
  db->n = 0;
  if (mode == LIFO) {
    db->i = db->size-SUFFIX;
    db->j = db->i-1;
  }
  else {
    db->i = PREFIX+1;
    db->j = PREFIX;
  }
  return(db);
}


/************************************************************
 *
 *   GLOBAL FUNCTION: createFrame
 *
 *   FUNCTION:
 *   Creates a new frame and returns a pointer to it.
 *
 *   PARAMETERS:  int size: the size of the datablocks in this
 *                          frame (not necessarily). (input)
 *                       
 *************************************************************/

FRAME *createFrame(size)
int size;  
{
  FRAME *frame;

  frame = (FRAME *)wzalloc(sizeof(FRAME));
  frame->dbSize = size;
  return (frame);
}



/************************************************************
 *
 *   GLOBAL FUNCTION: deleteFrame
 *
 *   FUNCTION:
 *   Deletes a frame. 
 *
 *   PARAMETERS: FRAME **frameAddr: the address of the frame
 *                            (pointer) to be deleted. 
 *************************************************************/
void deleteFrame(frameAddr)
FRAME **frameAddr;
{
  DATA_BLOCK *db1,*db2;  
  FRAME *frame;

  frame = *frameAddr;
  if (frame){
    for (db1 = frame->first; db1;) {
      db2 = db1->next;
      wfree(db1->data);
      wfree(db1);
      db1 = db2;
    }
    wfree(frame);
    *frameAddr = NULL;
  }
}


/************************************************************
 *
 *   GLOBAL FUNCTION: putFirstByte
 *
 *   FUNCTION:
 *   Inserts a given byte to the beginning of the frame.
 *   A new datablock is created when needed.
 *
 *   PARAMETERS: FRAME **frameAddr: the address of the frame (input)
 *               BYTE_TYPE byte:    the byte to be inserted. (input)           
 *************************************************************/

void putFirstByte(frameAddr,byte)
FRAME **frameAddr;
BYTE_TYPE byte;
{
  DATA_BLOCK *db;  
  FRAME *frame;

  frame = *frameAddr;
  if (frame == NULL)
     frame = *frameAddr = createFrame (DB_SIZE);

  if ((db = frame->first) == NULL) 
     db = frame->first = newDataBlock(frame->dbSize,LIFO);
  else  
     if (db->i == PREFIX) {
        db = newDataBlock(frame->dbSize,LIFO);
        db->next = frame->first;
        frame->first = db;
     }
  db->i -= 1;
  db->n += 1;
  db->data[db->i] = byte;
  frame->length += 1;
}


/************************************************************
 *
 *   GLOBAL FUNCTION: putnFirstBytes
 *
 *   FUNCTION:
 *   Inserts given bytes to the beginning of the frame.
 *
 *   PARAMETERS: FRAME **frameAddr: the address of the frame. (input)
 *               BYTE_TYPE *bytes:  the bytes to be inserted. (input)           
 *               int   n:           number of bytes to be inserted. (input)
 *************************************************************/

void putnFirstBytes(frameAddr,bytes,n)
FRAME **frameAddr;
BYTE_TYPE *bytes;
int n;
{
  DATA_BLOCK *db;  
  FRAME *frame;
  int nn;

  bytes += n;                  /* point to last byte + 1 */
  frame = *frameAddr;
  if (frame == NULL)
     frame = *frameAddr = createFrame (DB_SIZE);

  if ((db = frame->first) == NULL) 
     db = frame->first = newDataBlock(frame->dbSize,LIFO);

  frame->length += n;

  while (n > 0)
     {
     if (db->i == PREFIX) 
        {
        db = newDataBlock(frame->dbSize,LIFO);
        db->next = frame->first;
        frame->first = db;
        }

     if (n+PREFIX <= db->i)
	nn = n;			/* all fits to this block */
     else
	nn = db->i-PREFIX;	/* this much fits */

     db->i -= nn;
     db->n += nn;
     bytes -= nn;
     memcpy(&db->data[db->i], bytes, nn);
     n -= nn;
     }
}


    
/************************************************************
 *
 *   GLOBAL FUNCTION   putLastByte *
 *
 *   FUNCTION          Inserts a given byte to the end 
 *                     of the buffer.
 *
 *   PARAMETERS        FRAME **frameAddr : the address of the frame (input)     
 *                     BYTE_TYPE byte :    the byte to be inserted. (input)
 *
 *************************************************************/

void putLastByte(frameAddr,byte)
FRAME **frameAddr;
BYTE_TYPE byte;
{
  DATA_BLOCK *db, *db2;  
  FRAME *frame ;

  frame = *frameAddr ;
  if (frame == NULL)
     frame = *frameAddr = createFrame (DB_SIZE) ;
     
  if ((db = frame->first) == NULL) 
     db = frame->first = newDataBlock(frame->dbSize,FIFO) ;     
  else 
    for (; db->next != NULL; db = db->next) ;

  if (db->j == (db->size)-1) 
    db = db->next = newDataBlock(frame->dbSize,FIFO) ;

  db->j += 1 ; 
  db->n += 1;
  db->data[db->j] = byte;
  frame->length += 1;
}


/************************************************************
 *
 *   GLOBAL FUNCTION: putnLastBytes  
 *
 *   FUNCTION:        Inserts given bytes to the end of the frame.
 *
 *   PARAMETERS: FRAME **frameAddr: the address of the frame. (input)
 *               BYTE_TYPE *bytes:  the bytes to be inserted. (input)           
 *               int n:             number of bytes to be inserted. (input)
 *
 *************************************************************/

void putnLastBytes(frameAddr,bytes,n)
FRAME **frameAddr ;
BYTE_TYPE *bytes;
int n;
{
  DATA_BLOCK *db ;  
  FRAME *frame ;
  int nn ;

  frame = *frameAddr ;
  if (frame == NULL)
     frame = *frameAddr = createFrame (DB_SIZE) ;
     
  if ((db = frame->first) == NULL)
     db = frame->first = newDataBlock(frame->dbSize,FIFO) ; 
 
  frame->length += n ;
  
  for (; db->next != NULL; db = db->next) ;
  
  while (n > 0) {
    if (db->j == (db->size)-1)         /* current db is full */
       db = db->next = newDataBlock(frame->dbSize,FIFO) ;
       
    if ((db->j)+n >= db->size)    /* doesn't fit into current db */
       nn = db->size - (db->j)-1 ;
    else
       nn = n ;
    
    memcpy(&db->data[(db->j)+1], bytes, nn) ;
    db->j += nn ;
    bytes += nn ;
    db->n += nn ;

    n -= nn ;
  }
}


    
/************************************************************
 *
 *   GLOBAL FUNCTION: getFirstByte
 *
 *   FUNCTION:
 *   Returns the first byte of a given frame. The byte is 
 *   removed from the frame. If the frame becomes empty,
 *   it is deleted. 
 *
 *   PARAMETERS: FRAME **frameAddr: the address of the frame
 *                                  (pointer). 
 *************************************************************/
BYTE_TYPE getFirstByte(frameAddr)
FRAME **frameAddr;
{
  DATA_BLOCK *db;
  BYTE_TYPE byte;
  FRAME *frame;
 
  frame = *frameAddr;
  if (frame->length--) {
     db = frame->first;
     while (db->n == 0){
         frame->first = db->next;
         wfree(db->data);
         wfree(db);
         db = frame->first;
     }              
     byte = db->data[db->i++];
     db->n -= 1;
     while (db && db->n == 0){
         frame->first = db->next;
         wfree(db->data);
         wfree(db);
         db = frame->first;
     }
     if (db == NULL){
        wfree (frame);
        *frameAddr = NULL;
     }
     return (byte);
  }     
  else {
     /*error, frame is empty*/      
     return(0);
  }
}




/************************************************************
 *
 *   GLOBAL FUNCTION: getnFirstBytes
 *
 *   FUNCTION:
 *   Returns n first bytes of a given frame. The bytes are
 *   removed from the frame. If the frame becomes empty,
 *   it is deleted. 
 *
 *   PARAMETERS: FRAME **frameAddr: the address of the frame(pointer).
 *               BYTE_TYPE *bytes:  pointer to the place where
 *                                  the bytes are copied.(input)
 *               int n:            number of bytes to be taken.(input)
 *************************************************************/
int getnFirstBytes(frameAddr,bytes,n)
FRAME **frameAddr;
BYTE_TYPE *bytes;
int n;
{
  DATA_BLOCK *db;
  FRAME *frame;
  int N;
  int nn;
 
  frame = *frameAddr;
  if (frame && n>frame->length)
      n=frame->length;
  else 
      if (frame==NULL)               /*frame is empty*/
         return(0);

  frame->length -= n;                /* set size*/

  N=n;
  db = frame->first;
  while (n > 0)                       /*copy*/
     {
     while (db->n == 0)
         {
         frame->first = db->next;
         wfree(db->data);
         wfree(db);
         db = frame->first;
         }

     if(n < db->n) nn = n;
     else nn = db->n;

     memcpy(bytes, &db->data[db->i], nn);
     bytes += nn;
     db->n -= nn;
     db->i += nn;
     n -= nn;
     }  


  while (db && db->n == 0){                /*delete empty datablocks*/
      frame->first = db->next;
      wfree(db->data);
      wfree(db);
      db = frame->first;
  }              

  if (db == NULL){                         /*delete frame if empty*/
        wfree (frame);
        *frameAddr = NULL;
  }
      
  return (N);

}


/************************************************************
 *
 *   GLOBAL FUNCTION: readByte
 *
 *   FUNCTION:
 *   Returns a wanted byte of a frame.
 *
 *   PARAMETERS: FRAME *frame: the frame    (input)
 *               int n:        the number of the wanted byte.(input)
 *
 *************************************************************/

BYTE_TYPE readByte(frame,n)
FRAME *frame;
int n;
{
  DATA_BLOCK *db;
  int s;

  if (frame && n>0 && n <= frame->length){
    db = frame->first;
    for (s=0; n>(s += db->n); db=db->next);
    return(db->data[db->j -(s-n)]);
  }
  else 
     return(0);
}

/************************************************************
 *
 *   GLOBAL FUNCTION: writeByte
 *
 *   FUNCTION
 *   Changes the value of the n:th byte. 
 *
 *   PARAMETERS: FRAME *frame:   the frame. (input)
 *               int n:          the number of the byte. (input)
 *               BYTE_TYPE byte: the new value of the byte (input)
 *************************************************************/

void writeByte(frame,n,byte)
FRAME *frame;
int n;
BYTE_TYPE byte;
{
  DATA_BLOCK *db;
  int s;

  if (frame && n>0 && n <= frame->length){
    db = frame->first;
    for (s=0; n>(s += db->n); db=db->next);
    db->data[db->j -(s-n)] = byte;
  }          
}



/************************************************************
 *
 *   GLOBAL FUNCTION: frameLength
 *
 *   FUNCTION:
 *   Returns the number of bytes in a frame.
 *
 *   PARAMETERS: FRAME *frame:  (input)
 *               
 *************************************************************/
        
INT32 frameLength (frame)
FRAME *frame;
{
  if (frame)
     return(frame->length);
  else return((INT32) 0);
}



/************************************************************
 *
 *   GLOBAL FUNCTION: combineFrame
 *
 *   FUNCTION:
 *   Combines two frames. The second frame will be concatenated
 *   to the end of the first frame.
 *
 *   PARAMETERS: FRAME **frame1Addr: the first frame.   (input)
 *                               The concatenated frame.(output)
 *               FRAME **frame2Addr: the second frame.  (input)
 *                                   NULL               (output)
 *
 *************************************************************/

void combineFrame (frame1Addr,frame2Addr)
FRAME **frame1Addr,**frame2Addr;
{
  DATA_BLOCK *lastdb;
  FRAME *frame1,*frame2;

  frame1 = *frame1Addr;
  frame2 = *frame2Addr;

  if (frame2 && frame2->length){
     if (frame1 && frame1->length){
            frame1->length += frame2->length;
            for (lastdb=frame1->first;lastdb->next;lastdb=lastdb->next);
            lastdb->next = frame2->first;
            wfree (frame2);
            *frame2Addr = NULL;}
     else   {
            if (frame1) deleteFrame(frame1Addr);     
            *frame1Addr = *frame2Addr;
            *frame2Addr = NULL;
     }
  }
  else  
     if (frame2) deleteFrame(frame2Addr);
}



/************************************************************
 *
 *   GLOBAL FUNCTION: splitFrame
 *
 *   FUNCTION:
 *   Splits a given frame into two parts. 
 *   First one contains n first bytes of the frame and second 
 *   one the rest. The first n bytes are put into a new frame 
 *   (returned by the function) and the rest is left in the
 *   old frame.
 *
 *   PARAMETERS:  FRAME **frameAddr: the frame.
 *                int n : number of bytes to be splitted 
 *                        off from the given frame. (input)
 *                       
 *************************************************************/

FRAME *splitFrame (frameAddr,n)
FRAME **frameAddr;
unsigned n;
{
  FRAME *frame, *frame1;
  DATA_BLOCK *db,*db1, *prevDb;
  int s;
  int nn;

  frame = *frameAddr;
  if (frame && n > 0){
     if (n >= frame->length){
        frame1=frame;  
        *frameAddr = NULL;           
     }   
     else {
        frame1 = createFrame(frame->dbSize);
        /* get datablocks until more or equal to needed amount */
        db1 = frame1->first = frame->first;
        prevDb = NULL;
        s = db1->n;
        while(s < n)
        {
           prevDb = db1;
           db1=db1->next;
           s += db1->n;
        }

        frame->length -= (frame1->length = n);
        if (s==n){
           frame->first = db1->next;
           db1->next = NULL;
        }
        else 
        {  /* put the last taken datablock back in the input frame and
              and move bytes from the front of the datablock to 
              the output frame */

           frame->first = db = db1;
           s -= db1->n;

           while(s < n)
           {
               db1 = newDataBlock(frame->dbSize,LIFO);
               if(prevDb == NULL) frame1->first = db1;
               else prevDb->next = db1;
               prevDb = db1;
               nn = n - s;
               if(nn > (db1->size-PREFIX-SUFFIX))
                  nn = (db1->size-PREFIX-SUFFIX); /* number of available bytes in db */
               
               db1->i -= nn;
               db1->n += nn;
               memcpy(&db1->data[db1->i], &db->data[db->i], nn);
               db->n -= nn;
               db->i += nn;
               s += nn;
           }
        }
     }
     return(frame1);
  }
  else  return(NULL);

}


/************************************************************
 *
 *   GLOBAL FUNCTION: deleteBytes
 *
 *   FUNCTION:
 *   Deletes n first bytes of the frame. If n is more than 
 *   the length of the frame the whole frame will be deleted.
 *
 *   PARAMETERS: FRAME **frameAddr: the frame.
 *               int n :the number of bytes to be deleted.(input)
 *
 *************************************************************/

void deleteBytes(frameAddr,n)
FRAME **frameAddr;
int n;
{
  DATA_BLOCK *db,*dbnext;
  FRAME *frame;
  int s;
 
  frame = *frameAddr;
  if (frame && n>0)
     if (n>=frame->length) 
        deleteFrame(frameAddr);
     else {
        db = frame->first;
        s = db->n;
        while (s<=n){
           dbnext=db->next;
           wfree(db->data);
           wfree(db);
           db = dbnext;
           s += db->n;
        }
        frame->first = db;
        frame->length -= n;
        db->n = s-n; 
        db->i = db->j+1-(s-n);
     }
}  



/************************************************************
 *
 *   GLOBAL FUNCTION : toHex
 *
 *   PURPOSE : Convert a char to hexa format.
 *
 *   RETURNS : 0...15
 *
 *   NOTE : Returns -1 if error.
 *
 ************************************************************/

int toHex (ch)
char ch;
{
   if (ch >= '0' && ch <= '9') return (ch - '0');
   if (ch >= 'A' && ch <= 'F') return (ch - 'A' + 10);
   if (ch >= 'a' && ch <= 'f') return (ch - 'a' + 10);
   return(-1);  /* erroneous input */
}

#if !DVOPS

/************************************************************
 *
 *   GLOBAL FUNCTION : nextByte
 *
 *   PURPOSE : Return the next byte value from input.
 *             Recognizes backslash escapes \n, \t, \\ and
 *             \<hexadecimal value>.
 *             If a bad escape sequence is seen the returned
 *             value is somewhat arbitrary.
 *
 *   RETURNS : 0...255 or EOLN (-1) if end of line or file is seen.
 *
 ************************************************************/

#define EOLN (-1)

int nextByte ()
{
#if !NO_DEBUG
   int c, hex;

   c = wgetchar();

   if (c == '\n') {         /* end of line */
     c = EOLN;
   } else if (c == '\\') {  /* looks like backslash escape */
      c = wgetchar();
      switch (c) {
      case '\\':
         c = '\\';
         break;
      case 'n':             /* newline */
         c = '\n';
         break;
      case 't':	            /* tab */
         c = '\t';
         break;
      default:		    /* might be hexadecimal value */
         hex = toHex(c);
         if (hex == -1) {	/* was not */
            hex = c;
         } else {
            c = toHex(wgetchar());	/* convert next digit */
            if (c != -1) {
               hex = hex << 4 | c;
            }
         }
      	 c = hex;
         break;
      }
   }
   return c;
#endif
}

/************************************************************
 *
 *   GLOBAL FUNCTION: inputFrame
 *
 *   FUNCTION:
 *   Returns a frame filled with the given data.
 *   
 *   PARAMETERS: int size: size of the datablocks in the frame.(input)
 *
 *************************************************************/

FRAME *inputFrame(size)
int size;
{
#if !(NO_TERM_OUTPUT || NO_TERM_INPUT)
#define HEADER_LENGTH 100
   FRAME *frame;
   int c,k,s;
   DATA_BLOCK *db;

   frame = createFrame(size);
   /* ...+HEADER_LENGTH is a stupid patch but should not break anything. JeL */
   frame->first = db = newDataBlock(size+HEADER_LENGTH,LIFO);
   s = 0;
   k = PREFIX+HEADER_LENGTH;
   traceLine("\n>");

   c = nextByte();
   if (c == EOLN) c = nextByte();   /* allow an extra newline */

   while (c != EOLN) {
      while (c != EOLN) {
         if (k > db->j) {           /* data block became full */

            db->next = newDataBlock(size+HEADER_LENGTH,LIFO); 
            db->i = PREFIX+HEADER_LENGTH;
            db->n = k - db->i;
            db = db->next;
            k = PREFIX+HEADER_LENGTH;
         }
         db->data[k++] = c; 
         s += 1;
         c = nextByte();
      }
      traceLine(">");
      c = nextByte();           /* empty line will terminate input */
   }
   db->j = k - 1;
   db->i = PREFIX+HEADER_LENGTH; 
   db->n = k - db->i;
   frame->length = s;
   return(frame);
#endif
}


/************************************************************
 *
 *   GLOBAL FUNCTION: outputFrameBin
 *
 *   FUNCTION:
 *   Outputs the given frame in binary and ascii.
 *
 *   PARAMETERS: FRAME *frame:  (input)
 *               int count   :  (input) Nr of bytes to be
 *                              printed
 *
 *************************************************************/

void outputFrameBin (frame,count)
FRAME *frame;
int count;
{
#if !NO_DEBUG
#define SYMBOLS_ON_LINE 8
#define SYMBOL_LENGTH 8
  static char line [SYMBOL_LENGTH + 2];
  static char ascii [SYMBOLS_ON_LINE + 1];
  int i,j,k;
  unsigned c;

  if (frame)
  {
    for (j=i=1; i <= count; i++,j++)
    {
      c = readByte (frame,i);
      for (k = 0; k < SYMBOL_LENGTH; k++)
        line [k] = (c & (0x80 >> k)) ? '1' : '0';
      line [k] = ' ';
      line [k + 1] = '\0';      
      traceLine (line);
      ascii [j - 1] = (c >= ' ' && c <= '~' ? c : '?');
      if (j == SYMBOLS_ON_LINE || i == count) 
      {
        sprintf (&ascii [j],"\n");
        for (; j < SYMBOLS_ON_LINE; j++)
          traceLine ("         ");
        traceLine (ascii);
        j = 0;
      }
    }
  }
#endif
}

#endif	/* !DVOPS */

/****************************************************************
 *
 *   STATIC FUNCTION: copyBytesToArray
 *
 *   FUNCTION:   copies a frame into a bytearray
 *
 *   PARAMETERS: frame, bytearray
 ****************************************************************/

void copyBytesToArray (frame, byteArray)
FRAME *frame;
BYTE_TYPE *byteArray;
{
register DATA_BLOCK *db;

if (frame != NULL)
    for(db = frame->first;db != NULL;db = db->next)
        {
        memcpy(byteArray, &db->data[db->i], db->n);
        byteArray += db->n;
        }

return;
}

/****************************************************************
 *
 *   STATIC FUNCTION: copyFrameToString
 *
 *   FUNCTION:   copies a frame into a null terminated string
 *
 *   PARAMETERS: frame, string, maxbytes
 ****************************************************************/

int copyFrameToString (frame, string, maxbytes)
    FRAME *frame;
    char *string;
    int maxbytes;
{
    int length;

    if (maxbytes < 1) return (-2); /* error */

    if ((length = frameLength(frame)) >= maxbytes) {
        string[0] = 0;
        return(-1); /* error */
    } else {
        copyBytesToArray (frame, string);
        string [length] = 0;
        return(0); /* ok */
    }
}


/****************************************************************
 *
 *   STATIC FUNCTION: copyDataBlocks
 *
 *   FUNCTION:   Copies a list of datablocks into a frame
 *
 *   PARAMETERS: pointer to frame, start of datablock-list
 *
 ****************************************************************/

static void copyDataBlocks (framePo, db)
FRAME **framePo;
DATA_BLOCK *db;
{
    if(db != NULL)
    {
        copyDataBlocks(framePo, db->next);
        if(db->n != 0) putnFirstBytes(framePo, &db->data[db->i], db->n);
    }
}   
  
 
/****************************************************************
 *
 *   GLOBAL FUNCTION: copyFrame
 *
 *   FUNCTION:   returns a copy of the given frame
 *
 *   PARAMETERS: frame
 *
 ****************************************************************/

FRAME *copyFrame (frame)
FRAME *frame;
{
FRAME *copy;
INT16 dbSize;

if (frame != NULL && (dbSize = frameLength(frame)) != 0){
   dbSize += ((INT16) COPY_PREFIX);
   if(dbSize > MAX_DB_SIZE) dbSize = MAX_DB_SIZE;
   if(dbSize < DB_SIZE) dbSize = DB_SIZE;
   copy = createFrame(dbSize);
   copyDataBlocks(&copy, frame->first);
   if(copy != NULL) 
   {
      copy->dbSize = frame->dbSize;
      if(frameLength(copy) == 0) deleteFrame(&copy);
   }
   return (copy);
}   
else
  return(NULL);
}


/****************************************************************
 *
 *     STATIC FUNCTION: getFirstDataBlock
 *
 *     FUNCTION: Gets first data block of a frame.
 *               If frame becomes empty frame is deleted
 *
 *     PARAMETERS: pointer to frame
 *
 ***************************************************************/

DATA_BLOCK *getFirstDataBlock(frPo)
FRAME **frPo;
{
    DATA_BLOCK *db;
    
    if(frPo == NULL || *frPo == NULL) return(NULL);
    
    db = (*frPo)->first;
    if(db == NULL || ((*frPo)->first = db->next) == NULL) 
    {
        wfree(*frPo);
        *frPo = NULL;
    }
    else  (*frPo)->length -= db->n;

    return(db);
}

/***********************************************************
 *
 *   GLOBAL FUNCTION: showAscii
 *
 *   FUNCTION:        Convert hexa 0...F to printable ascii.
 *
 *   PARAMETERS:      Hexa code to be converted.
 *
 ************************************************************/

unsigned char showAscii (c)
   unsigned char c;
{
   if (c < 10) return (c + '0');      /* c >= 0 */
   if (c < 16) return (c + 'A' - 10);
   return ('.');                      /* erroneous input */
}

#if !DVOPS

/***********************************************************
 *
 *   GLOBAL FUNCTION: outputFrame
 *
 *   FUNCTION:        Print a frame in hexa and ascii. See
 *                    also outputFrameBin().
 *
 *   PARAMETERS:
 ************************************************************/

void outputFrame (frame, count)
   FRAME *frame;
   int count;
{
#if !NO_DEBUG
#define CHARS_ON_LINE   16
#define printGap() woutput("")  /* additional space between hexas and ascii */

   unsigned char hexas[CHARS_ON_LINE*3+1],
                 ascii[CHARS_ON_LINE+1],
                 c;
   int f,h,a;

   if (frame) {
      h=a=0;
      for (f = 1; f <= count; f++) {
          c = readByte (frame, f);

          ascii[a++] = (c >= 32 && c <= 126 ? c : '?'); /* is printable */

          hexas[h++] = showAscii ((c >> 4) & 15);
          hexas[h++] = showAscii (c & 15);
          hexas[h++] = ' ';

          if (a == CHARS_ON_LINE) {     /* this is the end of a line */
             hexas[h] = ascii[a] = '\0';
             woutput(hexas); printGap(); woutput(ascii); woutput("\n");
             h=a=0;
          }
      }                          /* the whole frame has been read */
      if (a > 0) {                      /* flush buffers */
             hexas[h] = ascii[a] = '\0';
             woutput(hexas); while (a++ < CHARS_ON_LINE) woutput("   ");
             printGap(); woutput(ascii); woutput("\n");
      }
   }
#endif
}

#endif	/* !DVOPS */

/***********************************************************
 *
 *   GLOBAL FUNCTION: frameBlock
 *
 *   FUNCTION:	Allocate a block of memory which can later
 *		be converted to a frame by 'fakeFrame'
 *
 * FrameBlock destructor in frame.cxx is tighly coupled with the
 * implementation of this function!  PREFIX must be the same in both files!
 *
 *   PARAMETERS:
 ************************************************************/

BYTE_TYPE *frameBlock(size)
int size;
{
   /* Allocate a similar block as newDataBlock would do */
   /* and return a pointer to the user data area. */
   return (BYTE_TYPE *)(PREFIX + walloc(PREFIX + size + SUFFIX));
}


/***********************************************************
 *
 *   GLOBAL FUNCTION: fakeFrame
 *
 *   FUNCTION:	Convert a block allocated with 'frameBlock'
 *		to a frame without copying the data. The block
 *		must not be modified or deallocated later.
 *		
 *
 *   PARAMETERS:
 ************************************************************/

FRAME *fakeFrame(buffer, size, length)
BYTE_TYPE *buffer;
int size;       /* size given to frameBlock */
int length;     /* length of data in buffer */
{
   DATA_BLOCK *db;
   FRAME *frame;
   
   /* create the header block for the buffer */
   db = (DATA_BLOCK *)walloc(sizeof(DATA_BLOCK));
   db->next = NULL;
   db->size = size + PREFIX + SUFFIX;
   db->data = buffer - PREFIX;	/* use given block as data */
   db->n = length;
   db->i = PREFIX;
   db->j = PREFIX + length - 1;

   /* create the frame */
   frame = (FRAME *)wzalloc(sizeof(FRAME));
   frame->dbSize = db->size;
   frame->first = db;
   frame->length = length;

   return frame;
}
