/*	Default implementation of `morecore'. Only use this if you lack
 *	`__default_morecore' in your C library.
 *	Written By Richard W.M. Jones <rwmj@doc.ic.ac.uk>
 *
 * Further notes (21/8/95):
 *	Lots more people than I anticipated don't have `__default_morecore'
 *	so this file has caused a bit of trouble. So I'll explain here what
 *	this does so I don't have to tell 100s of people the same thing
 *	separately. Basically, `__default_morecore' is required by GNU
 *	malloc to grab large chunks of memory from the OS. There are two
 *	ways of doing this in modern Unix systems. The original way was
 *	to extend the data area by calling `sbrk'. However, what with DLLs
 *	and advanced memory mapping, this method has generally been super-
 *	ceeded by making direct calls to `mmap' which gives a lot more
 *	control over where in the address space the new memory is placed.
 *	Now, there are two versions of `__default_morecore' here. The first
 *	(for Linux, etc.) is the `mmap' version. I recommend you try
 *	this first. If it doesn't work, use the second version (for
 *	Suns, SVR-4, etc.). If you get a particular machine to work with
 *	one or other version, then can you send me patches to this file to
 *	support the project for other users.
 *
 * Even more notes (12/1/96):
 *	As suggested by Eberhard Mattes <mattes@
 *	azu.informatik.uni-stuttgart.de>, I have added HAVE_MMAP and
 *	HAVE_DEFAULT_MORECORE symbols at the top of the file, to make it easier
 *	to fix this.
 *   ->	If you have `__default_morecore' in your C library already, set
 *	HAVE_DEFAULT_MORECORE.
 *   ->	If you have `mmap' (most modern Unix machines do), set HAVE_MMAP.
 */

#if defined(linux) || defined(hpux)
#define HAVE_MMAP
#undef HAVE_DEFAULT_MORECORE
#endif

#if defined(sun) || defined (__svr4__) || defined (_AIX) || defined (__osf__) || defined (ultrix) || defined (__EMX__)
#undef HAVE_MMAP
#undef HAVE_DEFAULT_MORECORE
#endif

#if !defined(HAVE_DEFAULT_MORECORE)

#if defined(HAVE_MMAP)

/* Note: `sbrk' doesn't seem to work properly under Linux. For some reason
 * it returns ENOMEM more often than it should. I'm using mmap/munmap here
 * which is (supposedly) faster anyhow.
 */

#include <stddef.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/mman.h>
#include "getpagesize.h"

#define PERMISSIONS (PROT_READ | PROT_WRITE)
#define MAPPING_TYPE (MAP_PRIVATE | MAP_FIXED | MAP_ANONYMOUS)

void *
__default_morecore (ptrdiff_t incr)
{
  static void *address = NULL;

  if (address == NULL)
    {
      unsigned pagesize = getpagesize ();

      address = sbrk (0);
      if (address == NULL) return NULL;

      /* Address may not be aligned to a page. Make it so. `malloc' seems
       * to only ask for multiples of whole pages so it should remain
       * aligned after this.
       */
      if (((unsigned) address & (pagesize-1)) != 0)
	address += pagesize - ((unsigned) address & (pagesize-1));
    }
  if (incr == 0)
    return address;
  if (incr < 0)		/* Allow the heap to shrink by unmapping memory. */
    {
      address += incr;
      munmap (address, -incr);
      return address;
    }
  if (mmap (address, incr, PERMISSIONS, MAPPING_TYPE, -1, 0) != address)
    return NULL;
  address += incr;
  return address-incr;
}

#else /* Not HAVE_MMAP */

/* This is the sbrk version of __default_morecore for use on Suns
 * and other machines.
 */

#include <stddef.h>
#include <sys/types.h>
#include <unistd.h>

#if defined (sun) || defined (_AIX)
void *sbrk (ptrdiff_t);
#endif

void *
__default_morecore (ptrdiff_t incr)
{
  void *ptr = sbrk (incr);
  if (ptr == (void *)-1)
    return NULL;
  else
    return ptr;
}

#endif /* Not HAVE_MMAP */

#endif /* Not HAVE_DEFAULT_MORECORE */
