      SUBROUTINE BVPSOG(FCN,BC,IVPSOL,N,M,T,X,EPS,ITMAX,INFO,
     *NONLIN,IRW,RW,IIW,IW,II2W,I2W)
C*    Begin Prologue BVPSOG
      IMPLICIT DOUBLEPRECISION(S)
      EXTERNAL FCN,BC,IVPSOL
      INTEGER N,M
      DOUBLE PRECISION T(M)
      DOUBLE PRECISION X(N,M)
      DOUBLE PRECISION EPS
      INTEGER ITMAX
      INTEGER INFO
      INTEGER NONLIN,IRW
      DOUBLE PRECISION RW(IRW)
      INTEGER IIW
      INTEGER IW(IIW)
      INTEGER II2W
      INTEGER I2W(II2W)
C
C     ------------------------------------------------------------
C
C*  Title
C
C     (B)oundary (V)alue (P)roblem (So)lver for highly nonlinear
C     two point boundary value problems using a (G)lobal sparse linear
C     solver for the solution of the arising linear subproblems.
C
C*  Written by        P. Deuflhard, G.Bader, L. Weimann
C*  Purpose           Solution of nonlinear two-point boundary value
C                     problems.
C*  Method            Global Nonlinear two-point Boundary Value
C                     Problems solver (Multiple shooting approach)
C*  Category          I1b2a - Differential and integral equations
C                             Two point boundary value problems
C*  Keywords          Nonlinear boundary value problems, Multiple
C                     shooting, Newton methods
C*  Version           1.1
C*  Revision          January 1991
C*  Latest Change     January 1991
C*  Library           CodeLib
C*  Code              Fortran 77, Double Precision
C*  Environment       Standard Fortran 77 environment on PC's,
C                     workstations and hosts.
C*  Copyright     (c) Konrad Zuse Zentrum fuer
C                     Informationstechnik Berlin
C                     Heilbronner Str. 10, D-1000 Berlin 31
C                     phone 0049+30+89604-0,
C                     telefax 0049+30+89604-125
C*  Contact           Lutz Weimann
C                     ZIB, Numerical Software Development
C                     phone: 0049+30+89604-185 ;
C                     e-mail:
C                     RFC822 notation: weimann@sc.zib-berlin.de
C                     X.400: C=de;A=dbp;P=zib-berlin;OU=sc;S=Weimann
C
C*    References:
C
C     /1/ R.Bulirsch:
C         Die Mehrzielmethode zur numerischen Loesung von
C         nichtlinearen Randwertproblemen und Aufgaben der
C         optimalen Steuerung.
C         Carl-Cranz-Gesellschaft: Tech.Rep. (Oct.1971)
C
C     /2/ J.Stoer, R.Bulirsch:
C         Einfuehrung in die Numerische Mathematik II.
C         Berlin, Heidelberg, New York: Springer (1st Ed. 1973)
C
C     /3/ P.Deuflhard:
C         A Modified Newton Method for the Solution of
C         Ill-Conditioned Systems of Nonlinear Equations with
C         Application to Multiple Shooting.
C         Numer. Math. 22, 289-315 (1974)
C
C     /4/ P.Deuflhard:
C         Recent Advances in Multiple Shooting Techniques.
C         (Survey Article including further References)
C         In: I.Gladwell, D.K.Sayers (Ed.): Computational
C         Techniques for Ordinary Differential Equations.
C         Section 10, P.217-272.
C         London, New York: Academic Press (1980)
C
C     /5/ P.Deuflhard, G.Bader:
C         Multiple Shooting Techniques Revisited.
C         Univ. Heidelberg, SFB 123, Tech. Rep. 163 (1982)
C
C     /6/ P. Deuflhard:
C         Newton Techniques for Highly Nonlinear Problems -
C         Theory, Algorithms, Codes.
C         Academic press, to appear.
C
C  ---------------------------------------------------------------
C
C* Licence
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time.
C    In any case you should not deliver this code without a special
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C* Warranty
C    This code has been tested up to a certain level. Defects and
C    weaknesses, which may be included in the code, do not establish
C    any warranties by ZIB. ZIB does not take over any liabilities
C    which may follow from aquisition or application of this code.
C
C* Software status
C    This code is under partial care of ZIB and belongs to ZIB software
C    class 2.
C
C     ------------------------------------------------------------
C
C     External subroutines (to be supplied by the user)
C     =================================================
C
C       FCN(N,T,Y,DY)         Right-hand side of system of
C                             first-order differential equations
C         N                   Input: Number of first order ODE's
C         T                   Input: Actual position in the
C                             interval  A ,  B 
C         Y(N)                Input: Values at T
C         DY(N)               Output: Derivatives at T
C
C       BC(YA,YB,R)           Two-point boundary conditions at
C                             (A = T(1),  B = T(M))
C         YA(N)               Input: Values at A = T(1)
C         YB(N)               Input: Values at B = T(M)
C         R(N)                Output: Values of
C                             boundary conditions function
C
C       IVPSOL(N,FCN,T,Y,TEND,TOL,HMAX,H,KFLAG)
C                             Initial value problem (IVP)
C                             integrator
C         N                   Number of first-order ODE's
C         FCN                 Right-hand side of the ODE's system
C                             ( see above )
C         T                   Input: Starting point of integration
C                             T.LT.TEND
C                             Output: Achieved final point of
C                             integration
C         Y(N)                Input and Output: Values at T
C         TEND                Input: Prescribed final point of
C                             integration
C         TOL                 Input: Prescribed relative precision
C                             (>0)
C         HMAX                Input: Maximum permitted stepsize
C         H                   Input: Initial stepsize guess
C                             Output: Stepsize proposal for next
C                             integration step ( H.EQ.0 ,  if
C                             IVPSOL fails to proceed )
C         KFLAG               Input: Print parameter
C                             Output: Error flag ( KFLAG.LT.0
C                             indicates an error ) .
C                             For further details, see IVPSOL .
C
C     Input parameters (* marks inout parameters)
C     ===========================================
C
C       N                     Number of first-order ordinary
C                             differential equations.
C       M                     Number of Shooting nodes.
C                             =2    Single Shooting
C                             >2    Multiple Shooting
C       T(M)                  Single/Multiple Shooting Nodes
C                             ( T(1)= A ,  T(M)= B )
C     * X(N,M)                Start data for Newton iteration.
C       EPS                   Required relative precision of
C                             solution.
C       ITMAX                 Maximum permitted number of
C                             iteration steps.
C     * INFO                  Print parameter:
C                             -1    No print
C                              0    Print initial data, iterative
C                                   values of level functions,
C                                   solution data (or final data,
C                                   respectively)
C                             +1    Additionally print iterates
C                                   T(J),X(I,J),  I = 1,...,N ,  J
C                                   = 1,...,M
C       NONLIN                Boundary value problem
C                             classification by user:
C                             0     Linear boundary value problem.
C                             1     Nonlinear boundary value
C                                   problem. Good initial data
C                                   available.
C                             2     Highly nonlinear boundary
C                                   value problem. Only bad
C                                   initial data available. Small
C                                   initial damping factor in
C                                   Gauss Newton method.
C
C       NRW                   Dimension of real workspace RW
C                             NRW.GE.N*N*(M+2)+N*(12*M+4)+LICN+M-1
C       RW(NRW)               Real workspace
C
C       NIW                   Dimension of integer workspace IW
C                             NIW.GE.2*N*N+M*N+3*N+8*M*N
C       IW(NIW)               Integer workspace
C
C       NI2W                  Dimension of short integer workspace
C                             I2W
C                             NI2W.GE.5*M*N+2*NZ+LICN+LIRN
C       I2W(NI2W)             Short integer workspace
C                             (In the actual implementation of the
C                              same type is IW)
C                             NZ = N*N*(M+1)+N*(M-1)
C                             LICN = 2*NZ
C                             LIRN = DMIN1(DMAX1(1.5*NZ,NZ+4*M*N),
C                             LICN)
C
C       Output parameters:
C       ==================
C
C       X(N,M)                Solution data ( or final data,
C                             respectively )
C       INFO                  Information output parameter
C                              >0   Number of iterations performed
C                                   to obtain the solution
C                              <0   BVPSOG termination
C                              -1   Gaussian elimination failed by
C                                   singular Jacobian
C                              -2   Iteration stops after ITMAX
C                                   iteration steps ( as indicated
C                                   by input parameter ITMAX )
C                              -3   Integrator failed to complete
C                                   the trajectory
C                              -4   Newton method failed to
C                                   converge
C                              -5   Given initial values
C                                   inconsistent with separable
C                                   linear boundary conditions
C                              -6   Multiple Shooting condition
C                                   too bad - insert new nodes
C                              -7   Reliable relative accuracy
C                                   greater than 1.0D-2
C                              -9   Too small storage for sparse
C                                   linear system solver
C                             -10   Real or integer work-space
C                                   exhausted
C
C     ------------------------------------------------------------
C
C*    End Prologue
C:    SMALL = squareroot of "smallest positive machine number
C     divided by relative machine precision"
      DOUBLE PRECISION SMALL
      PARAMETER (SMALL=4.94D-32)
      INTEGER M1,NM,NM1,NN,NRW,NIW,NI2W
      DOUBLE PRECISION RELDIF,TOL,XTHR
C:    Begin
C     ------------------------------------------------------------
C     1 Internal parameters
C     Standard values fixed below
C     Scaling threshold
      XTHR = SMALL
C     Prescribed relative precision for numerical integration
      TOL = EPS*1.0D-2
C     Prescribed relative deviation for numerical differentiation
      RELDIF = DSQRT(TOL)
      IF(INFO.GE.0)THEN
C       Print BVPSOG heading lines
1       FORMAT('1',2X,'B V P S O G',2X,5('*'),2X,'V e r s i o n',2
     *  X,'1 . 0',1X,3('*'),//,1X,'Newton',1('-'),'Method ','for ',
     *  'the ','solution ','of ','boundary ','value ','problems',/
     *  /)
        WRITE(6,1)
      ENDIF
C     Initial preparations
      M1 = M-1
      NN = N*N
      NM = N*M
      NM1 = N*M1
      NMX8 = 8*NM
      NZ = N*N*(M+1)+N*(M-1)
      LICNQ = 2*NZ
      LIRNQ = MAX0(3*NZ/2,NZ+4*M*N)
      LIRNQ = MIN0(LIRNQ,LICNQ)
      V = DBLE(LIRNQ)/DBLE(LICNQ)
      NRW = N*N*(M+2)+12*M*N+4*N+M-1
      NI2W = 2*NZ+5*M*N
      L1 = IRW-NRW
      L2 = IDINT(DBLE(II2W-NI2W)/(V+1.0D0))
      LICN = MIN0(L1,L2)
      LIRN = IDINT(V*DBLE(LICN))
      LISNQ = LICNQ+LIRNQ
      NKEEP = NMX8
      MINI2W = NI2W + LISNQ
C:    WorkSpace: I2W
        L4=1
        L5=L4+LIRN
        L6=L5+LICN
        L7=L6+NZ
        L8=L7+NZ
        L9=L8+NKEEP
        NI2W=L9-1
C.    End WorkSpace at NI2W
C:    WorkSpace: IW
        L10=1
        L11=L10+N
        L12=L11+N
        L13=L12+N
        L14=L13+N*N
        L15=L14+N*N
        L16=L15+NMX8
        NIW=L16-1
C.    End WorkSpace at NIW
C:    WorkSpace: RW
        L17=1
        L18=L17+N*N*M1
        L19=L18+N*N
        L20=L19+N*N
        L21=L20+LICN
        L22=L21+NM
        L23=L22+NM
        L24=L23+NM
        L25=L24+NM
        L26=L25+NM
        L27=L26+NM
        L28=L27+NM1
        L29=L28+NM1
        L30=L29+NM1
        L31=L30+NM1
        L32=L31+N
        L33=L32+N
        L34=L33+N
        L35=L34+N
        L36=L35+NM
        L37=L36+NM
        L38=L37+N
        L39=L38+N
        L40=L39+N
        L41=L40+N
        L42=L41+M1
        NRW=L42-1
C.    End WorkSpace at NRW
C     ------------------------------------------------------------
C     2 Check for sufficient real/integer workspace
      IF (INFO.GE.0) THEN
2       FORMAT('0','Minimal ','required ','work-space ',':',/,'0',
     *  'Real          ','array ','RW( ',I4,')',/,'0',
     *  'Integer       ','array ','IW( ',I4,')',/,'0',
     *  'Short Integer ','array ','I2W(',I4,')')
        WRITE(6,2)NRW,NIW,MINI2W
      ENDIF
      IF(NRW.LE.IRW.AND.NIW.LE.IIW.AND.MINI2W.LE.II2W)THEN
        CALL BVPG(FCN,BC,IVPSOL,N,M,M1,NM,NM1,NMX8,NZ,LICN,LIRN,
     *  LISNQ,NKEEP,T,X,EPS,TOL,RELDIF,NONLIN,ITMAX,INFO,XTHR,IW(
     *  L10),IW(L11),IW(L12),IW(L13),IW(L14),IW(L15),RW(L17),RW(
     *  L18),RW(L19),RW(L20),RW(L21),RW(L22),RW(L23),RW(L24),RW(
     *  L25),RW(L26),RW(L27),RW(L28),RW(L29),RW(L30),RW(L31),RW(
     *  L32),RW(L33),RW(L34),RW(L35),RW(L36),RW(L37),RW(L38),RW(
     *  L39),RW(L40),RW(L41),I2W(L6),I2W(L7),I2W(L5),I2W(L4),I2W(
     *  L8))
      ELSE
C       Fail exit work-space exhausted
        IF(INFO.GE.0.AND.NRW.GT.IRW)THEN
3         FORMAT('0','Error: ','real          ','work ','- ',
     *    'space ','exhausted',/)
          WRITE(6,3)
        ENDIF
        IF(INFO.GE.0.AND.NIW.GT.IIW)THEN
4         FORMAT('0','Error: ','integer       ','work ','- ',
     *    'space ','exhausted',/)
          WRITE(6,4)
        ENDIF
        IF(INFO.GE.0.AND.NI2W.GT.II2W)THEN
5         FORMAT('0','Error: ','short integer ','work ','- ',
     *    'space ','exhausted',/)
          WRITE(6,5)
        ENDIF
        INFO = -10
      ENDIF
      RETURN
C     End of driver routine BVPSOG
      END
      SUBROUTINE BVPG(FCN,BC,IVPSOL,N,M,M1,NM,NM1,NMX8,NZ,LICN,
     *LIRN,LISNQ,NKEEP,T,X,EPS,TOL,RELDIF,NONLIN,ITMAX,INFO,XTHR,
     *IROW,ICOLA,ICOLB,IA,IB,IW,G,A,B,E,WO,DX,DXQ,DXQA,XA,XW,XU,HH,
     *DHH,HHA,DE,R,DR,RA,U,DU,X1,XM,T1,T2,RF,IVECT,JVECT,ICN,IRN,
     *IKEEP)
      IMPLICIT DOUBLEPRECISION(S)
      EXTERNAL FCN,BC,IVPSOL
      INTEGER N,M,M1,NM,NM1,NMX8,LISNQ,NKEEP
      DOUBLE PRECISION T(M),X(NM)
      DOUBLE PRECISION EPS
      DOUBLE PRECISION TOL,RELDIF
      INTEGER NONLIN,ITMAX
      INTEGER INFO
      DOUBLE PRECISION XTHR
      INTEGER IRN(LIRN),ICN(LICN),IVECT(NZ),JVECT(NZ),IKEEP(
     *NKEEP)
      INTEGER IROW(N),ICOLA(N),ICOLB(N)
      INTEGER IA(N,N),IB(N,N)
      INTEGER IW(NMX8)
      DOUBLE PRECISION G(N,N,M1)
      DOUBLE PRECISION A(N,N),B(N,N)
      DOUBLE PRECISION E(LICN),WO(NM)
      DOUBLE PRECISION DX(NM),DXQ(NM),DXQA(NM),XA(NM),XW(NM),XU(
     *NM1),HH(NM1),DHH(NM1),HHA(NM1),DE(N),R(N),DR(N),RA(N),U(NM),
     *DU(NM),X1(N),XM(N),T1(N),T2(N),RF(M1)
C
C     Addtional dimensional integer variables:
C     ========================================
C
C       M1                M-1
C       NM                N*M
C       NM1               N*(M-1)
C       NMX8              8*N*M
C       LIRN,LICN,NKEEP,NZ
C                         See driver routine BVPSOG
C
C     Internal real arrays (workspace) :
C     ==================================
C
C       G(N,N,M1)        (N,N) -Wronskian Matrices G(1),...,G(M-1)
C                         .
C       A(N,N)            Wronskian Matrix on left boundary
C                         dBC/dX(X(1,...,N),T(1)).
C       B(N,N)            Wronskian Matrix on right boundary
C                         dBC/dX(X((N-1)*M+1,...,N*M),T(M)).
C       E(LICN)           Holds the values of the Jacobian stored
C                         in sparse mode.
C       DE(N)             Holds row scaling factors for the
C                         boundary conditions part of the Jacobian
C                         matrix.
C       DHH(NM1)          Holds the continuity residuals computed
C                         in BGSOLI .
C       DR(N)             Workspace for subroutine BGSOLI to hold
C                         the boundary residual
C                         BC(DXQ(1,...,N),DXQ((M-1)*N+1,...,M*N))+
C                         (A*DXQ(1,...,N))+B*DXQ((M-1)*N+1,...,M*N)
C                         .
C       DU(NM)            Used by BGSOLI . Gets the total residual
C                         for the current iterate.
C       DX(NM)            Actual newton correction.
C       DXQ(NM)           Simplified Newton correction J(k-1)*X(k)
C                         with the Jacobian J(k) and the iterate
C                         vector X(k) at the k-th iterate.
C       DXQA(NM)          Previous simplified Newton correction
C                         J(k-2)*X(k-1).
C       HH(NM1)           Elements (J-1)*N+1 to J*N are holding
C                         the values
C                         Y(T(J+1),X((J-1)*N+1,...,J*N))-X(J*N+1,
C                         ...,(J+1)*N)
C                         ( with the trajectory Y in
C                          T(J),T(J+1) , J = 1,...,M-1 ).
C       HHA(NM1)          Holds the previous value of HH .
C       R(N)              Value of the boundary condition function
C                         BC for the current iterate.
C       RA(N)             Previous values of R .
C       RF(M1)            Used by BGSOLI . Gets the norms of the
C                         Wronskian matrices.
C       T1(N)             Workspace used for miscellaneous
C                         purposes temporarely.
C       T2(N)             Workspace used for miscellaneous
C                         purposes temporarely.
C       U(NM)             Gets the right hand side of the linear
C                         system to be solved in each iteration
C                         step. Used in BGSOLI .
C       WO(NM)            Workspace needed for sparse solver. Must
C                         not be altered outside the sparse packet
C                         routines.
C       XA(NM)            Previous Newton iterate.
C       XU(NM1)           Elements (J-1)*N+1 to J*N are holding
C                         the values Y(T(J+1),X((J-1)*N+1,...,J*N))
C                         of the trajectory in the interval  T(J),
C                         T(J+1) , (for J = 1,...,M-1 ).
C       XW(NM)            Scaling factors for iteration vector.
C       X1(N)             Components of the iteration vector
C                         corresponding to the left boundary
C                         A = T(1).
C       XM(N)             Components of the iteration vector
C                         corresponding to the right boundary
C                         B = T(M).
C
C     Internal integer arrays (workspace)
C     ===================================
C
C       IROW(N)           Row permutations of boundary derivative
C                         matrices A and B .
C       ICOLA(N)          Column permutations of matrix A
C                         (left boundary).
C       ICOLB(N)          Column permutations of matrix B
C                         (right boundary).
C       IA(N,N)           Reflects the sparse structure of matrix
C                         A by values 0, 1.
C       IB(N,N)           Reflects the sparse structure of matrix
C                         B by values 0, 1.
C       IW(NMX8)          Workspace needed for sparse solver
C                         package.
C
C     Internal short integer arrays (workspace)
C     =========================================
C
C       IRN(LIRN)         Workspace for MA28/MA30 sparse package.
C                         On Input to routine MA28A, it must hold
C                         the row indices of the sparse matrix.
C       ICN(LICN)         Workspace for MA28/MA30 sparse package.
C                         On Input to routine MA28A, it must hold
C                         the column indices of the sparse matrix.
C       IVECT(NZ)         Input to routine MA28B: must hold the
C                         row indices of the sparse matrix.
C       JVECT(NZ)         Input to routine MA28B: must hold the
C                         column indices of the sparse matrix.
C       IKEEP(NKEEP)      Workspace array for MA28 sparse package.
C                         To be preserved across the calls of the
C                         routines MA28A,MA28B,MA28C .
C
C     Internal real variables:
C     ========================
C
C       COND              Gets the condition of the Jacobian
C                         matrix computed by BGSOLI .
C       CORR              Gets the 1-norm of the residual DU .
C                         Computed by BGSOLI .
C       CONV              Scaled maximum norm of DXQ computed by
C                         subroutine BGLVLS . Used for convergence
C                         test.
C       CONVA             Holds the previous value of CONV .
C       EPSMIN            Smallest reasonable permitted accuracy
C                         EPS that can be prescribed by the user.
C       FC                Actual Gauss Newton iteration damping
C                         factor.
C       FCA               Previous Gauss Newton iteration damping
C                         factor.
C       FCDNM             Used to compute the denominator of the
C                         damping factor FC during computation of
C                         it's predictor, corrector and
C                         aposteriori estimate (in the case of
C                         performing a Rank1 update) .
C       FCH               Temporarely used for storing the new FC
C                         when computing aposteriori estimate.
C       FCMIN             Minimum permitted relaxation factor. If
C                         FC becomes smaller than this value, one
C                         of the following may occur:
C                         a.    Recomputation of the sensitivity
C                               matrix by means of difference
C                               approximation (instead of Rank1
C                               update), if Rank1 - update
C                               previously was used
C                         b.    Rank reduction of sensitivity
C                               matrix E ,  if difference
C                               approximation was used previously
C                               and Rank(E).NE.0
C                         c.    Fail exit otherwise
C       FCMIN2            FCMIN**2 . Used for FC-predictor
C                         computation.
C       FCNUM             Gets the numerator of the aposteriori
C                         estimate of FC .
C       FCNUMP            Gets the numerator of the predictor
C                         computation of FC .
C       FCNUMK            Gets the numerator of the corrector
C                         computation of FC .
C       H                 Actual integrator stepsize.
C       HMAX              Maximum permitted integrator stepsize.
C                         Set to the length of the integration
C                         interval, e.g. the distance of the
C                         effected Shooting points.
C       HSAVE             Stepsize saved across the call of the
C                         integrator.
C       HSTART            Start stepsize for integration used by
C                         subroutines BGFCNI and BGDERG .
C       MUE               Temporary value used during computation
C                         of damping factors predictor.
C       REDH              Multi purpose reduction factor. (???)
C       RELDIF            Relative deviation for numerical
C                         differentation.
C       SIGMA             Decision parameter for Jacobian Rank1
C                         updates (SIGMA.GT.1) . Rank1 updates are
C                         inhibited, if SIGMA.GT.1/FCMIN is set.
C       SKAP              Used to compute and print out the
C                         incompatibility factor of the nonlinear
C                         boundary value (e.g. least squares)
C                         problem.
C       SUMF              Standard level of the current iterate,
C                         e.g. Norm2(F(X))**2
C                         with the nonlinear model function F on
C                         which Newton iteration is performed,
C                         arising from the Multiple Shooting
C                         approach.
C       SUMX              Natural level of the current iterate,
C                         e.g. Norm2(DX)
C                         with the Newton correcture DX
C                         (see above).
C       SUMXA             Natural level of the previous iterate.
C       TFAIL             Used to get and print out in case of an
C                         integrator failure the last reached T
C                         value as a proposal for insertion of a
C                         new Shooting point.
C       TOL               Prescribed relative precision for
C                         numerical integration.
C       TOLH              Temporary used for computation of TOL
C                         (may be obmitted|).
C       TOLMIN            Lower bound value for TOL .
C       XTHR              Threshold for scaling.
C       TJ                Used by BGFCNI to hold T(J).
C       TJ1               Used by BGFCNI to hold T(J+1).
C       EPH               Gets TOL*SIGDEL by BGSOLI . If EPH.GT.
C                         REDH ,  termination occurs, since
C                         Multiple Shooting condition is too bad.
C       SIGDEL            Used by BGSOLI to compute the required
C                         integrator accuracy from the multiple
C                         shooting condition.
C       SIGDLH            Used by BGSOLI temporary during
C                         determination of SIGDEL .
C
C     Internal integer variables
C     ==========================
C
C       IAF               Indicates, if sparse structure of
C                         Jacobian matrix must be reordered by
C                         sparse solver or not:
C                         0     Not necessary, can call MA28B to
C                               decompose Jacobian
C                         1     Must be done, so use MA28A .
C       IC                Permutated index. Used by BGSOLI .
C       ICA               Temporarely used during search for
C                         separable linear boundary conditions at
C                         T(1).
C       ICB               Temporarely used during search for
C                         separable linear boundary conditions at
C                         T(M).
C       IH                Temporarely used during search for
C                         separable linear boundary conditions at
C                         T(1)and T(M).
C       INZ               Count of nonzero Jacobian matrix
C                         elements.
C       IR                Temporary storage for a row permutation
C                         index.
C       IRANK             Rank of Jacobian matrix E of current
C                         iteration step estimated by sparse
C                         solver package (Common block variable).
C       IS                Additional DO loop index.
C       ISUM              Used for determination of sparse
C                         structure of matrices A and B as
C                         nonzeros location counter.
C       ITER              Iteration count.
C       JJ                Used as "reverse DO loop" index:
C                         JJ = IUPB-J in a loop like DO J = 1,IUPB
C                         ...
C       JRED              Damping factor reduction count during an
C                         iterate.
C       KC                Temporary storage for a column
C                         permutation index.
C       KFLAG             Gets the subintervall number of the
C                         failure from subroutine BGDERG ,  if the
C                         integrator failed.
C       KOUNT             Trajectory evaluations count.
C       KPRINT            Print parameter - copy of input
C                         parameter INFO .
C       LEVEL             Flow control parameter needed by
C                         subroutine BGSOLI :
C                         0     indicates computation of Newton
C                               correcture,
C                         1     indicates computation of
C                               simplified Newton correcture
C                               (after computation of the
C                               preliminary new iterate)
C       NA                Number of separable boundary conditions
C                         at T(1): N-NAQ
C       NAQ               Number of not separable boundary
C                         conditions at T(1)
C       NB                Number of separable boundary conditions
C                         at T(M)
C       NBQ               Number of not separable boundary
C                         conditions at T(M): N-NB
C       NEW               Count of subsequent performed Rank1
C                         (Broyden) updates.
C       NM                Number of rows and columns of the
C                         Jacobian matrix part to be decomposed by
C                         the sparse solver.
C       NRS               N-(NA+NB)
C:    End Parameter
C:    EPMACH = relative machine precision
      DOUBLE PRECISION EPMACH
      PARAMETER (EPMACH=2.23D-16)
C:    SMALL = squareroot of "smallest positive machine number
C     divided by relative machine precision"
      DOUBLE PRECISION SMALL
      PARAMETER (SMALL=4.94D-32)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      DOUBLE PRECISION REDH
      PARAMETER (REDH=1.0D-2)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION TWO
      PARAMETER (TWO=2.0D0)
      DOUBLE PRECISION EIGHT
      PARAMETER (EIGHT=8.0D0)
      DOUBLE PRECISION TEN
      PARAMETER (TEN=1.0D1)
      DOUBLE PRECISION FCMIN
      PARAMETER (FCMIN=1.0D-2)
      DOUBLE PRECISION FCMIN2
      PARAMETER (FCMIN2=1.0D-4)
      DOUBLE PRECISION FCNLIN
      PARAMETER (FCNLIN=1.0D-2)
      DOUBLE PRECISION SIGMA
      PARAMETER (SIGMA=2.0D0)
      INTEGER I,ICA,ICB,ICNCP,IH,IR,IRNCP,IRANK,IS,ISUM,ITER,
     *J,JRED,J0,J1,K,KFLAG,KOUNT,KPRINT,
     *L,LEVEL,MINIRN,MINICN,NB,NDIM,NAQ,NEW
      DOUBLE PRECISION COND,CORR,CONV,CONVA,EPH,EPSMIN,
     *EPX1H,EPSQ,FC,FCA,FCDNM,FCH,FCNMP2,FCNUM,FCNUMK,FCNUMP,
     *HSTART,MUE,RESID,RMIN,S,SIGDEL,SIGDLH,SUMF,
     *SUMX,SUMXA,TFAIL,TH,TOLH,TOLMIN,UQ
      LOGICAL ABORT1,ABORT2,DIFAPP,FCOMPT,GROW,IBLOCK,IVFAIL,
     *JACRFR,JACRST,NEXT
      COMMON /MA28ED/ LP,MP,IBLOCK,GROW
      COMMON /MA28FD/ EPSQ,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     *IRANK,ABORT1,ABORT2
      INTEGER L1,L2
      DOUBLE PRECISION S1
C:    Begin
C:    Begin of Segment BVPSOG.Body
C       ----------------------------------------------------------
C       1 Initialization
C       ----------------------------------------------------------
C       1.1 Internal parameters
C       Standard values fixed below
C       Minimum relative precision of integrator ( to be adapted )
        TOLMIN = EPMACH*TEN*TEN
C       Maximum permitted number of iterative refinements sweeps
C       ----------------------------------------------------------
C       1.1.1 Common parameters
C       Starting value of relaxation factor (FCMIN.LE.FC.LE.1.0)
        IF(NONLIN.LE.1)THEN
C         for linear or mildly nonlinear problems
          FC = ONE
        ELSE
C         for highly nonlinear problems
          FC = FCNLIN
        ENDIF
C       Minimum reasonable value for EPS
        EPSMIN = DSQRT(TEN*EPMACH)
        IF(EPS.LT.EPSMIN) EPS = EPSMIN
C       ----------------------------------------------------------
C       1.2 Initial preparations
        IF(FC.LT.FCMIN) FC = FCMIN
        IF(FC.GT.ONE) FC = ONE
        KPRINT = INFO
        ITER = 0
        KOUNT = 0
        INFO = -1000
        FCA = FC
        CONV = ZERO
        JACRFR = .FALSE.
        JACRST = .FALSE.
C:      Begin SetVec.Vec
        DO 6 L1=1,NM
          XA(L1)=X(L1)
6       CONTINUE
C.      End SetVec.Vec
        IF(TOL.LE.ZERO) TOL = EPS/TEN
        IF(TOL.LT.TOLMIN) TOL = TOLMIN
        DIFAPP = .TRUE.
        HSTART =(T(2)-T(1))*REDH
        LP = 0
        MP = 0
        IBLOCK = .FALSE.
        EPSQ = 0.1D0
        UQ = 0.1D0
        SIGDLH = ZERO
        SUMF = ONE
C:      Mat IA = Scalar (Rows 1,N ; Cols 1,N)
        L1 = 0
        DO 7 L2=1,N
        DO 7 L43=1,N
          IA(L2,L43)=L1
7       CONTINUE
C.      End SetIntMat.S
C:      Mat IB = Scalar (Rows 1,N ; Cols 1,N)
        L1 = 0
        DO 8 L2=1,N
        DO 8 L43=1,N
          IB(L2,L43)=L1
8       CONTINUE
C.      End SetIntMat.S
C:      CubeMat G (layer 1)= Scalar (Rows 1,N ; Cols 1,N)
        S1 = ZERO
        DO 9 L1=1,N
        DO 9 L2=1,N
          G(L1,L2,1)=S1
9       CONTINUE
C.      End SetCubeMat.S
        IF(KPRINT.GE.0)THEN
C         Print Start vector data, predescribed precision and max
C         iteration steps
10        FORMAT('0','Initial ','data',//)
          WRITE(6,10)
          DO 11 J=1,M
12          FORMAT(D13.5,2X)
            WRITE(6,12)T(J)
13          FORMAT((14X,3(D20.10,1X)))
            WRITE(6,13)(X(L1),L1=(J-1)*N+1,J*N)
11        CONTINUE
14        FORMAT('0','N ','=',I2,2X,'M ','=',I2,/,'0',
     *    'Prescribed ','relative ','precision',D10.2,2X,/,'0',
     *    'Maximum ','permitted ','number ','of ','iteration ',
     *    'steps',1X,I3,//,'1')
          WRITE(6,14)N,M,EPS,ITMAX
          IF(KPRINT.EQ.0)THEN
15          FORMAT('0',1X,66('*'))
            WRITE(6,15)
16          FORMAT('0',4X,'It',7X,'Levelf',10X,'Levelx',8X,
     *      'Rel.Fc.')
            WRITE(6,16)
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       1.3 Startup step
C       ----------------------------------------------------------
C       1.3.1 Computation of the residual vector
        CALL BGFCNI(IVPSOL,FCN,BC,N,M,NM,NM1,ITER,KPRINT,HSTART,
     *  FCMIN,T,X,X1,XM,T1,XU,HH,R,TOL,FC,FCOMPT,IVFAIL,KFLAG,
     *  KOUNT,INFO)
C
C       Main iteration loop
C       ===================
C
C:      While (expression)
17      IF(INFO.EQ.-1000)THEN
C:          Begin of Segment BVPSOG.Core
C             ----------------------------------------------------
C             2 Startup of iteration step
              IF(.NOT.(JACRFR.OR.JACRST))THEN
                LEVEL = 0
C               --------------------------------------------------
C               2.1 Scaling of variables X(NM)
                CALL BGSCLE(N,M,NM,NM1,X,XU,XW,XTHR)
                IF(ITER.NE.0)THEN
C:                Begin SetVec.Vec
                  DO 18 L1=1,NM
                    DXQA(L1)=DXQ(L1)
18                CONTINUE
C.                End SetVec.Vec
C:                FCNUM = Sum of Formula Elements (for 1,NM)
                  FCNUM = 0.0D0
                  DO 19 L1=1,NM
                    FCNUM=FCNUM+((DX(L1)/XW(L1))**2)
19                CONTINUE
C.                End MakeSum.Comp
C:                FCNMP2 = Sum of Formula Elements (for 1,NM)
                  FCNMP2 = 0.0D0
                  DO 20 L1=1,NM
                    FCNMP2=FCNMP2+((DXQ(L1)/XW(L1))**2)
20                CONTINUE
C.                End MakeSum.Comp
                  FCNUMP = FCNUM*FCNMP2
                ENDIF
                IF(ITER.NE.0)THEN
                  TH = FC-ONE
C:                FCDNM = Sum of Formula Elements (for 1,NM)
                  FCDNM = 0.0D0
                  DO 21 L1=1,NM
                    FCDNM=FCDNM+(((DXQ(L1)+TH*DX(L1))/XW(L1))**2)
21                CONTINUE
C.                End MakeSum.Comp
                  FCH = DSQRT(FCNUM/FCDNM)*FC*FC*HALF
C                 ------------------------------------------------
C                 2.1.1 Decision criterion for Jacobian updating
C                       technique:
C                       DIFAPP.EQ..TRUE. numerical
C                       differentiation,
C                       DIFAPP.EQ..FALSE. rank1 updating
                  DIFAPP = FC.LT.FCA.AND.NEW.GT.0.OR.FCH.LT.FC*
     *            SIGMA
                  FCA = FC
                  IF(NONLIN.GT.0) FC = DMIN1(FCH,ONE)
                ENDIF
C               --------------------------------------------------
C               2.2 Difference approximation of jacobian matrix A
C                   ( If Difapp.EQ..TRUE. ) or
C                   Rank-1 update of jacobian matrix A ( If Difapp
C                   .EQ..FALSE. )
                CALL BGDERA(BC,N,M,NM,XW,X1,XM,R,T2,A,B,RELDIF)
C               --------------------------------------------------
C               2.3 Determination of sparse structure of matrices
C                   A and B
                IAF = 0
                DO 22 I=1,N
                  S = ZERO
                  DO 23 K=1,N
                    TH = DABS(A(I,K))*XW(K)
                    S = S+TH
                    TH = DABS(B(I,K))*XW(K+NM1)
                    S = S+TH
23                CONTINUE
                  IF(S.LT.XTHR) S = XTHR
                  DE(I)=ONE/S
                  DO 24 K=1,N
                    IF(IA(I,K).LE.0)THEN
                      IF(A(I,K).NE.ZERO)THEN
                        IA(I,K)=1
                        IAF = 1
                      ENDIF
                    ENDIF
                    IF(IB(I,K).LE.0)THEN
                      IF(B(I,K).NE.ZERO)THEN
                        IB(I,K)=1
                        IAF = 1
                      ENDIF
                    ENDIF
24                CONTINUE
22              CONTINUE
                IF(IAF.NE.0)THEN
C                 ------------------------------------------------
C                 2.3.1 Determination of row and column
C                       permutation vectors
                  DO 25 I=1,N
                    ICOLA(I)=I
                    ICOLB(I)=I
                    IROW(I)=I
25                CONTINUE
C                 ------------------------------------------------
C                 2.3.2 Search for separable linear boundary
C                       conditions at T(1)
                  NAQ = N
                  DO 26 I=1,N
                      DO 27 K=1,N
                        IF(IB(I,K).NE.0) GOTO 9996
27                    CONTINUE
                      ISUM = 0
                      DO 28 K=1,N
                        IF(IA(I,K).NE.0)THEN
                          ISUM = ISUM+1
                          ICA = K
                        ENDIF
28                    CONTINUE
                      IF(ISUM.LE.1)THEN
                        DO 29 IS=1,N
                          IH = ICOLA(IS)
                          IF(IH.EQ.ICA) ICOLA(IS)=ICOLA(NAQ)
                          IH = IROW(IS)
                          IF(IH.EQ.I) IROW(IS)=IROW(NAQ)
29                      CONTINUE
                        ICOLA(NAQ)=ICA
                        IROW(NAQ)=I
                        NAQ = NAQ-1
                        IF(DABS(R(I)).GT.TEN*EPMACH*DABS(X(ICA)))
     *                  THEN
                          INFO = -5
                          GOTO 9998
                        ENDIF
                      ENDIF
9996                CONTINUE
26                CONTINUE
                  IF(KPRINT.GE.0.AND.NAQ.EQ.0)THEN
30                  FORMAT('0','Warning: ','attempt ','to ',
     *              'solve ','initial ','value ','problem')
                    WRITE(6,30)
                  ENDIF
                  NA = N-NAQ
C                 ------------------------------------------------
C                 2.3.3 Search for separable linear boundary
C                       conditions at T(M)
                  NB = 0
                ENDIF
                IF(IAF.NE.0.AND.NAQ.NE.0)THEN
                  DO 31 I=1,NAQ
                      IR = IROW(I)
                      DO 32 K=1,N
                        IF(IA(IR,K).NE.0) GOTO 9995
32                    CONTINUE
                      ISUM = 0
                      DO 33 K=1,N
                        IF(IB(IR,K).NE.0)THEN
                          ISUM = ISUM+1
                          ICB = K
                        ENDIF
33                    CONTINUE
                      IF(ISUM.LE.1)THEN
                        NB = NB+1
                        DO 34 IS=1,N
                          IH = ICOLB(IS)
                          IF(IH.EQ.ICB) ICOLB(IS)=ICOLB(NB)
34                      CONTINUE
                        ICOLB(NB)=ICB
                        IROW(I)=IROW(NB)
                        IROW(NB)=IR
                        IF(DABS(R(IR)).GT.TEN*EPMACH*DABS(X(ICB+
     *                  NM1)))THEN
                          INFO = -5
                          GOTO 9998
                        ENDIF
                      ENDIF
9995                CONTINUE
31                CONTINUE
                  IF(KPRINT.GE.0.AND.NB.EQ.N)THEN
35                  FORMAT('0','Warning: ','attempt ','to ',
     *              'solve ','initial ','value ','problem')
                    WRITE(6,35)
                  ENDIF
                  NBQ = N-NB
C                 ------------------------------------------------
C                 2.3.4 Count non-zeroes in jacobian and store
C                       their locations
                  NDIM = NM-NA-NB
                  NRS = N-NA-NB
                  INZ = 0
                  DO 36 I=1,N
                      IF(NAQ.NE.0)THEN
                        DO 37 K=1,NAQ
                          INZ = INZ+1
                          IVECT(INZ)=I
                          JVECT(INZ)=K
37                      CONTINUE
                      ENDIF
                      II = I
                      IF(M.EQ.2)THEN
                        II = 0
                        IF(NBQ.EQ.0) GOTO 9994
                        DO 38 L=1,NBQ
                          IF(ICOLB(NB+L).EQ.I) II = L
38                      CONTINUE
                        IF(II.EQ.0) GOTO 9994
                      ENDIF
                      INZ = INZ+1
                      IVECT(INZ)=I
                      JVECT(INZ)=NAQ+II
9994                CONTINUE
36                CONTINUE
                  IF(M.NE.2)THEN
                    DO 39 J=2,M1
                      DO 40 I=1,N
                          DO 41 K=1,N
                            INZ = INZ+1
                            IVECT(INZ)=(J-1)*N+I
                            JVECT(INZ)=(J-1)*N+K-NA
41                        CONTINUE
                          II = I
                          IF(J.EQ.M1)THEN
                            II = 0
                            IF(NBQ.EQ.0) GOTO 9993
                            DO 42 L=1,NBQ
                              IF(ICOLB(NB+L).EQ.I) II = L
42                          CONTINUE
                            IF(II.EQ.0) GOTO 9993
                          ENDIF
                          INZ = INZ+1
                          IVECT(INZ)=(J-1)*N+I
                          JVECT(INZ)=J*N+II-NA
9993                    CONTINUE
40                    CONTINUE
39                  CONTINUE
                  ENDIF
                  IF(NRS.NE.0)THEN
                    DO 43 I=1,NRS
                      II = I+NB
                      IF(NAQ.NE.0)THEN
                        DO 44 K=1,NAQ
                          IF(IA(IROW(II),ICOLA(K)).NE.0)THEN
                            INZ = INZ+1
                            IVECT(INZ)=NM1+I
                            JVECT(INZ)=K
                          ENDIF
44                      CONTINUE
                      ENDIF
                      IF(NBQ.NE.0)THEN
                        DO 45 KK=1,NBQ
                          K = KK+NB
                          IF(IB(IROW(II),ICOLB(K)).NE.0)THEN
                            INZ = INZ+1
                            IVECT(INZ)=NM1+I
                            JVECT(INZ)=NM1+KK-NA
                          ENDIF
45                      CONTINUE
                      ENDIF
43                  CONTINUE
                  ENDIF
                ENDIF
              ENDIF
              JACRFR = .FALSE.
              IF(.NOT.JACRST)THEN
                IF(DIFAPP)THEN
                  NEW = 0
                  KFLAG = 0
                  CALL BGDERG(FCN,N,NAQ,M,M1,NM,NM1,T,X,XU,XW,T2,
     *            TFAIL,G,ICOLA,IVPSOL,HSTART,TOL,RELDIF,KFLAG)
                  IF(KFLAG.LT.0)THEN
                    INFO = -3
                    GOTO 9998
                  ENDIF
                  IF(M.GT.2) KOUNT = KOUNT+N
                  IF(M.EQ.2) KOUNT = KOUNT+NAQ
                ELSE
                  NEW = NEW+1
                  CALL BGRK1G(N,M,M1,NM,NM1,XW,DX,HH,HHA,T1,G,FCA)
                ENDIF
              ENDIF
              JACRST = .FALSE.
C             ----------------------------------------------------
C             2.3.5 Storing of total sparse jacobian
C                   ( including row and column scaling )
              INZ = 0
              DO 46 I=1,N
                IF(NAQ.NE.0)THEN
                  DO 47 K=1,NAQ
                    INZ = INZ+1
                    E(INZ)=-G(I,ICOLA(K),1)*XW(ICOLA(K))/XW(I+N)
47                CONTINUE
                ENDIF
                II = I
                  IF(M.EQ.2)THEN
                    II = 0
                    IF(NBQ.EQ.0) GOTO 9992
                    DO 48 L=1,NBQ
                      IF(ICOLB(NB+L).EQ.I) II = L
48                  CONTINUE
                    IF(II.EQ.0) GOTO 9992
                  ENDIF
                  INZ = INZ+1
                  E(INZ)=ONE
9992            CONTINUE
46            CONTINUE
              IF(M.NE.2)THEN
                DO 49 J=2,M1
                  J0 =(J-1)*N
                  J1 = J0+N
                  DO 50 I=1,N
                    DO 51 K=1,N
                      INZ = INZ+1
                      E(INZ)=-G(I,K,J)*XW(K+J0)/XW(I+J1)
51                  CONTINUE
                    II = I
                      IF(J.EQ.M1)THEN
                        II = 0
                        IF(NBQ.EQ.0) GOTO 9991
                        DO 52 L=1,NBQ
                          IF(ICOLB(NB+L).EQ.I) II = L
52                      CONTINUE
                        IF(II.EQ.0) GOTO 9991
                      ENDIF
                      INZ = INZ+1
                      E(INZ)=ONE
9991                CONTINUE
50                CONTINUE
49              CONTINUE
              ENDIF
              IF(NRS.NE.0)THEN
                DO 53 I=1,NRS
                  II = I+NB
                  IF(NAQ.NE.0)THEN
                    DO 54 K=1,NAQ
                      IF(IA(IROW(II),ICOLA(K)).NE.0)THEN
                        INZ = INZ+1
                        E(INZ)=-A(IROW(II),ICOLA(K))*DE(IROW(II))*
     *                  XW(ICOLA(K))
                      ENDIF
54                  CONTINUE
                  ENDIF
                  IF(NBQ.NE.0)THEN
                    DO 55 KK=1,NBQ
                      K = KK+NB
                      IF(IB(IROW(II),ICOLB(K)).NE.0)THEN
                        INZ = INZ+1
                        E(INZ)=-B(IROW(II),ICOLB(K))*DE(IROW(II))*
     *                  XW(ICOLB(K)+NM1)
                      ENDIF
55                  CONTINUE
                  ENDIF
53              CONTINUE
              ENDIF
C             ----------------------------------------------------
C             2.4 Save values of R(N)and HH((M-1)*N)
C:            Begin SetVec.Vec
              DO 56 L1=1,N
                RA(L1)=R(L1)
56            CONTINUE
C.            End SetVec.Vec
C:            Begin SetVec.Vec
              DO 57 L1=1,NM1
                HHA(L1)=HH(L1)
57            CONTINUE
C.            End SetVec.Vec
              NEXT = .FALSE.
C             ----------------------------------------------------
C             3 Main-part of iteration step
C             ----------------------------------------------------
C             3.1 Solution of the linear system
C             ----------------------------------------------------
C             3.1.1 Decomposition of (N,N)-matrix A
C             ----------------------------------------------------
C             3.1.2 LU-decomposition of(NDIM,NDIM)-MATRIX E
              IF(IAF.EQ.1)THEN
                DO 58 I=1,INZ
                  IRN(I)=IVECT(I)
                  ICN(I)=JVECT(I)
58              CONTINUE
                CALL MA28AD(NDIM,INZ,E,LICN,IRN,LIRN,ICN,UQ,IKEEP,
     *          IW,WO,IFLAG)
                IAF = 0
              ELSE
                CALL MA28BD(NDIM,INZ,E,LICN,IVECT,JVECT,ICN,IKEEP,
     *          IW,WO,IFLAG)
                IF(RMIN.LT.1.0D-4) IAF = 1
                IF(IAF.EQ.1)THEN
                  JACRST = .TRUE.
                  GOTO 9998
                ENDIF
              ENDIF
              IF(IFLAG.EQ.-1.OR.IFLAG.EQ.-2)THEN
                INFO = -1
                GOTO 9998
              ENDIF
              IF(IFLAG.LE.-3)THEN
                INFO = -9
                GOTO 9998
              ENDIF
C             ----------------------------------------------------
C             3.1.3 Solution of linear (N,N)-system
              CALL BGSOLI(N,M,M1,NM,NM1,NDIM,LICN,NKEEP,NA,NAQ,NB,
     *        NBQ,NRS,ITER,LEVEL,KPRINT,EPS,REDH,TOLMIN,FC,FCA,TOL,
     *        RELDIF,EPH,EPX1H,SIGDEL,SIGDLH,COND,CORR,HH,DHH,R,A,
     *        B,G,U,DE,DU,T1,DXQ,XW,DR,RF,WO,E,IROW,ICOLA,ICOLB,
     *        ICN,IKEEP,INFO)
              IF(INFO.NE.-1000) GOTO 9998
C             ----------------------------------------------------
C             3.2 Evaluation of scaled natural level function SUMX
C                 scaled maximum error norm CONV
C                 evaluation of (scaled) standard level function
C                 SUMF ( SUMF only, if KPRINT.GE.0 )
C                 and computation of ordinary newton corrections
C                 DX(N)
              CALL BGLVLS(N,M,NM,NM1,XW,DXQ,HH,R,DE,CONV,SUMX,SUMF,
     *        KPRINT)
C:            Begin SetVec.Vec
              DO 59 L1=1,NM
                DX(L1)=DXQ(L1)
59            CONTINUE
C.            End SetVec.Vec
C:            Begin SetVec.Vec
              DO 60 L1=1,NM
                XA(L1)=X(L1)
60            CONTINUE
C.            End SetVec.Vec
              SUMXA = SUMX
              CONVA = CONV
C             ----------------------------------------------------
C             3.3 a - priori estimate of relaxation factor FC
              JRED = 0
              IF(ITER.NE.0.AND.NONLIN.NE.0)THEN
                IF(NEW.EQ.0)THEN
C                 ------------------------------------------------
C                 3.3.1 Computation of the denominator of a-priori
C                       estimate
C:                FCDNM = Sum of Formula Elements (for 1,NM)
                  FCDNM = 0.0D0
                  DO 61 L1=1,NM
                    FCDNM=FCDNM+(((DX(L1)-DXQA(L1))/XW(L1))**2)
61                CONTINUE
C.                End MakeSum.Comp
C                 ------------------------------------------------
C                 3.3.2 New relaxation factor
                  FCDNM = FCDNM*SUMX
                  IF(FCDNM.GE.FCNUMP*FCMIN2)THEN
                    MUE = FCA*DSQRT(FCNUMP/FCDNM)
                    FC = DMIN1(MUE,ONE)
                  ELSE
                    FC = ONE
                  ENDIF
                ENDIF
                IF(FC.LT.FCMIN)THEN
                  INFO = -4
                  GOTO 9997
                ENDIF
              ENDIF
              LEVEL = 1
C             ----------------------------------------------------
C             3.4 Save natural level for later computations of
C                 corrector and print iterate
              FCNUMK = SUMX
              IF(KPRINT.GE.0)THEN
C               Print Standard - and natural level
                IF(KPRINT.GT.0)THEN
62                FORMAT('0',1X,66('*'))
                  WRITE(6,62)
63                FORMAT('0',4X,'It',7X,'Levelf',10X,'Levelx',18X,
     *            'New')
                  WRITE(6,63)
                ENDIF
64              FORMAT('0',4X,I2,5X,D10.3,2X,4X,D10.3,2X,13X,I2)
                WRITE(6,64)ITER,SUMF,SUMXA,NEW
                IF(KPRINT.GT.0)THEN
65                FORMAT('0',1X,66('*'))
                  WRITE(6,65)
                ENDIF
              ENDIF
C
C             Relaxation-factor reduction loop
C             ================================
C
C:            DO (Until)
66            CONTINUE
C               --------------------------------------------------
C               3.5 Preliminary new iterate
C:              DO (Until)
67              CONTINUE
                  FCOMPT = .FALSE.
C:                Vec X = Vec XA + Vec DX * Scalar (for 1,NM)
                  S1 = FC
                  DO 68 L1=1,NM
                    X(L1)=XA(L1)+DX(L1)*S1
68                CONTINUE
C.                End SetVec.Vec&VecxS
                  IF(ITER.GT.ITMAX)THEN
                    INFO = -2
                    GOTO 9997
                  ENDIF
C                 ------------------------------------------------
C                 3.5.1 Computation of the residual vector
                  CALL BGFCNI(IVPSOL,FCN,BC,N,M,NM,NM1,ITER,KPRINT,
     *            HSTART,FCMIN,T,X,X1,XM,T1,XU,HH,R,TOL,FC,FCOMPT,
     *            IVFAIL,KFLAG,KOUNT,INFO)
                  IF(IVFAIL)THEN
                    INFO = -4
                    GOTO 9997
                  ENDIF
                  IF(INFO.NE.-1000) GOTO 9997
                IF(.NOT.(FCOMPT)) GOTO  67
C.              UNTIL ( expression - negated above)
C               --------------------------------------------------
C               3.5.2 Solution of linear (N,N)-system
                CALL BGSOLI(N,M,M1,NM,NM1,NDIM,LICN,NKEEP,NA,NAQ,
     *          NB,NBQ,NRS,ITER,LEVEL,KPRINT,EPS,REDH,TOLMIN,FC,
     *          FCA,TOL,RELDIF,EPH,EPX1H,SIGDEL,SIGDLH,COND,CORR,
     *          HH,DHH,R,A,B,G,U,DE,DU,T1,DXQ,XW,DR,RF,WO,E,IROW,
     *          ICOLA,ICOLB,ICN,IKEEP,INFO)
                IF(INFO.NE.-1000) GOTO 9998
C               --------------------------------------------------
C               3.5.3 Evaluation of scaled natural level function
C                     SUMX
C                     scaled maximum error norm CONV and
C                     evaluation of (scaled) standard level
C                     function SUMF
                CALL BGLVLS(N,M,NM,NM1,XW,DXQ,HH,R,DE,CONV,SUMX,
     *          SUMF,KPRINT)
C               --------------------------------------------------
C               3.6 Convergence test
                IF(CONV.LE.EPS)THEN
                  INFO = 0
                  GOTO 9997
                ENDIF
C               --------------------------------------------------
C               3.7 Natural monotonicity test
                IF(SUMX.GT.SUMXA)THEN
C                 ------------------------------------------------
C                 3.8 Output of iterate
                  IF(KPRINT.GE.0)THEN
C                   Print Standard - and natural level, and
C                   damping factor
                    IF(KPRINT.GT.0)THEN
69                    FORMAT('0',1X,66('*'))
                      WRITE(6,69)
70                    FORMAT('0',4X,'It',7X,'Levelf',10X,'Levelx',
     *                8X,'Rel.Fc.')
                      WRITE(6,70)
                    ENDIF
71                  FORMAT('0',4X,I2,5X,D10.3,2X,4X,D10.3,2X,4X,F5.3)
                    WRITE(6,71)ITER,SUMF,SUMX,FC
                    IF(KPRINT.GT.0)THEN
72                    FORMAT('0',1X,66('*'))
                      WRITE(6,72)
                    ENDIF
                  ENDIF
                  JRED = JRED+1
                  IF(NONLIN.EQ.0)THEN
                    INFO = -4
                    GOTO 9997
                  ENDIF
C                 ------------------------------------------------
C                 3.9 Compute reduced relaxation factor
                  TH = FC-ONE
C:                FCDNM = Sum of Formula Elements (for 1,NM)
                  FCDNM = 0.0D0
                  DO 73 L1=1,NM
                    FCDNM=FCDNM+(((DXQ(L1)+TH*DX(L1))/XW(L1))**2)
73                CONTINUE
C.                End MakeSum.Comp
                  FC = DSQRT(FCNUMK/FCDNM)*FC*FC*HALF
C                 ------------------------------------------------
C                 3.10 Fail exit, if relaxation factor to small
                  JACRFR = FC.LT.FCMIN.OR.NEW.GT.0.AND.JRED.GT.1
                  IF(JACRFR.AND.NEW.EQ.0)THEN
                    INFO = -4
                    GOTO 9998
                  ENDIF
                ENDIF
              IF(.NOT.(SUMX.LE.SUMXA.OR.JACRFR)) GOTO  66
C.            UNTIL ( expression - negated above)
C
C             End of relaxation-factor reduction loop
C             =======================================
C
              IF(JACRFR)THEN
C               --------------------------------------------------
C               3.11 Restore former values for repeting iteration
C                    step
C               Restore former values
                LEVEL = 0
C:              Begin SetVec.Vec
                DO 74 L1=1,N
                  R(L1)=RA(L1)
74              CONTINUE
C.              End SetVec.Vec
C:              Begin SetVec.Vec
                DO 75 L1=1,N
                  X1(L1)=XA(L1)
75              CONTINUE
C.              End SetVec.Vec
C:              Begin SetVec.Vec
                DO 76 L1=1,N
                  XM(L1)=XA(L1+NM1)
76              CONTINUE
C.              End SetVec.Vec
C:              Begin SetVec.Vec
                DO 77 L1=1,NM
                  X(L1)=XA(L1)
77              CONTINUE
C.              End SetVec.Vec
C:              Begin SetVec.Vec&Vec
                DO 78 L1=1,NM1
                  XU(L1)=X(L1+N)+HHA(L1)
78              CONTINUE
C.              End SetVec.Vec&Vec
C:              Begin SetVec.Vec
                DO 79 L1=1,NM1
                  HH(L1)=HHA(L1)
79              CONTINUE
C.              End SetVec.Vec
                IF(KPRINT.GE.0)THEN
80                FORMAT('0',5X,I2,1X,'Not ','accepted ',
     *            'relaxation ','factor',5X,F5.3)
                  WRITE(6,80)ITER,FC
                ENDIF
                IF(ITER.EQ.0)THEN
                  FC = FCMIN
                ENDIF
                DIFAPP = .TRUE.
                JACRFR = .TRUE.
                GOTO 9998
              ENDIF
C             ----------------------------------------------------
C             4 Preparations to start the following iteration step
              ITER = ITER+1
              FCA = FC
C             ----------------------------------------------------
C             4.1 Print values
              IF(KPRINT.GE.0)THEN
C               Print Standard - and natural level, and damping
C               factor
                IF(KPRINT.GT.0)THEN
81                FORMAT('0',1X,66('*'))
                  WRITE(6,81)
82                FORMAT('0',4X,'It',7X,'Levelf',10X,'Levelx',8X,
     *            'Rel.Fc.')
                  WRITE(6,82)
                ENDIF
83              FORMAT('0',4X,I2,5X,D10.3,2X,4X,D10.3,2X,4X,F5.3)
                WRITE(6,83)ITER,SUMF,SUMX,FC
                IF(KPRINT.GT.0)THEN
84                FORMAT('0',1X,66('*'))
                  WRITE(6,84)
                  DO 85 J=1,M
86                  FORMAT(D13.5,2X)
                    WRITE(6,86)T(J)
87                  FORMAT((14X,3(D20.10,1X)))
                    WRITE(6,87)(X(L1),L1=(J-1)*N+1,J*N)
85                CONTINUE
                ENDIF
              ENDIF
9997        CONTINUE
C.          End of Segment BVPSOG.Core
9998      CONTINUE
        GOTO 17
        ENDIF
C.      EndWhile
C
C       End of main iteration loop
C       ==========================
C
C       ----------------------------------------------------------
C       5 Exits
C       ----------------------------------------------------------
C       5.1 Solution exit
        IF(INFO.EQ.0)THEN
          ITER = ITER+1
C:        Vec X = Vec X + Vec DXQ (for 1,NM)
          DO 88 L1=1,NM
            X(L1)=X(L1)+DXQ(L1)
88        CONTINUE
C.        End SetVec.&Vec
          INFO = ITER
          IF(KPRINT.LT.0)THEN
            GOTO 9999
          ENDIF
          IF(KPRINT.GT.0)THEN
C           Print levels, damping factor of last iteration step
C           and solution info
89          FORMAT('0',1X,66('*'))
            WRITE(6,89)
90          FORMAT('0',4X,'It',7X,'Levelf',10X,'Levelx',8X,
     *      'Rel.Fc.')
            WRITE(6,90)
          ENDIF
91        FORMAT('0',4X,I2,5X,D10.3,2X,4X,D10.3,2X,4X,F5.3)
          WRITE(6,91)ITER,SUMF,SUMX,FC
92        FORMAT('0',1X,66('*'))
          WRITE(6,92)
93        FORMAT('1')
          WRITE(6,93)
C         Print solution info
94        FORMAT('0','Solution ','of',1X,'boundary ','value ',
     *    'problem',' obtained',/,'0','BVPSOG',' required',I3,1X,
     *    'Iteration ','steps ','with',I4,1X,'trajectory',
     *    ' evaluations',//)
          WRITE(6,94)ITER,KOUNT
95        FORMAT('0','Achieved ','relative ','accuracy',D10.3,2X)
          WRITE(6,95)CONV
          IF(EPH.GT.CONV) CONV = EPH
96        FORMAT('0','Reliable ','relative ','accuracy',D10.3,2X,/)
          WRITE(6,96)CONV
          S = DLOG(DBLE((M-1)*(2*N+M-1))*EPMACH)
          DO 97 J=1,M1
            S = S+DLOG(RF(J))
97        CONTINUE
          IF(S.LT.DLOG(EPS))THEN
98          FORMAT('0','This ','boundary ','value ','problem ',
     *      'can ','also ','be ','solved ','by ','BVPSOL',/)
            WRITE(6,98)
          ENDIF
        ENDIF
C       ----------------------------------------------------------
C       5.2 Fail exit messages
C       ----------------------------------------------------------
C       5.2.1 Gaussian decomposition failed by singular jacobian
        IF(INFO.EQ.-1.AND.KPRINT.GE.0)THEN
99        FORMAT('0','Gaussian ','elimination ','failed ','by ',
     *    'singular ','Jacobian',/)
          WRITE(6,99)
        ENDIF
C       ----------------------------------------------------------
C       5.2.2 Termination after more than itmax iterations
        IF(INFO.EQ.-2.AND.KPRINT.GE.0)THEN
100       FORMAT('0','Iteration ','terminates ','after ','itmax ',
     *    '=',I3,2X,'iteration ','steps')
          WRITE(6,100)ITMAX
        ENDIF
C       ----------------------------------------------------------
C       5.2.3 Singular trajectory
        IF(INFO.EQ.-3.AND.KPRINT.GE.0)THEN
101       FORMAT('0','Singular ','trajectory ','by ','difference ',
     *    'approximation ','of ','the ','jacobian ','matrix',/)
          WRITE(6,101)
          J1 =-KFLAG
102       FORMAT('0','BVPSOG ','terminates',/,'Subinterval',I3,1X,
     *    'possibly ','insert ','new ','node',D20.11,2X,/)
          WRITE(6,102)J1,TFAIL
        ENDIF
C       ----------------------------------------------------------
C       5.2.4 Convergence fail of Gauss - Newton method
        IF(INFO.EQ.-4.AND.KPRINT.GE.0)THEN
103       FORMAT('0','Gauss ','Newton ','method ','fails ','to ',
     *    'converge',/)
          WRITE(6,103)
        ENDIF
C       ----------------------------------------------------------
C       5.2.5 Inconsistent initial data
        IF(INFO.EQ.-5.AND.KPRINT.GE.0)THEN
104       FORMAT('0','Error: ','initial ','data ','and ',
     *    'boundary ','conditions ','are ','inconsistent',/)
          WRITE(6,104)
        ENDIF
C       ----------------------------------------------------------
C       5.2.6 Multiple shooting condition too bad -insert new
C             nodes
        IF(INFO.EQ.-6.AND.KPRINT.GE.0)THEN
105       FORMAT('0','Termination ','since ','Multiple ',
     *    'Shooting ','condition',/,' ','or ','condition ','of ',
     *    'Jacobian ','is ','too ','bad',/,'insert ','new ',
     *    'nodes',/)
          WRITE(6,105)
          S = REDH/TOL
          DO 106 J=1,M1
            IF(RF(J).GT.S)THEN
107           FORMAT('0',8X,'in ','subinterval',2X,I3,/)
              WRITE(6,107)J
            ENDIF
106       CONTINUE
        ENDIF
C       ----------------------------------------------------------
C       5.2.7 Insufficient error tolerance for integrator
        IF(INFO.EQ.-7.AND.KPRINT.GE.0)THEN
          TOLH = DMIN1(REDH/COND,EPS)/SIGDEL
          RELDIF = DSQRT(TOLH/SIGDEL)
108       FORMAT('0','Suggested ','integrator ','accuracy',D10.1
     *    ,2X,/,'0','Suggested ','relative ','deviation ',
     *    'parameter',D10.1,2X,/)
          WRITE(6,108)TOLH,RELDIF
109       FORMAT('0','Reduce ','relative ','error ','tolerance ',
     *    'for ','integrator ','to',D10.1,2X,/,2X,'or ','insert ',
     *    'new ','nodes',/)
          WRITE(6,109)TOLH
          S = REDH/TOL
          DO 110 J=1,M1
            IF(RF(J).GT.S)THEN
111           FORMAT('0',8X,'in ','subinterval',2X,I3,/)
              WRITE(6,111)J
            ENDIF
110       CONTINUE
112       FORMAT('0','Reliable ','relative ','accuracy ',
     *    'greater ','than',1X,D6.1,2X,/)
          WRITE(6,112)1.0D-2
        ENDIF
C       ----------------------------------------------------------
C       5.2.8 Too small storage for sparse linear system solver
        IF(INFO.EQ.-9.AND.KPRINT.GE.0)THEN
113       FORMAT('0','Too ','small ','storage ','for ','linear ',
     *    'system ','solver',/)
          WRITE(6,113)
        ENDIF
C       ----------------------------------------------------------
C       5.3 Common exit
        IF(KPRINT.GE.0)THEN
C
114       FORMAT('0','Condition ','of ','Jacobian',D10.3,2X,/,'0',
     *    'Multiple ','shooting ','condition',D10.3,2X,/,'1')
          WRITE(6,114)COND,SIGDLH
          IF(INFO.GT.0)THEN
115         FORMAT('0','Solution ','data',/)
            WRITE(6,115)
          ENDIF
          IF(INFO.LT.0)THEN
116         FORMAT('0','Final ','data',/)
            WRITE(6,116)
          ENDIF
          DO 117 J=1,M
118         FORMAT(D13.5,2X)
            WRITE(6,118)T(J)
119         FORMAT((14X,3(D20.10,1X)))
            WRITE(6,119)(X(L1),L1=(J-1)*N+1,J*N)
117       CONTINUE
        ENDIF
C       End of exits
C       End of subroutine BVPSOG
9999  CONTINUE
C.    End of Segment BVPSOG.Body
      RETURN
      END
      SUBROUTINE BGFCNI(IVPSOL,FCN,BC,N,M,NM,NM1,ITER,KPRINT,
     *HSTART,FCMIN,T,X,X1,XM,T1,XU,HH,R,TOL,FC,FCOMPT,IVFAIL,KFLAG,
     *KOUNT,INFO)
      IMPLICIT DOUBLEPRECISION(S)
      EXTERNAL FCN,IVPSOL,BC
      INTEGER N,M,NM,NM1,ITER,KPRINT
      DOUBLE PRECISION HSTART,FCMIN
      INTEGER KOUNT,KFLAG,INFO
      LOGICAL IVFAIL
      DOUBLE PRECISION TOL,FC
      LOGICAL FCOMPT
      DOUBLE PRECISION T(M),X(NM)
      DOUBLE PRECISION XU(NM1),HH(NM1),R(N),X1(N),XM(N),T1(N)
C:    End Parameter
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      INTEGER J,J1,KB,KB2
      DOUBLE PRECISION HMAX,HSAVE,TJ,TJ1,H
      INTEGER L1
C:    Begin
C:    Begin of Segment FcnInt.Body
C       Computation of the trajectories (solution of M1 initial
C         value problems)
        KOUNT = KOUNT+1
        HSAVE = HSTART
        DO 120 J=1,M-1
          J1 = J+1
          TJ = T(J)
          TJ1 = T(J1)
          H = HSAVE
          HMAX = DABS(TJ1-TJ)
          KFLAG = 0
          KB =(J-1)*N
C:        Begin SetVec.Vec
          DO 121 L1=1,N
            T1(L1)=X(L1+KB)
121       CONTINUE
C.        End SetVec.Vec
          CALL IVPSOL(N,FCN,TJ,T1,TJ1,TOL,HMAX,H,KFLAG)
          IF(H.EQ.ZERO)THEN
C           singular trajectory
            IF(ITER.EQ.0)THEN
              INFO = -3
              GOTO 9990
            ENDIF
            IF(KPRINT.GE.0)THEN
122           FORMAT('0','Singular ','trajectory, ','relaxation ',
     *        'factor ','or ','pseudo-rank ','reduced',/)
              WRITE(6,122)
            ENDIF
            FC = FC*HALF
            IF(FC.LT.FCMIN)THEN
              IVFAIL = .TRUE.
              GOTO 9990
            ENDIF
            FCOMPT = .FALSE.
            GOTO 9990
          ENDIF
          FCOMPT = .TRUE.
C         continuity conditions
C:        Begin SetVec.Vec
          DO 123 L1=1,N
            XU(L1+KB)=T1(L1)
123       CONTINUE
C.        End SetVec.Vec
          KB2 = KB+N
C:        Begin SetVec.Vec-Vec
          DO 124 L1=1,N
            HH(L1+KB)=T1(L1)-X(L1+KB2)
124       CONTINUE
C.        End SetVec.Vec-Vec
120     CONTINUE
C       two-point boundary conditions
C:      Begin SetVec.Vec
        DO 125 L1=1,N
          XM(L1)=X(L1+NM1)
125     CONTINUE
C.      End SetVec.Vec
C:      Begin SetVec.Vec
        DO 126 L1=1,N
          X1(L1)=X(L1)
126     CONTINUE
C.      End SetVec.Vec
        CALL BC(X1,XM,R)
9990  CONTINUE
C.    End of Segment FcnInt.Body
      RETURN
      END
      SUBROUTINE BGSOLI(N,M,M1,NM,NM1,NDIM,LICN,NKEEP,NA,NAQ,NB,
     *NBQ,NRS,ITER,LEVEL,KPRINT,EPS,REDH,TOLMIN,FC,FCA,TOL,RELDIF,
     *EPH,EPX1H,SIGDEL,SIGDLH,COND,CORR,HH,DHH,R,A,B,G,U,DE,DU,T1,
     *DXQ,XW,DR,RF,WO,E,IROW,ICOLA,ICOLB,ICN,IKEEP,INFO)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N,M,M1,NM,NM1,NDIM,LICN,NKEEP,NA,NAQ,NB,NBQ,NRS,ITER,
     *LEVEL,KPRINT
      DOUBLE PRECISION EPS,REDH,TOLMIN,FC,FCA
      DOUBLE PRECISION TOL,RELDIF,EPH,EPX1H,SIGDEL,SIGDLH,COND,
     *CORR
      INTEGER INFO
      INTEGER IROW(N),ICOLA(N),ICOLB(N)
      INTEGER ICN(LICN),IKEEP(NKEEP)
      DOUBLE PRECISION G(N,N,M1)
      DOUBLE PRECISION A(N,N),B(N,N)
      DOUBLE PRECISION DXQ(NM),XW(NM),HH(NM1),DHH(NM1),DE(N),R(N),
     *DR(N),U(NM),DU(NM),T1(N),RF(M),E(LICN),WO(NM)
C:    End Parameter
C:    EPMACH = relative machine precision
      DOUBLE PRECISION EPMACH
      PARAMETER (EPMACH=2.23D-16)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION TEN
      PARAMETER (TEN=1.0D1)
      INTEGER I,IC,I0,J,J1,K,MTYPE
      DOUBLE PRECISION S,TOLH
      REAL ST
      INTEGER L1,L2
      DOUBLE PRECISION S1
C:    Begin
C:    Begin of Segment SolvIn.Body
C       ----------------------------------------------------------
C       1 Computation of right - hand side U(NDIM)
        DO 127 J=1,M1
          J0 =(J-1)*N
          J1 = J0+N
          DO 128 I=1,N
            U(I+J0)=HH(I+J0)/XW(I+J1)
128       CONTINUE
127     CONTINUE
        IF(NRS.NE.0)THEN
          DO 129 I=1,NRS
            I0 = IROW(NB+I)
            U(NM1+I)=DE(I0)*R(I0)
129       CONTINUE
        ENDIF
C       ----------------------------------------------------------
C       2 Solution of linear(NDIM,NDIM) -system
        MTYPE = 1
        CALL MA28CD(NDIM,E,LICN,ICN,IKEEP,U,WO,MTYPE)
        COND = ZERO
        IF(NAQ.NE.0)THEN
          DO 130 I=1,NAQ
            IC = ICOLA(I)
            S = U(I)
            DXQ(IC)=S*XW(IC)
            COND = COND+DABS(S)
130       CONTINUE
        ENDIF
        IF(M.NE.2)THEN
          DO 131 J=2,M1
            J0 =(J-1)*N
            DO 132 I=1,N
              S = U(J0-NA+I)
              DXQ(I+J0)=S*XW(I+J0)
              COND = COND+DABS(S)
132         CONTINUE
131       CONTINUE
        ENDIF
        IF(NBQ.NE.0)THEN
          DO 133 I=1,NBQ
            IC = ICOLB(NB+I)
            S = U(NM1-NA+I)
            DXQ(IC+NM1)=S*XW(IC+NM1)
            COND = COND+DABS(S)
133       CONTINUE
        ENDIF
        IF(NB.NE.0)THEN
          DO 134 K=1,NB
            IC = ICOLB(K)
            DXQ(IC+NM1)=ZERO
134       CONTINUE
        ENDIF
        IF(NA.NE.0)THEN
          DO 135 K=1,NA
            IC = ICOLA(NAQ+K)
            DXQ(IC)=ZERO
135       CONTINUE
        ENDIF
C       ----------------------------------------------------------
C       3 Iterative refinement
        IF(KPRINT.GT.0)THEN
136       FORMAT('0','Iterative ','refinement',/)
          WRITE(6,136)
        ENDIF
C       ----------------------------------------------------------
C       4 Computation of required continuity residuals DHH(NM1)
        DO 137 J=1,M1
          J0 =(J-1)*N
          J1 = J0+N
          DO 138 I=1,N
            S = HH(I+J0)
            DO 139 K=1,N
              S = S+G(I,K,J)*DXQ(K+J0)
139         CONTINUE
            DHH(I+J0)=S-DXQ(I+J1)
138       CONTINUE
137     CONTINUE
C       ----------------------------------------------------------
C       5 Computation of boundary residual DR(N)
C:      Begin SetVec.MatxVec
        DO 140 L1=1,N
          S1=0.0
          DO 141 L2=1,N
            S1=S1+A(L1,L2)*DXQ(L2)
141       CONTINUE
          DR(L1)=S1
140     CONTINUE
C.      End SetVec.MatxVec
C:      Begin SetVec.MatxVec
        DO 142 L1=1,N
          S1=0.0
          DO 143 L2=1,N
            S1=S1+B(L1,L2)*DXQ(L2+NM1)
143       CONTINUE
          T1(L1)=S1
142     CONTINUE
C.      End SetVec.MatxVec
C:      Vec DR = Formula (for 1,N)
        DO 144 L1=1,N
          DR(L1)=R(L1)+DR(L1)+T1(L1)
144     CONTINUE
C.      End SetVec.Comp
C       ----------------------------------------------------------
C       6 Computation of residual DU(NDIM)
        DO 145 J=1,M1
          J0 =(J-1)*N
          J1 = J0+N
          DO 146 I=1,N
            DU(J0+I)=DHH(I+J0)/XW(I+J1)
146       CONTINUE
145     CONTINUE
        IF(NRS.NE.0)THEN
          DO 147 I=1,NRS
            I0 = IROW(NB+I)
            DU(NM1+I)=DE(I0)*DR(I0)
147       CONTINUE
        ENDIF
C       ----------------------------------------------------------
C       7 Computation of correction
        MTYPE = 1
        CALL MA28CD(NDIM,E,LICN,ICN,IKEEP,DU,WO,MTYPE)
C       ----------------------------------------------------------
C       8 Refinement of DXQ
        CORR = ZERO
        IF(NAQ.NE.0)THEN
          DO 148 I=1,NAQ
            IC = ICOLA(I)
            S = DU(I)
            DXQ(IC)=DXQ(IC)+S*XW(IC)
            CORR = CORR+DABS(S)
148       CONTINUE
        ENDIF
        IF(M.NE.2)THEN
          DO 149 J=2,M1
            J0 =(J-1)*N
            DO 150 I=1,N
              S = DU(J0-NA+I)
              DXQ(I+J0)=DXQ(I+J0)+S*XW(I+J0)
              CORR = CORR+DABS(S)
150         CONTINUE
149       CONTINUE
        ENDIF
        IF(NBQ.NE.0)THEN
          DO 151 I=1,NBQ
            IC = ICOLB(NB+I)
            S = DU(NM1-NA+I)
            DXQ(IC+NM1)=DXQ(IC+NM1)+S*XW(IC+NM1)
            CORR = CORR+DABS(S)
151       CONTINUE
        ENDIF
        COND = CORR/(COND*EPMACH)
        IF(KPRINT.GT.0)THEN
152       FORMAT('0','Norm ','of ','residual',D12.3,2X)
          WRITE(6,152)CORR
        ENDIF
C       End of iterative refinement
C       ----------------------------------------------------------
C       9 Determination and adaptation of parameters TOL and
C         RELDIF
        IF(LEVEL.NE.0)THEN
          DO 153 J=1,M1
            J0 =(J-1)*N
            J1 = J0+N
            SIGDEL = ZERO
            ST = ZERO
            DO 154 I=1,N
              II = N
              IF(J.EQ.1) II = NAQ
              S = DABS(DXQ(I+J0))/XW(I+J0)
              IF(ST.LT.S) ST = S
              S = ZERO
              DO 155 K=1,II
                KK = K
                IF(J.EQ.1) KK = ICOLA(KK)
                S = S+G(I,KK,J)*DXQ(KK+J0)/XW(I+J1)
155           CONTINUE
              S = DABS(S)
              IF(SIGDEL.LT.S) SIGDEL = S
154         CONTINUE
            RF(J)=SIGDEL/ST+ONE
153       CONTINUE
          IF(KPRINT.GT.0)THEN
156         FORMAT('0','Norms ','of ','wronskians')
            WRITE(6,156)
157         FORMAT((1X,5(D12.3,1X)))
            WRITE(6,157)(RF(L1),L1=1,M1)
          ENDIF
          SIGDLH = ZERO
          DO 158 J=1,M1
            IF(SIGDLH.LT.RF(J)) SIGDLH = RF(J)
158       CONTINUE
          SIGDEL = SIGDLH
          IF(FC.EQ.ONE.AND.FCA.EQ.ONE.AND.ITER.GT.0) SIGDEL =
     *    SIGDLH*COND
          SIGDEL = DMAX1(SIGDEL,TEN)
          EPH = TOL*SIGDEL
          IF(EPH.GT.REDH)THEN
            INFO = -6
            GOTO 9989
          ENDIF
          TOLH = EPS/SIGDEL
          IF(TOLH.LT.TOLMIN) TOLH = TOLMIN
CWEI;     TOL = TOLH
CWEI;     RELDIF = DSQRT(TOL/SIGDEL)
          IF(KPRINT.GE.0)THEN
159         FORMAT('0','Suggested ','integrator ','accuracy',D10.1
     *      ,2X,/,'0','Suggested ','relative ','deviation ',
     *      'parameter',D10.1,2X,/)
            WRITE(6,159)TOLH,RELDIF
160         FORMAT('0','Adapted ','in ','the ','next ',
     *      'iteration ','step',/)
            WRITE(6,160)
          ENDIF
        ENDIF
9989  CONTINUE
C.    End of Segment SolvIn.Body
      RETURN
      END
      SUBROUTINE BGSCLE(N,M,NM,NM1,X,XU,XW,XTHR)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N
      INTEGER M
      INTEGER NM
      INTEGER NM1
      DOUBLE PRECISION X(NM)
      DOUBLE PRECISION XW(NM)
      DOUBLE PRECISION XU(NM1)
      DOUBLE PRECISION XTHR
C:    End Parameter
C     PROVIDES SCALING XW(NM)OF VARIABLES X(NM)
C:    EPMACH = relative machine precision
      DOUBLE PRECISION EPMACH
      PARAMETER (EPMACH=2.23D-16)
C:    SMALL = squareroot of "smallest positive machine number
C     divided by relative machine precision"
      DOUBLE PRECISION SMALL
      PARAMETER (SMALL=4.94D-32)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      DOUBLE PRECISION HALF
      PARAMETER (HALF=0.5D0)
      DOUBLE PRECISION RED
      PARAMETER (RED=1.0D-2)
      INTEGER I,J,J0,J1
      DOUBLE PRECISION XMAX
      INTEGER L1
C:    Begin
C:    Vec XW = Formula (for 1,N)
      DO 161 L1=1,N
        XW(L1)=DABS(X(L1))
161   CONTINUE
C.    End SetVec.Comp
C     ------------------------------------------------------------
C     1 Arithmetic mean for XW(2*N)... XW(M*N)
      DO 162 J=1,M-1
        J0 =(J-1)*N
        J1 = J0+N
        DO 163 I=1,N
          XW(I+J1)=(DABS(X(I+J1))+DABS(XU(I+J0)))*HALF
163     CONTINUE
162   CONTINUE
C     ------------------------------------------------------------
C     2 Threshold
      DO 164 I=1,N
        XMAX = ZERO
        DO 165 J=0,NM1,N
          IF(XMAX.LT.XW(I+J)) XMAX = XW(I+J)
165     CONTINUE
        XMAX = XMAX*RED
        IF(XMAX.LT.XTHR) XMAX = XTHR
        DO 166 J=0,NM1,N
          IF(XW(I+J).LT.XMAX) XW(I+J)=XMAX
166     CONTINUE
164   CONTINUE
      RETURN
C     End of subroutine BGSCLE
      END
      SUBROUTINE BGLVLS(N,M,NM,NM1,XW,DXQ,HH,R,DE,CONV,SUMX,SUMF,
     *KPRINT)
      IMPLICIT DOUBLEPRECISION(S)
C
      INTEGER N,M,NM,NM1,KPRINT
      DOUBLE PRECISION XW(NM),DXQ(NM),HH(NM1),R(N),DE(N)
      DOUBLE PRECISION CONV,SUMX,SUMF
C:    End Parameter
C:    SMALL = squareroot of "smallest positive machine number
C     divided by relative machine precision"
      DOUBLE PRECISION SMALL
      PARAMETER (SMALL=4.94D-32)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,J,J0,J1
      DOUBLE PRECISION S
      INTEGER L1
C:    Begin
C     ------------------------------------------------------------
C     1 Evaluation of scaled natural level function SUMX and
C       scaled maximum error norm CONV
      CONV = ZERO
      SUMX = ZERO
      DO 167 J=1,NM
        S = DABS(DXQ(J)/XW(J))
        IF(CONV.LT.S) CONV = S
        SUMX = SUMX+S*S
167   CONTINUE
C     ------------------------------------------------------------
C     2 Evaluation of (scaled) standard level function sumfs (only
C       if needed for print)
C:    SUMF = Sum of Formula Elements (for 1,N)
      SUMF = 0.0D0
      DO 168 L1=1,N
        SUMF=SUMF+((R(L1)*DE(L1))**2)
168   CONTINUE
C.    End MakeSum.Comp
      DO 169 J=1,M-1
        J0 =(J-1)*N
        J1 = J0+N
        DO 170 I=1,N
          SUMF = SUMF+(HH(I+J0)/XW(I+J1))**2
170     CONTINUE
169   CONTINUE
C     End of subroutine BGLVLS
      RETURN
      END
      SUBROUTINE BGDERA(BC,N,M,NM,XW,X1,XM,R,RH,A,B,RELDIF)
      IMPLICIT DOUBLEPRECISION(S)
      EXTERNAL BC
      INTEGER N,M,NM
      DOUBLE PRECISION XW(NM),X1(N),XM(N),R(N),RH(N)
      DOUBLE PRECISION A(N,N),B(N,N)
      DOUBLE PRECISION RELDIF
C:    End Parameter
C     Difference approx. of boundary derivative matrices A(N,N)and
C       B(N,N)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,K,NM1
      DOUBLE PRECISION S,XH
C:    Begin
      NM1 = N*(M-1)
      DO 171 K=1,N
        XH = X1(K)
        S = RELDIF*XW(K)
        IF(XH.LT.ZERO) S =-S
        X1(K)=XH+S
        CALL BC(X1,XM,RH)
        X1(K)=XH
        S = ONE/S
        DO 172 I=1,N
          A(I,K)=(RH(I)-R(I))*S
172     CONTINUE
        XH = XM(K)
        S = RELDIF*XW(K+NM1)
        IF(XH.LT.ZERO) S =-S
        XM(K)=XH+S
        CALL BC(X1,XM,RH)
        XM(K)=XH
        S = ONE/S
        DO 173 I=1,N
          B(I,K)=(RH(I)-R(I))*S
173     CONTINUE
171   CONTINUE
C     END SUBROUTINE BGDERA
      RETURN
      END
      SUBROUTINE BGDERG(FCN,N,NAQ,M,M1,NM,NM1,T,X,XU,XW,XJ,TJ,G,
     *ICOLA,IVPSOL,HSTART,TOL,RELDIF,KFLAG)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N
      INTEGER NAQ
      INTEGER M
      INTEGER M1
      INTEGER NM
      INTEGER NM1
      DOUBLE PRECISION T(M)
      DOUBLE PRECISION X(NM)
      DOUBLE PRECISION XU(NM1)
      DOUBLE PRECISION XW(NM)
      DOUBLE PRECISION XJ(N)
      DOUBLE PRECISION TJ
      DOUBLE PRECISION G(N,N,M1)
      INTEGER ICOLA(N)
      EXTERNAL IVPSOL
      DOUBLE PRECISION HSTART
      DOUBLE PRECISION TOL
      DOUBLE PRECISION RELDIF
      INTEGER KFLAG
C:    End Parameter
C     Difference approximation of Wronskian Matrices G(1),.., G(M1)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,IK,J,J0,J1,K
      DOUBLE PRECISION HMAX,H,HSAVE,S,TJ1,TJA,TH
      EXTERNAL FCN
C:    Begin
      HSAVE = HSTART
      DO 174 J=1,M-1
        J0 =(J-1)*N
        J1 = J+1
        TJA = T(J)
        TJ1 = T(J1)
        HMAX = DABS(TJ1-TJA)
        DO 175 IK=1,N
          I = ICOLA(IK)
          H = HSAVE
          IF(J.NE.1.OR.IK.LE.NAQ)THEN
            TJ = TJA
            KFLAG = 0
            DO 176 K=1,N
              XJ(K)=X(K+J0)
176         CONTINUE
            TH = XJ(I)
            S = RELDIF*XW(I+J0)
            IF(TH.LT.ZERO) S =-S
            XJ(I)=TH+S
            S = ONE/S
            CALL IVPSOL(N,FCN,TJ,XJ,TJ1,TOL,HMAX,H,KFLAG)
            IF(H.EQ.ZERO)THEN
              KFLAG =-J
              RETURN
            ENDIF
            DO 177 K=1,N
              G(K,I,J)=S*(XJ(K)-XU(K+J0))
177         CONTINUE
          ENDIF
175     CONTINUE
        HSAVE = H
174   CONTINUE
      KFLAG = 0
C     END OF SUBROUTINE BGDERG
      RETURN
      END
      SUBROUTINE BGRK1G(N,M,M1,NM,NM1,XW,DX,HH,HHA,DXJ,G,FCA)
      IMPLICIT DOUBLEPRECISION(S)
      INTEGER N,M,M1,NM,NM1
      DOUBLE PRECISION XW(NM),DX(NM),HH(NM1),HHA(NM1),DXJ(N)
      DOUBLE PRECISION G(N,N,M1)
      DOUBLE PRECISION FCA
C:    End Parameter
C     RANK-1 UPDATES OF WRONSKIAN MATRICES G(1),..., G(M1)
      DOUBLE PRECISION ONE
      PARAMETER (ONE=1.0D0)
      DOUBLE PRECISION ZERO
      PARAMETER (ZERO=0.0D0)
      INTEGER I,J,J0,K
      DOUBLE PRECISION DNM,FCH,S,T
C:    Begin
      FCH = FCA-ONE
      DO 178 J=1,M1
        J0 =(J-1)*N
        DNM = ZERO
        DO 179 I=1,N
          T = DX(I+J0)/XW(I+J0)
          DXJ(I)=T/XW(I+J0)
          DNM = DNM+T*T
179     CONTINUE
        DNM = DNM*FCA
        IF(DNM.NE.ZERO)THEN
          DO 180 K=1,N
            T = DXJ(K)/DNM
            DO 181 I=1,N
              S = G(I,K,J)
              IF(S.NE.ZERO) G(I,K,J)=S+T*(HH(I+J0)+FCH*HHA(I+J0))
181         CONTINUE
180       CONTINUE
        ENDIF
178   CONTINUE
C     END OF SUBROUTINE BGRK1G
      RETURN
      END
C
C*    Group  Initial value problem solver (Code DIFEX1)   
C
      SUBROUTINE DIFEX1 (N,FCN,T,Y,TEND,TOL,HMAX,H,KFLAG)
C
C* Begin Prologue DFEX1
C
C  ---------------------------------------------------------------------
C
C* Title
C
C  Explicit Extrapolation Integrator
C  for Non-Stiff Systems of Ordinary First-Order Differential Equations
C
C* Written by        P. Deuflhard, U. Nowak, U. Poehle
C* Purpose           Solution of systems of initial value problems
C* Method            Implicit mid-point rule discretization with
C                    h**2-extrapolation
C* Category          i1a1a. - System of nonstiff first order
C                             differential equations
C* Keywords          extrapolation, ODE, explicit mid-point rule,
C                    nonstiff
C* Version           1.0 , February 1988
C* Latest Change     January 1991
C* Library           CodeLib
C* Code              Fortran 77
C                    Double Precision
C* Environment       Standard version for FORTRAN77 environments on
C                    PCs, workstations, and hosts
C* Copyright     (c) Konrad-Zuse-Zentrum fuer Informationstechnik
C                    Berlin (ZIB)
C                    Heilbronner Str. 10, D-1000 Berlin 31
C                    phone:   0049+30/89604-0
C                    telefax: 0049+30/89604-125
C* Contact           ZIB
C                    Numerical Software Development
C                    Uwe Poehle
C                    phone:   0049+30/89604-184
C                    e-mail:  poehle@sc.zib-berlin.de
C
C  ---------------------------------------------------------------------
C
C* Licence
C  -------
C
C  You may use or modify this code for your own non-commercial
C  purposes for an unlimited time. 
C  In any case you should not deliver this code without a special 
C  permission of ZIB.
C  In case you intend to use the code commercially, we oblige you
C  to sign an according licence agreement with ZIB.
C
C
C* Warranty
C  --------
C 
C  This code has been tested up to a certain level. Defects and
C  weaknesses, which may be included in the code, do not establish
C  any warranties by ZIB. ZIB does not take over any liabilities
C  which may follow from aquisition or application of this code.
C
C
C* Software status 
C  ---------------
C
C  This code is under care of ZIB and belongs to ZIB software
C  class I.
C
C
C  ---------------------------------------------------------------------
C
C* References:
C
C /1/ W. B. Gragg:
C     On Extrapolation Algorithms for Ordinary
C     Initial Value Problems
C     SIAM J. Numer. Anal. 2, 384-404 (1965)
C
C /2/ R. Bulirsch, J. Stoer:
C     Numerical Treatment of Ordinary Differential Equations
C     by Extrapolation Methods
C     Num. Math. 8, 1-13 (1966)
C
C /3/ P. Deuflhard:
C     Order and Stepsize Control in Extrapolation Methods
C     Numer. Math. 41, 399-422 (1983)
C
C
C* External Subroutine: (to be Supplied by the User)
C
C    FCN           EXT  Subroutine FCN (N,T,Y,DY)
C                       Right-Hand Side of First-Order
C                       Differential Equations
C                       N      Number of First-Order ODE'S
C                       T      Actual Position
C                       Y(N)   Values at T
C                       DY(N)  Derivatives at T
C
C
C* Parameters: (* Marks Inout Parameters)
C
C    N         I   IN   Number of First-Order ODE'S
C  * T         D   IN   Starting Point of Integration
C                       (T .LE. TEND)
C                  OUT  Achieved Final Point of Integration
C  * Y         D   IN   Array of Initial Values Y(1),...,Y(N)
C                  OUT  Array of Final Values
C    TEND      D   IN   Prescribed Final Point of Integration
C    TOL       D   IN   Prescribed Relative Precision (.GT.0)
C    HMAX      D   IN   Maximum Permitted Stepsize
C  * H         D   IN   Initial Stepsize Guess
C                  OUT  Stepsize Proposal for Next Integration Step
C                       (H .EQ. 0. ,if DIFEX1 Fails to Proceed)
C  * KFLAG     I   IN   Print Parameter
C                        0   no Output
C                        1   Integration Monitor
C                        2   Intermediate Solution Points  T,Y(I),I=1,N
C                        3   Integration Monitor and Solution Points
C                  OUT  Error Flag
C                       .GE. 0  Successful Integration
C                               (KFLAG not Altered Internally)
C                       -1   TEND .LT. T
C                       -2   More Than NSTMAX Basic Integration Steps
C                            per Interval Have Been Performed
C                       -3   More Than JRMAX Stepsize Reductions
C                            Occurred per Basic Integration Step
C                       -4   Stepsize Proposal for Next Basic
C                            Integration Step too Small
C
C    COMMON /STAT/ NFCN, NSTEP, NACCPT, NREJCT, NDEC, NSOL
C                       Internally Initialized, for Statistical
C                       Purposes
C    NFCN               Number of FCN-Evaluatios
C    NSTEP              Number of Integration Steps
C    NACCPT             Number of Steps Accepted
C    NREJCT             Number of Steps Rejected
C    NDEC               Number of Decompositions
C    NSOL               Number of Substitutions
C
C* Type Declaration
C
      INTEGER I, J, JK, JL, JM, JMACT, JOPT, JRED, JRMAX, J1, K, KFIN,
     2KFLAG, KM, KMACT, KOPT, K1, L, LOUT, M, MAXODE, MDT, M1, N,
     3NACCPT, NDEC, NFCN, NJ, NREJCT, NSOL, NSTEP, NSTMAX
C
      DOUBLE PRECISION ALPHA, AWK, B1, C, D, DBLE, DMAX1, DMIN1, DT,
     2DUMMY, DY, DYM, DZ, D1ERRN, EPMACH, EPSAFE, ERR, FC, FCK, FCM,
     3FCO, FMIN, FNJ, FN1, FN, H, HALF, HJ,  HJ2, HMAX, HMAXU, HN,
     4HREST, HRTRN, OMJ, OMJO, ONE, PCT101, PCT90, QUART, RED, RMAX, RO,
     5SAFE, SMALL, T, TEND, TN, TOL, TOLH, TOLMIN, U, V, W, Y, YA, YK,
     6YM, YMAX, YWGT, ZERO
C
      LOGICAL QFIRST, QKONV, QLAST, QPRMON, QPRSOL, QRED
C
      CHARACTER CHGDAT*20, PRODCT*8
C
C
C* Constants Problem Oriented: (to be Supplied by the User)
C
C    MAXODE    I   K    Maximal Number of First-Order ODE'S
C
      PARAMETER ( MAXODE = 51            )
C
C* Constants Machine Oriented: (to be Verified by the User)
C  (Adapted to Siemens 7.865, IBM 370-Compatible)
C
C    EPMACH    D   K    Relative Machine Precision
C    LOUT      I   K    Output is Written on Logical Unit LOUT
C    SMALL     D   K    Square-Root of Smallest Positive Machine Number
C
      PARAMETER ( EPMACH = 2.22D-16      ,
     2            EPSAFE = EPMACH*10.0D0 ,
     3            LOUT   = 6             ,
     4            SMALL  = 7.35D-40      )
C
C* Other Constants:
C
C    HALF      D   K    1/2
C    ONE       D   K    1
C    PCT101    D   K    101 Percent
C    PCT90     D   K    90 Percent
C    QUART     D   K    1/4
C    ZERO      D   K    0
C
      PARAMETER ( HALF   = 0.5  D0       ,
     2            ONE    = 1.0  D0       ,
     3            PCT101 = 1.01 D0       ,
     4            PCT90  = 0.9  D0       ,
     5            QUART  = 0.25 D0       ,
     6            ZERO   = 0.0  D0       )
C
C* Control Parameters: (to be Supplied by the User)
C  Standard Values Fixed Below
C
C    NSTMAX    I   K    Maximum Permitted Number of Integration Steps
C                       per Interval  =  10000
C    JRMAX     I   K    Maximum Permitted Number of Stepsize Reductions
C    KM        I   K    Prescribed Maximum Column Number
C    JM        I   K    Associated Maximum Row Number
C                       (JM = KM + 1)
C    MDT       I   K    Associated Dimension of DT
C    D1SEQ         EXT  Subroutine D1SEQ(JM,NJ)
C                       Generate Stepsize Sequence with Respect to /1/
C                       JM     Maximum Row Number
C                       NJ     Array(JM) of Stepsize Sequence
C    D1SCAL        EXT  Subroutine D1SCAL (MODE, Y, N, YOLD, YWGT,
C                                          YMAX, THREL, THABS)
C                       Scaling for DIFEX1
C                       MODE   ='INITIAL '    Initial Scaling
C                              ='INTERNAL'    Scaling during Discret.
C                              ='ACCEPTED'    Rescaling if Step Accepted
C                              Else           Error
C                       Y      Array of Values Y(1),...,Y(N)
C                       N      Length of Vectors Y, YOLD, YWGT, and YMAX
C                       YOLD   Array of Old Values
C                       YWGT   Array of Scaled Values
C                       YMAX   Array of Maximum Values
C                       THREL  Relative Threshold Value
C                       THABS  Absolute Threshold Value
C    D1ERRN        EXT  Double Precision Function D1ERRN(Y, N, YWGT)
C                       Scaled Root Mean Square Error
C                       Y      Array of Values Y(1),...,Y(N)
C                       N      Length of Vectors Y and YWGT
C                       YWGT   Array of Scaled Values
C
      PARAMETER ( NSTMAX = 10000         ,
     2            JRMAX  = 10            ,
     3            KM     = 8             ,
     4            JM     = KM + 1        ,
     5            MDT    = MAXODE*JM     )
C
C* Internal Parameters: (Modification not Recommended)
C
C
      PARAMETER ( FMIN   = 1.0  D-3      ,
     2            RMAX   = 0.75 D0       ,
     3            RO     = QUART         ,
     4            SAFE   = 0.7  D0       )
C
C
C* Local Variables: (Workspace)
C
C
C
C    QFIRST    L   V    First Integration Step
C    QKONV     L   V    Convergence Detected
C    QLAST     L   V    Last Integration Step
C    QPRMON    L   V    Print Integration Monitor
C    QPRSOL    L   V    Print Intermediate Solution Points
C
C* Dimensions:
C
      DIMENSION ALPHA(JM,JM), AWK(JM), D(JM,JM), DT(MAXODE,JM),
     2DY(MAXODE), DYM(MAXODE), DZ(MAXODE), FCK(KM), NJ(JM),
     3Y(MAXODE), YK(MAXODE), YM(MAXODE), YMAX(MAXODE), YWGT(MAXODE)
C
      COMMON /STAT/ NFCN, NSTEP, NACCPT, NREJCT, NDEC, NSOL
C
C*******  Revision 1 *******  Latest Change:
      DATA      CHGDAT      /'February 25, 1988   '/
      DATA      PRODCT      /'DIFEX1'/
C***************************
C
C
      DATA  DT/MDT*0.D0/
C
C---1. Initial Preparations
      QPRMON = (KFLAG .EQ. 1 .OR. KFLAG .EQ. 3)
      QPRSOL = (KFLAG .GE. 2)
      IF (TEND .LT. T) THEN
C        Error 1
         IF (QPRMON) WRITE (LOUT, 10001) PRODCT, T, TEND
         KFLAG = -1
         GOTO 9
C        Exit to Return
      ENDIF
      DO 1001 I = 1, N
 1001    YMAX(I) = ZERO
C     ENDDO
      HREST = TEND - T
      H = DMIN1 (H, HREST)
      HMAXU = HMAX
      IF (HMAX .GT. EPSAFE) THEN
         FCM = DMAX1(H/HMAX, FMIN)
      ELSE
         FCM = FMIN
      ENDIF
      KMACT = KM
      JMACT = JM
      CALL D1SEQ (JM, NJ)
      FN = DBLE (N)
      FN1 = DBLE (NJ(1))
      TOLH = RO*TOL
      TOLMIN = EPSAFE*FN
      IF (TOL .LT. TOLMIN) THEN
         WRITE (LOUT, 10002) PRODCT, TOL, TOLMIN
         TOL = TOLMIN
      ENDIF
C
C---  Compute Amount of Work per Row of Extrapolation Tableau
      AWK(1) = FN1 + ONE
      DO 101 J=2,JM
         J1 = J - 1
         FNJ = DBLE (NJ(J))
         V = AWK(J1) + FNJ
         AWK(J) = V
         DO 1011 K=1,J1
 1011       D(J,K) = (FNJ / DBLE (NJ(K)))*(FNJ / DBLE (NJ(K)))
C        ENDDO
         IF (J .NE. 2) THEN
            W = V - AWK(1) + ONE
            DO 1012 K1=2,J1
               K = K1 - 1
               U = (AWK(K1) - V) / (W*DBLE(K + K1))
               U = TOLH**U
 1012          ALPHA(J1,K) = U
C           ENDDO
         ENDIF
 101     CONTINUE
C     ENDDO
C
C---1.2 Determination of Maximum Column Number in Extrapolation
C---    Tableau (Information Theoretic Concept, Ref./3/)
      KOPT = 1
      JOPT = 2
 121  CONTINUE
C     DO WHILE (JOPT .LT. KMACT .AND.
C               AWK(JOPT+1)*PCT101 .LE. AWK(JOPT)*ALPHA(JOPT,KOPT))
         IF (JOPT .GE. KMACT .OR.
     2      AWK(JOPT+1)*PCT101 .GT. AWK(JOPT)*ALPHA(JOPT,KOPT)) GOTO 122
C                                                         Exit 121
         KOPT = JOPT
         JOPT = JOPT + 1
         GOTO  121
C     ENDDO
 122  KMACT = KOPT + 1
      JMACT = JOPT
      IF (QPRMON) WRITE (LOUT, 11221)
     2   PRODCT, CHGDAT, TOL, KMACT, NJ
C
      IF (QPRSOL) WRITE (LOUT, 11222)
      NSTEP = 0
      QFIRST = .TRUE.
      QLAST = .FALSE.
CWEI; NFCN = 0
      KFIN = 0
      OMJO = ZERO
      CALL D1SCAL ('INITIAL ', Y, N, DUMMY, YWGT, YMAX, TOL, ONE)
C
C---2. Basic Integration Step
 2    CONTINUE
C     DO WHILE (T .NE. TEND)
         IF (QPRMON) WRITE (LOUT, 12001) NSTEP,NFCN,T,KFIN,KOPT
         IF (QPRSOL) WRITE (LOUT, 12002) NSTEP,NFCN,T,H,(Y(I),I=1,N)
         JRED = 0
C
C---     Explicit Euler Starting Step
         CALL FCN (N, T, Y, DZ)
         NFCN = NFCN + 1
C
C---3.   Basic Discretization Step
 3       CONTINUE
C        DO WHILE (JRED .LE. JRMAX .AND. .NOT. QKONV)
            IF (QLAST) THEN
               TN = TEND
            ELSE
               TN = T + H
            ENDIF
            IF (TN .EQ. T) THEN
C              Error 4
               IF (QPRMON) WRITE (LOUT, 13001) PRODCT
               KFLAG = -4
               GOTO  9
C              Exit to Return
            ENDIF
C
C---3.1     Internal Discretization
            DO 31 J=1,JMACT
               M = NJ(J)
               M1 = M - 1
               KFIN = J - 1
               FNJ = DBLE (M)
               HJ = H / FNJ
               HJ2 = HJ + HJ
               DO 3101 I=1,N
                  YK(I) = Y(I)
 3101             YM(I) = Y(I) + HJ*DZ(I)
C              ENDDO
C
C---3.1.3      Explicit Mid-Point Rule
               DO 313 K=1,M1
                  CALL FCN (N, T + HJ*DBLE (K), YM, DY)
                  NFCN = NFCN + 1
                  DO 3135 I=1,N
                     U = YK(I) + HJ2*DY(I)
                     YK(I) = YM(I)
 3135                YM(I) = U
C                 ENDDO
 313              CONTINUE
C              ENDDO
C
C---3.1.4      Smoothing Final Step
               CALL FCN (N, TN, YM, DY)
               NFCN = NFCN + 1
               DO 3141 I = 1,N
 3141             YM(I) = (YM(I) + YK(I) + HJ*DY(I))*HALF
C              ENDDO
C
C---3.1.5      Extrapolation
               ERR = ZERO
               DO 315 I=1,N
                  C = YM(I)
                  V = DT(I,1)
                  DT(I,1) = C
                  IF (J .NE. 1) THEN
                     YA = C
                     DO 3151 K=2,J
                        JK = J - K + 1
                        B1 = D(J,JK)
                        W = C - V
                        U = W / (B1 - ONE)
                        C = B1*U
                        V = DT(I,K)
                        DT(I,K) = U
 3151                   YA = U + YA
C                    ENDDO
                     YM(I) = YA
                     DYM(I) = U
                  ENDIF
 315              CONTINUE
C              ENDDO
               IF (J .NE. 1) THEN
C
C---3.1.6         Convergence Monitor
                  CALL D1SCAL ('INTERNAL',YM,N,Y,YWGT,YMAX,TOL,ONE)
                  ERR = D1ERRN (DYM, N, YWGT)
                  QKONV = ERR .LE. TOL
                  ERR = ERR / TOLH
C
C---              Order Control
                  K = J - 1
                  FC = ERR**(ONE / DBLE(K + J))
                  FCK(K) = FC
C
C---              Order Window
                  IF (J .GE. KOPT .OR. QFIRST .OR. QLAST) THEN
                     IF (QKONV) GOTO 25
C                                Exit 3 for Next Basic Integration Step
C
C---                 Check for Possible Stepsize Reduction
                     RED = ONE / FC
                     QRED = .FALSE.
                     IF (K .EQ. KMACT .OR. K .EQ. JOPT) THEN
                        RED = RED*SAFE
                        QRED = .TRUE.
                     ELSE
                        IF (K .EQ. KOPT) THEN
                           RED = RED*ALPHA(JOPT,KOPT)
                           IF (RED .LT. ONE) THEN
                              RED = ONE / FC
                              QRED = .TRUE.
                           ENDIF
                        ELSE
                           IF (KOPT .EQ. KMACT) THEN
                              RED = RED*ALPHA(KMACT,K)
                              IF (RED .LT. ONE) THEN
                                 RED = RED * SAFE
                                 QRED = .TRUE.
                              ENDIF
                           ELSE
                              RED = RED*ALPHA(JOPT,K)
                              IF (RED .LT. ONE) THEN
                                 RED = ALPHA(KOPT,K) / FC
                                 QRED = .TRUE.
                              ENDIF
                           ENDIF
                        ENDIF
                     ENDIF
                     IF (QRED) GOTO 32
C                              Exit 3.1 to Stepsize Reduction
                  ENDIF
               ENDIF
 31            CONTINUE
C           ENDDO
C
C---3.2     Prepare Stepsize Reduction
 32         CONTINUE
C
C---3.5     Stepsize Reduction
            RED = DMIN1 (RED, RMAX)
            H = H*RED
            IF (NSTEP .GT. 0) QLAST = .FALSE.
            JRED = JRED + 1
            IF (QPRMON) WRITE (LOUT, 13501) JRED,RED,
     2         KFIN,KOPT,KMACT
            IF (JRED .GT. JRMAX) THEN
C              Error 3
               IF (QPRMON) WRITE (LOUT, 13502) JRMAX
               KFLAG = -3
               GOTO  9
C              Exit to Return
            ENDIF
            GOTO  3
C        ENDDO
C
C        ************************************************
C---2.5  Preparations for Next Basic Integration Step
 25      NSTEP = NSTEP + 1
         QFIRST = .FALSE.
         IF (NSTEP .GT. NSTMAX) THEN
C           Error 2
C           Emergency Exit, If too Many Steps Taken
            IF (QPRMON) WRITE (LOUT, 12501) PRODCT, NSTMAX
            KFLAG = -2
            GOTO  9
C           Exit to Return
         ENDIF
C
C---     Restoring
         DO 251 I=1, N
 251        Y(I) = YM(I)
C        ENDDO
         T = TN
         IF (T .EQ. TEND) GOTO 9
C                         Exit to Return
         CALL D1SCAL ('ACCEPTED', Y, N, DUMMY, YWGT, YMAX, TOL, ONE)
C
C---2.7  Order and Stepsize Selection
C
C---2.7.1 Stepsize Restrictions
         HMAX = DMIN1(HMAXU,H/FMIN)
         FCM = H / HMAX
C
C---2.7.2 Optimal Order Determination
         KOPT = 1
         JOPT = 2
         FCO = DMAX1 (FCK(1), FCM)
         OMJO = FCO*AWK(2)
         IF (KFIN .GE. 2) THEN
            DO 272 L=2,KFIN
               JL = L + 1
               FC = DMAX1 (FCK(L), FCM)
               OMJ = FC*AWK(JL)
               IF (OMJ*PCT101 .LE. OMJO .AND. L .LT. KMACT) THEN
                  KOPT = L
                  JOPT = JL
                  OMJO = OMJ
                  FCO = FC
               ENDIF
 272           CONTINUE
C           ENDDO
         ENDIF
         HREST = TEND - T
         HN = H / FCO
C
C---2.7.3 Possible Increase of Order
         IF (HN .LT. HREST) THEN
            IF ((JRED .EQ. 0 .OR. NSTEP .EQ. 0) .AND.
     2           KOPT .GE. KFIN .AND. KOPT .NE. KMACT) THEN
               FC = DMAX1 (FCO/ALPHA(JOPT,KOPT), FCM)
               JL = JOPT + 1
               IF (AWK(JL)*FC*PCT101 .LE. OMJO .AND.
     2               JOPT .LT. KMACT) THEN
                  FCO = FC
                  HN = H / FCO
                  KOPT = JOPT
                  JOPT = JOPT + 1
               ENDIF
            ENDIF
         ENDIF
C
C---2.7.4 Stepsize Selection
         H = HN
         HRTRN = H
         IF (H .GT. HREST*PCT90) THEN
            H = HREST
            QLAST = .TRUE.
         ENDIF
         GO TO  2
C     ENDDO
C
C---9. Exit
 9    HMAX = HMAXU
      IF (KFLAG .LT. 0) THEN
C        Fail Exit
         H = ZERO
      ELSE
C        Solution Exit
         H = HRTRN
         IF (QPRMON) WRITE (LOUT, 12001) NSTEP,NFCN,T,KFIN,KOPT
         IF (QPRSOL) WRITE (LOUT, 12002) NSTEP,NFCN,T,H,(Y(I),I=1,N)
      ENDIF
      RETURN
C
C
10001 FORMAT(//,' ',A8,'  - ERROR -  '
     2      ,   ' Direction of integration is reverse to convention.')
10002 FORMAT(//,' ',A8,'  - WARNING -'
     2      ,   ' Desired tolerance ', D10.3, ' too small.', /,
     3      22X,' Tolerance set to  ', D10.3, '.')
11221 FORMAT(1H0,A8,' - ',A20,/,
     2       1H0,' rel.prec. TOL ',D10.3,' max.col.',I3,
     3       ' Sequence ',(1H ,13I4))
11222 FORMAT(//,5X,4HStep,3X,7HF-Calls,8X,1HT,25X,1HH,5X,7HY1(T)..,//)
12001 FORMAT(1H ,2I9,D20.11,I9,I6)
12002 FORMAT(1H ,2I9,D20.11,D12.5,4D20.11,/,(1H ,50X,4D20.11))
12501 FORMAT(//,' ',A8,'  - ERROR -  '
     2      ,18H More than NSTMAX=,I3,18H integration steps,//)
13001 FORMAT(//,' ',A8,'  - ERROR -  '
     2      ,40H Stepsize reduction failed to succeed  ,//)
13501 FORMAT(1H ,I3,27H Stepsize reduction factor ,D10.3,
     2      ' KFIN',I3,' KOPT',I3,' KMAX',I3)
13502 FORMAT(//,' ',A8,'  - ERROR -  '
     2      ,17H More then JRMAX=,I3,29H stepsize reductions per step,/)
C
C
C End DIFEX1
C
      END
      SUBROUTINE D1SEQ(M,NJ)
      INTEGER I, M, NJ
      DIMENSION NJ(M)
C
C  Set Stepsize Sequence for DIFEX1
C
      NJ(1) = 2
      DO 10 I=2,M
        NJ(I) = NJ(I-1) + 2
 10     CONTINUE
C     ENDDO
      RETURN
      END
      SUBROUTINE D1SCAL (MODE, Y, N, YOLD, YWGT, YMAX, THREL, THABS)
C
C     Scaling for DIFEX1
C
C       (for Real Life Applications to be Altered
C        by the Skillful User)
C
C
C* Parameters:
C
C    MODE      C*8 IN   ='INITIAL '    Initial Scaling
C                       ='INTERNAL'    Scaling during Discretization
C                       ='ACCEPTED'    Rescaling if Step Accepted
C                       Else           Error
C    Y         D   IN   Array of Values Y(1),...,Y(N)
C    N         I   IN   Length of Vectors Y, YOLD, YWGT, and YMAX
C    YOLD      D   IN   Array of Old Values
C    YWGT      D   OUT  Array of Scaled Values New
C    YMAX      D   IN   Array of Maximum Values Old
C                  OUT  Array of Maximum Values New
C    THREL     D   IN   Relative Threshold Value
C    THABS     D   IN   Absolute Threshold Value
C
C* Local Variables:
C
C    YUSER     D   V    User Defined Array of Maximum Values
C
C* Type Declaration
C
      INTEGER I, LOUT, MAXODE, N
C
      DOUBLE PRECISION DABS, DMAX1, EPMACH, ONE, THABS, THREL, U, Y,
     2YMAX, YOLD, YUSER, YWGT, ZERO
C
      CHARACTER MODE*8
C
C* Constants:
C
C    EPMACH    D   K    Relative Machine Precision
C    LOUT      I   K    Output is Written on Logical Unit LOUT
C    MAXODE    I   K    Maximal Number of First-Order ODE's
C    ONE       D   K    1.0
C    ZERO      D   K    0.0
C
      PARAMETER ( EPMACH = 2.22D-16      ,
     2            LOUT   = 6             ,
     3            MAXODE = 51            ,
     4            ONE    = 1.0  D0       ,
     5            ZERO   = 0.0  D0       )
C
      DIMENSION Y(N), YOLD(N), YWGT(N), YMAX(N), YUSER(MAXODE)
      SAVE YUSER
      IF (MODE .EQ.          'INITIAL '         ) THEN
C                             --------
         DO 100 I=1,N
            YUSER(I) = DABS (YMAX(I))
            U = DABS (Y(I))
            IF (U .LT. EPMACH) U = ONE
            YMAX(I) = DMAX1 (U, YUSER(I), THABS)
 100        YWGT(I) = YMAX(I)
C        ENDDO
      ELSE IF (MODE .EQ.     'INTERNAL'         ) THEN
C                             --------
         DO 200 I=1,N
 200        YWGT(I) = DMAX1 (YMAX(I)*THREL, DABS(Y(I)),
     2                       DABS(YOLD(I)), YUSER(I), THABS)
C        ENDDO
      ELSE IF (MODE .EQ.     'ACCEPTED'         ) THEN
C                             --------
         DO 300 I=1,N
 300        YMAX(I) = DMAX1 (YMAX(I), DABS(Y(I)))
C        ENDDO
      ELSE
         WRITE (LOUT, '(//,A,/)')
     2      ' D1SCAL    - ERROR -   Illegal Mode'
      ENDIF
      RETURN
      END
      DOUBLE PRECISION FUNCTION D1ERRN(Y, N, YWGT)
C* Title:
C
C  Scaled Root Mean Square Error
C
C
C* Parameters:
C
C    Y         D   IN   Array of Values Y(1),...,Y(N)
C    N         I   IN   Length of Vectors Y and YWGT
C    YWGT      D   IN   Array of Scaled Values
C
C* Type Declaration
C
      INTEGER I, N
C
      DOUBLE PRECISION DBLE, DSQRT, SUM, Y, YWGT, ZERO
C
C* Constants:
C
C    ZERO      D   K    0
C
      PARAMETER ( ZERO   = 0.0  D0       )
C
      DIMENSION Y(N), YWGT(N)
C
      SUM = ZERO
      DO 100 I=1,N
 100     SUM = SUM + (Y(I) / YWGT(I)) * (Y(I) / YWGT(I))
C     ENDDO
      D1ERRN = DSQRT(SUM / DBLE(N))
      RETURN
      END
C
C*    Group  Sparse Linear System Solver MA28 from HARWELL Library
C
C I AND J ARE IBM FORTRAN DOUBLE AND SINGLE LENGTH VERSIONS  ISDJ/
C D AND S ARE STANDARD FORTRAN DOUBLE AND SINGLE LENGTH VERSIONS
C     SUBROUTINE MC20A (NC,MAXA,A,INUM,JPTR,JNUM,JDISP)              JS/
      SUBROUTINE MC20AD(NC,MAXA,A,INUM,JPTR,JNUM,JDISP)
C
      INTEGER   INUM(MAXA),JNUM(MAXA)                                
C     INTEGER*2 INUM(MAXA),JNUM(MAXA)                                DI/
C     REAL A(MAXA)                                                   JS/
      DOUBLE PRECISION A(MAXA),ACE,ACEP
      DIMENSION JPTR(NC)
C
C     ******************************************************************
C
      NULL=-JDISP
C**      CLEAR JPTR
      DO 60 J=1,NC
   60 JPTR(J)=0
C**      COUNT THE NUMBER OF ELEMENTS IN EACH COLUMN.
      DO 120 K=1,MAXA
      J=JNUM(K)+JDISP
      JPTR(J)=JPTR(J)+1
  120 CONTINUE
C**      SET THE JPTR ARRAY
      K=1
      DO 150 J=1,NC
      KR=K+JPTR(J)
      JPTR(J)=K
  150 K=KR
C
C**      REORDER THE ELEMENTS INTO COLUMN ORDER.  THE ALGORITHM IS AN
C        IN-PLACE SORT AND IS OF ORDER MAXA.
      DO 230 I=1,MAXA
C        ESTABLISH THE CURRENT ENTRY.
      JCE=JNUM(I)+JDISP
      IF(JCE.EQ.0) GO TO 230
      ACE=A(I)
      ICE=INUM(I)
C        CLEAR THE LOCATION VACATED.
      JNUM(I)=NULL
C        CHAIN FROM CURRENT ENTRY TO STORE ITEMS.
      DO 200 J=1,MAXA
C        CURRENT ENTRY NOT IN CORRECT POSITION.  DETERMINE CORRECT
C        POSITION TO STORE ENTRY.
      LOC=JPTR(JCE)
      JPTR(JCE)=JPTR(JCE)+1
C        SAVE CONTENTS OF THAT LOCATION.
      ACEP=A(LOC)
      ICEP=INUM(LOC)
      JCEP=JNUM(LOC)
C        STORE CURRENT ENTRY.
      A(LOC)=ACE
      INUM(LOC)=ICE
      JNUM(LOC)=NULL
C        CHECK IF NEXT CURRENT ENTRY NEEDS TO BE PROCESSED.
      IF(JCEP.EQ.NULL) GO TO 230
C        IT DOES.  COPY INTO CURRENT ENTRY.
      ACE=ACEP
      ICE=ICEP
  200 JCE=JCEP+JDISP
C
  230 CONTINUE
C
C**      RESET JPTR VECTOR.
      JA=1
      DO 250 J=1,NC
      JB=JPTR(J)
      JPTR(J)=JA
  250 JA=JB
      RETURN
      END
C     SUBROUTINE MC20B(NC,MAXA,A,INUM,JPTR)                          JS/
      SUBROUTINE MC20BD(NC,MAXA,A,INUM,JPTR)
C     REAL A(MAXA)                                                   JS/
      DOUBLE PRECISION A(MAXA),ACE
      INTEGER   INUM(MAXA)
C     INTEGER*2 INUM(MAXA)                                           DS/
      DIMENSION JPTR(NC)
C
C     ******************************************************************
C
      KMAX=MAXA
      DO 30 JJ=1,NC
      J=NC+1-JJ
      KLO=JPTR(J)+1
      IF(KLO.GT.KMAX)GO TO 30
      KOR=KMAX
      DO 25 KDUMMY=KLO,KMAX
C ITEMS KOR, KOR+1, .... ,KMAX ARE IN ORDER
      ACE=A(KOR-1)
      ICE=INUM(KOR-1)
      DO 10 K=KOR,KMAX
      IK=INUM(K)
      IF(IABS(ICE).LE.IABS(IK))GO TO 20
      INUM(K-1)=IK
10    A(K-1)=A(K)
      K=KMAX+1
20    INUM(K-1)=ICE
      A(K-1)=ACE
25    KOR=KOR-1
C        NEXT COLUMN
30    KMAX=KLO-2
      RETURN
      END
C I IS THE IBM FORTRAN VERSION                                       IS/
C S IS THE STANDARD FORTRAN VERSION
      SUBROUTINE MC13D(N,ICN,LICN,IP,LENR,IOR,IB,NUM,IW)
      INTEGER IP(N)
      INTEGER ICN(LICN),LENR(N),IOR(N),IB(N),IW(N,3)
C     INTEGER*2 ICN(LICN),LENR(N),IOR(N),IB(N),IW(N,3)                I/
      CALL MC13E(N,ICN,LICN,IP,LENR,IOR,IB,NUM,IW(1,1),IW(1,2),IW(1,3))
      RETURN
      END
      SUBROUTINE MC13E(N,ICN,LICN,IP,LENR,ARP,IB,NUM,LOWL,NUMB,PREV)
      INTEGER STP,DUMMY
      INTEGER IP(N)
C
C ARP(I) IS ONE LESS THAN THE NUMBER OF UNSEARCHED EDGES LEAVING
C     NODE I.  AT THE END OF THE ALGORITHM IT IS SET TO A
C     PERMUTATION WHICH PUTS THE MATRIX IN BLOCK LOWER
C     TRIANGULAR FORM.
C IB(I) IS THE POSITION IN THE ORDERING OF THE START OF THE ITH
C     BLOCK.  IB(N+1-I) HOLDS THE NODE NUMBER OF THE ITH NODE
C     ON THE STACK.
C LOWL(I) IS THE SMALLEST STACK POSITION OF ANY NODE TO WHICH A PATH
C     FROM NODE I HAS BEEN FOUND.  IT IS SET TO N+1 WHEN NODE I
C     IS REMOVED FROM THE STACK.
C NUMB(I) IS THE POSITION OF NODE I IN THE STACK IF IT IS ON
C     IT, IS THE PERMUTED ORDER OF NODE I FOR THOSE NODES
C     WHOSE FINAL POSITION HAS BEEN FOUND AND IS OTHERWISE ZERO.
C PREV(I) IS THE NODE AT THE END OF THE PATH WHEN NODE I WAS
C     PLACED ON THE STACK.
      INTEGER ICN(LICN),LENR(N),ARP(N),IB(N),LOWL(N),NUMB(N),          
     1PREV(N)                                                          
C     INTEGER *2 ICN(LICN),LENR(N),ARP(N),IB(N),LOWL(N),NUMB(N),     I/
C    1PREV(N)                                                        I/
C
C
C   ICNT IS THE NUMBER OF NODES WHOSE POSITIONS IN FINAL ORDERING HAVE
C     BEEN FOUND.
      ICNT=0
C NUM IS THE NUMBER OF BLOCKS THAT HAVE BEEN FOUND.
      NUM=0
      NNM1=N+N-1
C
C INITIALIZATION OF ARRAYS.
      DO 20 J=1,N
      NUMB(J)=0
      ARP(J)=LENR(J)-1
   20 CONTINUE
C
C
      DO 120 ISN=1,N
C LOOK FOR A STARTING NODE
      IF (NUMB(ISN).NE.0) GO TO 120
      IV=ISN
C IST IS THE NUMBER OF NODES ON THE STACK ... IT IS THE STACK POINTER.
      IST=1
C PUT NODE IV AT BEGINNING OF STACK.
      LOWL(IV)=1
      NUMB(IV)=1
      IB(N)=IV
C
C THE BODY OF THIS LOOP PUTS A NEW NODE ON THE STACK OR BACKTRACKS.
      DO 110 DUMMY=1,NNM1
      I1=ARP(IV)
C HAVE ALL EDGES LEAVING NODE IV BEEN SEARCHED.
      IF (I1.LT.0) GO TO 60
      I2=IP(IV)+LENR(IV)-1
      I1=I2-I1
C
C LOOK AT EDGES LEAVING NODE IV UNTIL ONE ENTERS A NEW NODE OR
C     ALL EDGES ARE EXHAUSTED.
      DO 50 II=I1,I2
      IW=ICN(II)
C HAS NODE IW BEEN ON STACK ALREADY.
      IF (NUMB(IW).EQ.0) GO TO 100
C UPDATE VALUE OF LOWL(IV) IF NECESSARY.
C 50  LOWL(IV)=MIN0(LOWL(IV),LOWL(IW))                      WEI;      S/
   50 IF (LOWL(IW).LT.LOWL(IV)) LOWL(IV)=LOWL(IW)
C
C THERE ARE NO MORE EDGES LEAVING NODE IV.
      ARP(IV)=-1
C IS NODE IV THE ROOT OF A BLOCK.
   60 IF (LOWL(IV).LT.NUMB(IV)) GO TO 90
C
C ORDER NODES IN A BLOCK.
      NUM=NUM+1
      IST1=N+1-IST
      LCNT=ICNT+1
C PEEL BLOCK OFF THE TOP OF THE STACK STARTING AT THE TOP AND
C     WORKING DOWN TO THE ROOT OF THE BLOCK.
      DO 70 STP=IST1,N
      IW=IB(STP)
      LOWL(IW)=N+1
      ICNT=ICNT+1
      NUMB(IW)=ICNT
      IF (IW.EQ.IV) GO TO 80
   70 CONTINUE
   80 IST=N-STP
      IB(NUM)=LCNT
C ARE THERE ANY NODES LEFT ON THE STACK.
      IF (IST.NE.0) GO TO 90
C HAVE ALL THE NODES BEEN ORDERED.
      IF (ICNT.LT.N) GO TO 120
      GO TO 130
C
C BACKTRACK TO PREVIOUS NODE ON PATH.
   90 IW=IV
      IV=PREV(IV)
C UPDATE VALUE OF LOWL(IV) IF NECESSARY.
C     LOWL(IV)=MIN0(LOWL(IV),LOWL(IW))                  WEI;          S/
      IF (LOWL(IW).LT.LOWL(IV)) LOWL(IV)=LOWL(IW)
      GO TO 110
C
C PUT NEW NODE ON THE STACK.
 100  ARP(IV)=I2-II-1
      PREV(IW)=IV
      IV=IW
      IST=IST+1
      LOWL(IV)=IST
      NUMB(IV)=IST
      K=N+1-IST
      IB(K)=IV
  110 CONTINUE
C
  120 CONTINUE
C
C
C PUT PERMUTATION IN THE REQUIRED FORM.
  130 DO 140 I=1,N
      II=NUMB(I)
 140  ARP(II)=I
      RETURN
      END
C I AND J ARE IBM FORTRAN SINGLE AND DOUBLE PRECISION CODES RESP.  JISD/
C S AND D ARE STANDARD FORTRAN SINGLE AND DOUBLE PRECISION CODES RESP.
C     SUBROUTINE MC22A(N,ICN,A,NZ,LENROW,IP,IQ,IW,IW1)               IS/
      SUBROUTINE MC22AD(N,ICN,A,NZ,LENROW,IP,IQ,IW,IW1)
C     REAL A(NZ)                                                     IS/
      DOUBLE PRECISION A(NZ),AVAL
      INTEGER IW(N,2)
      INTEGER   ICN(NZ),LENROW(N),IP(N),IQ(N),IW1(NZ)
C     INTEGER*2 ICN(NZ),LENROW(N),IP(N),IQ(N),IW1(NZ)                ID/
      IF (NZ.LE.0) GO TO 1000
      IF (N.LE.0) GO TO 1000
C SET START OF ROW I IN IW(I,1) AND LENROW(I) IN IW(I,2)
      IW(1,1)=1
      IW(1,2)=LENROW(1)
      DO 10 I=2,N
      IW(I,1)=IW(I-1,1)+LENROW(I-1)
 10   IW(I,2)=LENROW(I)
C PERMUTE LENROW ACCORDING TO IP.  SET OFF-SETS FOR NEW POSITION
C     OF ROW IOLD IN IW(IOLD,1) AND PUT OLD ROW INDICES IN IW1 IN
C     POSITIONS CORRESPONDING TO THE NEW POSITION OF THIS ROW IN A/ICN.
      JJ=1
      DO 20 I=1,N
      IOLD=IP(I)
      IOLD=IABS(IOLD)
      LENGTH=IW(IOLD,2)
      LENROW(I)=LENGTH
      IF (LENGTH.EQ.0) GO TO 20
      IW(IOLD,1)=IW(IOLD,1)-JJ
      J2=JJ+LENGTH-1
      DO 15 J=JJ,J2
 15   IW1(J)=IOLD
      JJ=J2+1
 20   CONTINUE
C SET INVERSE PERMUTATION TO IQ IN IW(.,2).
      DO 30 I=1,N
      IOLD=IQ(I)
      IOLD=IABS(IOLD)
 30   IW(IOLD,2)=I
C PERMUTE A AND ICN IN PLACE, CHANGING TO NEW COLUMN NUMBERS.
C
C ***   MAIN LOOP   ***
C EACH PASS THROUGH THIS LOOP PLACES A CLOSED CHAIN OF COLUMN INDICES
C     IN THEIR NEW (AND FINAL) POSITIONS ... THIS IS RECORDED BY
C     SETTING THE IW1 ENTRY TO ZERO SO THAT ANY WHICH ARE SUBSEQUENTLY
C     ENCOUNTERED DURING THIS MAJOR SCAN CAN BE BYPASSED.
      DO 200 I=1,NZ
      IOLD=IW1(I)
      IF (IOLD.EQ.0) GO TO 200
      IPOS=I
      JVAL=ICN(I)
C IF ROW IOLD IS IN SAME POSITIONS AFTER PERMUTATION GO TO 150.
      IF (IW(IOLD,1).EQ.0) GO TO 150
      AVAL=A(I)
C **  CHAIN LOOP  **
C EACH PASS THROUGH THIS LOOP PLACES ONE (PERMUTED) COLUMN INDEX
C     IN ITS FINAL POSITION  .. VIZ. IPOS.
      DO 100 ICHAIN=1,NZ
C NEWPOS IS THE ORIGINAL POSITION IN A/ICN OF THE ELEMENT TO BE PLACED
C IN POSITION IPOS.  IT IS ALSO THE POSITION OF THE NEXT ELEMENT IN
C     THE CHAIN.
      NEWPOS=IPOS+IW(IOLD,1)
C IS CHAIN COMPLETE ?
      IF (NEWPOS.EQ.I) GO TO 130
      A(IPOS)=A(NEWPOS)
      JNUM=ICN(NEWPOS)
      ICN(IPOS)=IW(JNUM,2)
      IPOS=NEWPOS
      IOLD=IW1(IPOS)
      IW1(IPOS)=0
C **  END OF CHAIN LOOP  **
 100  CONTINUE
 130  A(IPOS)=AVAL
 150  ICN(IPOS)=IW(JVAL,2)
C ***   END OF MAIN LOOP   ***
 200  CONTINUE
C
 1000 RETURN
      END
C J AND I ARE IBM DOUBLE AND SINGLE VERSIONS.    JISD/
C D AND S ARE STANDARD FORTRAN DOUBLE AND SINGLE VERSIONS.
C     SUBROUTINE MC23A(N,ICN,A,LICN,LENR,IDISP,IP,IQ,LENOFF,IW,IW1)  IS/
      SUBROUTINE MC23AD(N,ICN,A,LICN,LENR,IDISP,IP,IQ,LENOFF,IW,IW1)
C     REAL A(LICN)                                                   IS/
      DOUBLE PRECISION A(LICN)
      INTEGER IDISP(2),IW1(N,2)
      LOGICAL ABORT
      INTEGER   ICN(LICN),LENR(N),IP(N),IQ(N),LENOFF(N),IW(N,5)
C     INTEGER*2 ICN(LICN),LENR(N),IP(N),IQ(N),LENOFF(N),IW(N,5)      ID/
C INPUT ... N,ICN .. A,ICN,LENR ....
C
C SET UP POINTERS IW(.,1) TO THE BEGINNING OF THE ROWS AND SET LENOFF
C     EQUAL TO LENR.
C     COMMON /MC23B/ LP,NUMNZ,NUM,LARGE,ABORT                        IS/
      COMMON /MC23BD/ LP,NUMNZ,NUM,LARGE,ABORT
      SAVE /MC23BD/
      IW1(1,1)=1
      LENOFF(1)=LENR(1)
      IF (N.EQ.1) GO TO 20
      DO 10 I=2,N
      LENOFF(I)=LENR(I)
   10 IW1(I,1)=IW1(I-1,1)+LENR(I-1)
C IDISP(1) POINTS TO THE FIRST POSITION IN A/ICN AFTER THE
C     OFF-DIAGONAL BLOCKS AND UNTREATED ROWS.
   20 IDISP(1)=IW1(N,1)+LENR(N)
C
C FIND ROW PERMUTATION IP TO MAKE DIAGONAL ZERO-FREE.
      CALL MC21A(N,ICN,LICN,IW1,LENR,IP,NUMNZ,IW)
C
C POSSIBLE ERROR RETURN FOR STRUCTURALLY SINGULAR MATRICES.
      IF (NUMNZ.NE.N.AND.ABORT) GO TO 170
C
C IW1(.,2) AND LENR ARE PERMUTATIONS OF IW1(.,1) AND LENR/LENOFF
C     SUITABLE FOR ENTRY
C     TO MC13D SINCE MATRIX WITH THESE ROW POINTER AND LENGTH ARRAYS
C     HAS MAXIMUM NUMBER OF NON-ZEROS ON THE DIAGONAL.
      DO 30 II=1,N
      I=IP(II)
      IW1(II,2)=IW1(I,1)
   30 LENR(II)=LENOFF(I)
C
C FIND SYMMETRIC PERMUTATION IQ TO BLOCK LOWER TRIANGULAR FORM.
      CALL MC13D(N,ICN,LICN,IW1(1,2),LENR,IQ,IW(1,4),NUM,IW)
C
      IF (NUM.NE.1) GO TO 60
C
C ACTION TAKEN IF MATRIX IS IRREDUCIBLE.
C WHOLE MATRIX IS JUST MOVED TO THE END OF THE STORAGE.
      DO 40 I=1,N
      LENR(I)=LENOFF(I)
      IP(I)=I
   40 IQ(I)=I
      LENOFF(1)=-1
C IDISP(1) IS THE FIRST POSITION AFTER THE LAST ELEMENT IN THE
C     OFF-DIAGONAL BLOCKS AND UNTREATED ROWS.
      NZ=IDISP(1)-1
      IDISP(1)=1
C IDISP(2) IS THE POSITION IN A/ICN OF THE FIRST ELEMENT IN THE
C     DIAGONAL BLOCKS.
      IDISP(2)=LICN-NZ+1
      LARGE=N
      IF (NZ.EQ.LICN) GO TO 230
      DO 50 K=1,NZ
      J=NZ-K+1
      JJ=LICN-K+1
      A(JJ)=A(J)
   50 ICN(JJ)=ICN(J)
C 230 = RETURN
      GO TO 230
C
C DATA STRUCTURE REORDERED.
C
C FORM COMPOSITE ROW PERMUTATION ... IP(I) = IP(IQ(I)).
   60 DO 70 II=1,N
      I=IQ(II)
   70 IW(II,1)=IP(I)
      DO 80 I=1,N
   80 IP(I)=IW(I,1)
C
C RUN THROUGH BLOCKS IN REVERSE ORDER SEPARATING DIAGONAL BLOCKS
C     WHICH ARE MOVED TO THE END OF THE STORAGE.  ELEMENTS IN
C     OFF-DIAGONAL BLOCKS ARE LEFT IN PLACE UNLESS A COMPRESS IS
C     NECESSARY.
C
C IBEG INDICATES THE LOWEST VALUE OF J FOR WHICH ICN(J) HAS BEEN
C     SET TO ZERO WHEN ELEMENT IN POSITION J WAS MOVED TO THE
C     DIAGONAL BLOCK PART OF STORAGE.
      IBEG=LICN+1
C IEND IS THE POSITION OF THE FIRST ELEMENT OF THOSE TREATED ROWS
C     WHICH ARE IN DIAGONAL BLOCKS.
      IEND=LICN+1
C LARGE IS THE DIMENSION OF THE LARGEST BLOCK ENCOUNTERED SO FAR.
      LARGE=0
C
C NUM IS THE NUMBER OF DIAGONAL BLOCKS.
      DO 150 K=1,NUM
      IBLOCK=NUM-K+1
C I1 IS FIRST ROW (IN PERMUTED FORM) OF BLOCK IBLOCK.
C I2 IS LAST ROW (IN PERMUTED FORM) OF BLOCK IBLOCK.
      I1=IW(IBLOCK,4)
      I2=N
      IF (K.NE.1) I2=IW(IBLOCK+1,4)-1
      LARGE=MAX0(LARGE,I2-I1+1)
C GO THROUGH THE ROWS OF BLOCK IBLOCK IN THE REVERSE ORDER.
      DO 140 II=I1,I2
      INEW=I2-II+I1
C WE NOW DEAL WITH ROW INEW IN PERMUTED FORM (ROW IOLD IN ORIGINAL
C     MATRIX).
      IOLD=IP(INEW)
C IF THERE IS SPACE TO MOVE UP DIAGONAL BLOCK PORTION OF ROW GO TO 110
      IF (IEND-IDISP(1).GE.LENOFF(IOLD)) GO TO 110
C
C IN-LINE COMPRESS.
C MOVES SEPARATED OFF-DIAGONAL ELEMENTS AND UNTREATED ROWS TO
C     FRONT OF STORAGE.
      JNPOS=IBEG
      ILEND=IDISP(1)-1
      IF (ILEND.LT.IBEG) GO TO 190
      DO 90 J=IBEG,ILEND
      IF (ICN(J).EQ.0) GO TO 90
      ICN(JNPOS)=ICN(J)
      A(JNPOS)=A(J)
      JNPOS=JNPOS+1
   90 CONTINUE
      IDISP(1)=JNPOS
      IF (IEND-JNPOS.LT.LENOFF(IOLD)) GO TO 190
      IBEG=LICN+1
C RESET POINTERS TO THE BEGINNING OF THE ROWS.
      DO 100 I=2,N
  100 IW1(I,1)=IW1(I-1,1)+LENOFF(I-1)
C
C ROW IOLD IS NOW SPLIT INTO DIAG. AND OFF-DIAG. PARTS.
  110 IROWB=IW1(IOLD,1)
      LENI=0
      IROWE=IROWB+LENOFF(IOLD)-1
C BACKWARD SCAN OF WHOLE OF ROW IOLD (IN ORIGINAL MATRIX).
      IF (IROWE.LT.IROWB) GO TO 130
      DO 120 JJ=IROWB,IROWE
      J=IROWE-JJ+IROWB
      JOLD=ICN(J)
C IW(.,2) HOLDS THE INVERSE PERMUTATION TO IQ.
C     ..... IT WAS SET TO THIS IN MC13D.
      JNEW=IW(JOLD,2)
C IF (JNEW.LT.I1) THEN ....
C ELEMENT IS IN OFF-DIAGONAL BLOCK AND SO IS LEFT IN SITU.
      IF (JNEW.LT.I1) GO TO 120
C ELEMENT IS IN DIAGONAL BLOCK AND IS MOVED TO THE END OF THE STORAGE.
      IEND=IEND-1
      A(IEND)=A(J)
      ICN(IEND)=JNEW
      IBEG=MIN0(IBEG,J)
      ICN(J)=0
      LENI=LENI+1
  120 CONTINUE
C
      LENOFF(IOLD)=LENOFF(IOLD)-LENI
  130 LENR(INEW)=LENI
  140 CONTINUE
C
      IP(I2)=-IP(I2)
  150 CONTINUE
C RESETS IP(N) TO POSITIVE VALUE.
      IP(N)=-IP(N)
C IDISP(2) IS POSITION OF FIRST ELEMENT IN DIAGONAL BLOCKS.
      IDISP(2)=IEND
C
C THIS COMPRESS IS USED TO MOVE ALL OFF-DIAGONAL ELEMENTS TO THE
C     FRONT OF THE STORAGE.
      IF (IBEG.GT.LICN) GO TO 230
      JNPOS=IBEG
      ILEND=IDISP(1)-1
      DO 160 J=IBEG,ILEND
      IF (ICN(J).EQ.0) GO TO 160
      ICN(JNPOS)=ICN(J)
      A(JNPOS)=A(J)
      JNPOS=JNPOS+1
  160 CONTINUE
C IDISP(1) IS FIRST POSITION AFTER LAST ELEMENT OF OFF-DIAGONAL BLOCKS.
      IDISP(1)=JNPOS
      GO TO 230
C
C
C ERROR RETURN
  170 IF (LP.NE.0) WRITE(LP,180) NUMNZ
  180 FORMAT(33X,41H MATRIX IS STRUCTURALLY SINGULAR, RANK = ,I6)
      IDISP(1)=-1
      GO TO 210
  190 IF (LP.NE.0) WRITE(LP,200) N
  200 FORMAT(33X,33H LICN NOT BIG ENOUGH INCREASE BY ,I6)
      IDISP(1)=-2
  210 IF (LP.NE.0) WRITE(LP,220)
C
C 220 FORMAT(33H+ERROR RETURN FROM MC23A  BECAUSE)                   IS/
  220 FORMAT(33H+ERROR RETURN FROM MC23AD BECAUSE)
  230 RETURN
      END
C J AND I ARE IBM DOUBLE AND SINGLE VERSIONS      JISD/
C D AND S ARE STANDARD FORTRAN DOUBLE AND SINGLE VERSIONS.
C     SUBROUTINE MC24A(N,ICN,A,LICN,LENR,LENRL,W)  IS/
      SUBROUTINE MC24AD(N,ICN,A,LICN,LENR,LENRL,W)
C     REAL A(LICN),W(N)  IS/
      DOUBLE PRECISION A(LICN),W(N),AMAXL,WROWL,AMAXU,ZERO
      INTEGER   ICN(LICN),LENR(N),LENRL(N)
C     INTEGER*2 ICN(LICN),LENR(N),LENRL(N)                           ID/
C     DATA ZERO/0.0E0/  IS/
      DATA ZERO/0.0D0/
      AMAXL=ZERO
      DO 10 I=1,N
 10   W(I)=ZERO
      J0=1
      DO 100 I=1,N
      IF (LENR(I).EQ.0) GO TO 100
      J2=J0+LENR(I)-1
      IF (LENRL(I).EQ.0) GO TO 50
C CALCULATION OF 1-NORM OF L.
      J1=J0+LENRL(I)-1
      WROWL=ZERO
      DO 30 JJ=J0,J1
C30   WROWL=WROWL+ABS(A(JJ))  IS/
 30   WROWL=WROWL+DABS(A(JJ))
C AMAXL IS THE MAXIMUM NORM OF COLUMNS OF L SO FAR FOUND.
C     AMAXL=AMAX1(AMAXL,WROWL)  IS/
      AMAXL=DMAX1(AMAXL,WROWL)
      J0=J1+1
C CALCULATION OF NORMS OF COLUMNS OF U (MAX-NORMS).
 50   J0=J0+1
      IF (J0.GT.J2) GO TO 90
      DO 80 JJ=J0,J2
      J=ICN(JJ)
C80   W(J)=AMAX1(ABS(A(JJ)),W(J))  IS/
 80   W(J)=DMAX1(DABS(A(JJ)),W(J))
 90   J0=J2+1
 100  CONTINUE
C AMAXU IS SET TO MAXIMUM MAX-NORM OF COLUMNS OF U.
      AMAXU=ZERO
      DO 200 I=1,N
C200  AMAXU=AMAX1(AMAXU,W(I))  IS/
 200  AMAXU=DMAX1(AMAXU,W(I))
C GROFAC IS MAX U MAX-NORM TIMES MAX L 1-NORM.
      W(1)=AMAXL*AMAXU
      RETURN
      END
C I AND J ARE IBM FORTRAN SINGLE AND DOUBLE PRECISION CODES JISD/
C S AND D ARE STANDARD FORTRAN SINGLE AND DOUBLE PRECISION CODES RESP.
C     SUBROUTINE MA30A(NN,ICN,A,LICN,LENR,LENRL,IDISP,IP,IQ,IRN,  IS/
C    1LIRN,LENC,IFIRST,LASTR,NEXTR,LASTC,NEXTC,IPTR,IPC,U,IFLAG) IS/
      SUBROUTINE MA30AD(NN,ICN,A,LICN,LENR,LENRL,IDISP,IP,IQ,IRN,
     1LIRN,LENC,IFIRST,LASTR,NEXTR,LASTC,NEXTC,IPTR,IPC,U,IFLAG)
C     REAL A(LICN)                                                   IS/
      DOUBLE PRECISION A(LICN),U,AU,UMAX,AMAX,ZERO
      INTEGER IPTR(NN),PIVOT,PIVEND,DISPC,OLDPIV,OLDEND,PIVROW
      INTEGER ROWI
      INTEGER IPC(NN),IDISP(2)
      LOGICAL ABORT1,ABORT2,ABORT3
      INTEGER   ICN(LICN),LENR(NN),LENRL(NN),IP(NN),IQ(NN),LENC(NN),
     1IRN(LIRN)
C     INTEGER*2 ICN(LICN),LENR(NN),LENRL(NN),IP(NN),IQ(NN),LENC(NN), ID/
C    1IRN(LIRN)                                                      ID/
      INTEGER   IFIRST(NN),LASTR(NN),NEXTR(NN),LASTC(NN),NEXTC(NN)
C     INTEGER*2 IFIRST(NN),LASTR(NN),NEXTR(NN),LASTC(NN),NEXTC(NN)   ID/
C     COMMON /MA30E/ LP,ABORT1,ABORT2,ABORT3  IS/
      COMMON /MA30ED/ LP,ABORT1,ABORT2,ABORT3
C     COMMON /MA30F/ IRNCP,ICNCP,IRANK,MINIRN,MINICN  IS/
      COMMON /MA30FD/ IRNCP,ICNCP,IRANK,MINIRN,MINICN
      SAVE /MA30ED/,/MA30FD/
C     DATA UMAX/.9999/                                               IS/
      DATA UMAX/.999999999D0/
C     DATA ZERO/0.0/                                                 IS/
      DATA ZERO/0.0D0/
C
      MINIRN=0
      MINICN=IDISP(1)-1
      MOREI=0
      IRANK=NN
      IRNCP=0
      ICNCP=0
      IFLAG=0
C RESET U IF NECESSARY.
C     U=AMIN1(U,UMAX)                                                IS/
      U=DMIN1(U,UMAX)
C     U=AMAX1(U,ZERO)                                                IS/
      U=DMAX1(U,ZERO)
C IBEG IS THE POSITION OF THE NEXT PIVOT ROW AFTER ELIMINATION STEP
C     USING IT.
      IBEG=IDISP(1)
C IACTIV IS THE POSITION OF THE FIRST ENTRY IN THE ACTIVE PART OF A/ICN.
      IACTIV=IDISP(2)
C NZROW IS CURRENT NUMBER OF NON-ZEROS IN ACTIVE AND UNPROCESSED PART
C     OF ROW FILE ICN.
      NZROW=LICN-IACTIV+1
      MINICN=NZROW+MINICN
C
C COUNT THE NUMBER OF DIAGONAL BLOCKS AND SET UP POINTERS TO THE
C     BEGINNINGS OF THE ROWS.
C NUM IS THE NUMBER OF DIAGONAL BLOCKS.
      NUM=1
      IPTR(1)=IACTIV
      IF (NN.EQ.1) GO TO 20
      NNM1=NN-1
      DO 10 I=1,NNM1
      IF (IP(I).LT.0) NUM=NUM+1
   10 IPTR(I+1)=IPTR(I)+LENR(I)
C ILAST IS THE LAST ROW IN THE PREVIOUS BLOCK.
   20 ILAST=0
C
C ***********************************************
C ****    LU DECOMPOSITION OF BLOCK NBLOCK   ****
C ***********************************************
C
C EACH PASS THROUGH THIS LOOP PERFORMS LU DECOMPOSITION ON ONE
C     OF THE DIAGONAL BLOCKS.
      DO 950 NBLOCK=1,NUM
      ISTART=ILAST+1
      DO 30 IROWS=ISTART,NN
      IF (IP(IROWS).LT.0) GO TO 40
   30 CONTINUE
      IROWS=NN
   40 ILAST=IROWS
C N IS THE NUMBER OF ROWS IN THE CURRENT BLOCK.
C ISTART IS THE INDEX OF THE FIRST ROW IN THE CURRENT BLOCK.
C ILAST IS THE INDEX OF THE LAST ROW IN THE CURRENT BLOCK.
C IACTIV IS THE POSITION OF THE FIRST ELEMENT IN THE BLOCK.
C ITOP IS THE POSITION OF THE LAST ELEMENT IN THE BLOCK.
      N=ILAST-ISTART+1
      IF (N.NE.1) GO TO 100
C
C CODE FOR DEALING WITH 1X1 BLOCK.
      LENRL(ILAST)=0
      ISING=ISTART
      IF (LENR(ILAST).NE.0) GO TO 60
C BLOCK IS STRUCTURALLY SINGULAR.
      IRANK=IRANK-1
      ISING=-ISING
      IF (IFLAG.NE.2.AND.IFLAG.NE.-5) IFLAG=1
      IF (.NOT.ABORT1) GO TO 90
      IDISP(2)=IACTIV
      IFLAG=-1
      IF (LP.NE.0) WRITE(LP,50)
   50 FORMAT(67H ERROR RETURN FROM MA30A/AD BECAUSE MATRIX IS STRUCTURAL
     1LY SINGULAR)
C     RETURN
      GO TO 1110
   60 IF (A(IACTIV).NE.ZERO) GO TO 80
      ISING=-ISING
      IRANK=IRANK-1
      IPTR(ILAST)=0
      IF (IFLAG.NE.-5) IFLAG=2
      IF (.NOT.ABORT2) GO TO 80
      IDISP(2)=IACTIV
      IFLAG=-2
      IF (LP.NE.0) WRITE(LP,70)
   70 FORMAT(66H ERROR RETURN FROM MA30A/AD BECAUSE MATRIX IS NUMERICALL
     1Y SINGULAR)
      GO TO 1110
   80 A(IBEG)=A(IACTIV)
      ICN(IBEG)=ICN(IACTIV)
      IACTIV=IACTIV+1
      IPTR(ISTART)=0
      IBEG=IBEG+1
      NZROW=NZROW-1
   90 LASTR(ISTART)=ISTART
      LASTC(ISTART)=ISING
      GO TO 950
C
C NON-TRIVIAL BLOCK.
  100 ITOP=LICN
      IF (ILAST.NE.NN) ITOP=IPTR(ILAST+1)-1
C
C SET UP COLUMN ORIENTED STORAGE.
      DO 110 I=ISTART,ILAST
      LENRL(I)=0
  110 LENC(I)=0
      IF (ITOP-IACTIV.LT.LIRN) GO TO 120
      MINIRN=ITOP-IACTIV+1
      PIVOT=ISTART-1
      GO TO 1050
C
C CALCULATE COLUMN COUNTS.
  120 DO 130 II=IACTIV,ITOP
      I=ICN(II)
  130 LENC(I)=LENC(I)+1
C SET UP COLUMN POINTERS SO THAT IPC(J) POINTS TO POSITION AFTER END
C     OF COLUMN J IN COLUMN FILE.
      IPC(ILAST)=LIRN+1
      J1=ISTART+1
      DO 140 JJ=J1,ILAST
      J=ILAST-JJ+J1-1
  140 IPC(J)=IPC(J+1)-LENC(J+1)
      DO 160 INDROW=ISTART,ILAST
      J1=IPTR(INDROW)
      J2=J1+LENR(INDROW)-1
      IF (J1.GT.J2) GO TO 160
      DO 150 JJ=J1,J2
      J=ICN(JJ)
      IPOS=IPC(J)-1
      IRN(IPOS)=INDROW
      IPC(J)=IPOS
  150 CONTINUE
  160 CONTINUE
C DISPC IS THE LOWEST INDEXED ACTIVE LOCATION IN THE COLUMN FILE.
      DISPC=IPC(ISTART)
      NZCOL=LIRN-DISPC+1
      MINIRN=MAX0(NZCOL,MINIRN)
      NZMIN=1
C
C INITIALIZE ARRAY IFIRST.  IFIRST(I) = +/- K INDICATES THAT ROW/COL
C     K HAS I NON-ZEROS.  IF IFIRST(I) = 0, THERE IS NO ROW OR COLUMN
C     WITH I NON ZEROS.
      DO 170 I=1,N
  170 IFIRST(I)=0
C
C COMPUTE ORDERING OF ROW AND COLUMN COUNTS.
C FIRST RUN THROUGH COLUMNS (FROM COLUMN N TO COLUMN 1).
      DO 190 JJ=ISTART,ILAST
      J=ILAST-JJ+ISTART
      NZ=LENC(J)
      IF (NZ.NE.0) GO TO 180
      IPC(J)=0
      LASTC(J)=0
      GO TO 190
  180 ISW=IFIRST(NZ)
      IFIRST(NZ)=-J
      LASTC(J)=0
      NEXTC(J)=-ISW
      ISW1=IABS(ISW)
      IF (ISW.NE.0) LASTC(ISW1)=J
  190 CONTINUE
C NOW RUN THROUGH ROWS (AGAIN FROM N TO 1).
      DO 210 II=ISTART,ILAST
      I=ILAST-II+ISTART
      NZ=LENR(I)
      IF (NZ.NE.0) GO TO 200
      IPTR(I)=0
      LASTR(I)=0
      GO TO 210
  200 ISW=IFIRST(NZ)
      IFIRST(NZ)=I
      IF (ISW.GT.0) GO TO 205
      NEXTR(I)=0
      LASTR(I)=ISW
      GO TO 210
 205  NEXTR(I)=ISW
      LASTR(I)=LASTR(ISW)
      LASTR(ISW)=I
  210 CONTINUE
C
C
C **********************************************
C ****    START OF MAIN ELIMINATION LOOP    ****
C **********************************************
      DO 930 PIVOT=ISTART,ILAST
C
C FIRST FIND THE PIVOT USING MARKOWITZ CRITERION WITH STABILITY
C     CONTROL.
C JCOST IS THE MARKOWITZ COST OF THE BEST PIVOT SO FAR,.. THIS
C     PIVOT IS IN ROW IPIV AND COLUMN JPIV.
      NZ2=NZMIN
      JCOST=N*N
C
C EXAMINE ROWS/COLUMNS IN ORDER OF ASCENDING COUNT.
      DO 290 L=1,2
      LL=L
C A PASS WITH L EQUAL TO 2 IS ONLY PERFORMED IN THE CASE OF SINGULARITY.
      DO 280 NZ=NZ2,N
      IF (JCOST.LE.(NZ-1)**2) GO TO 380
      IJFIR=IFIRST(NZ)
      IF (IJFIR) 212,211,215
 211  IF (LL.EQ.1) NZMIN=NZ+1
      GO TO 280
 212  LL=2
      IJFIR=-IJFIR
      GO TO 245
 215  LL=2
C SCAN ROWS WITH NZ NON-ZEROS.
      DO 235 IDUMMY=1,N
      IF (IJFIR.EQ.0) GO TO 240
C ROW IJFIR IS NOW EXAMINED.
      I=IJFIR
      IJFIR=NEXTR(I)
C FIRST CALCULATE MULTIPLIER THRESHOLD LEVEL.
      AMAX=ZERO
      J1=IPTR(I)+LENRL(I)
      J2=IPTR(I)+LENR(I)-1
      DO 220 JJ=J1,J2
C220  AMAX=AMAX1(AMAX,ABS(A(JJ)))                                    IS/
  220 AMAX=DMAX1(AMAX,DABS(A(JJ)))
      AU=AMAX*U
C SCAN ROW FOR POSSIBLE PIVOTS
      DO 230 JJ=J1,J2
C     IF (ABS(A(JJ)).LE.AU.AND.L.EQ.1) GO TO 230                     IS/
      IF (DABS(A(JJ)).LE.AU.AND.L.EQ.1) GO TO 230
      J=ICN(JJ)
      KCOST=(NZ-1)*(LENC(J)-1)
      IF (KCOST.GE.JCOST) GO TO 230
C BEST PIVOT SO FAR IS FOUND.
      JCOST=KCOST
      IJPOS=JJ
      IPIV=I
      JPIV=J
      IF (JCOST.LE.(NZ-1)**2) GO TO 380
  230 CONTINUE
 235  CONTINUE
C
C COLUMNS WITH NZ NON-ZEROS NOW EXAMINED.
 240  IJFIR=IFIRST(NZ)
      IJFIR=-LASTR(IJFIR)
 245  IF (JCOST.LE.NZ*(NZ-1)) GO TO 380
      DO 270 IDUMMY=1,N
      IF (IJFIR.EQ.0) GO TO 280
      J=IJFIR
      IJFIR=NEXTC(IJFIR)
      I1=IPC(J)
      I2=I1+NZ-1
C SCAN COLUMN J.
      DO 260 II=I1,I2
      I=IRN(II)
      KCOST=(NZ-1)*(LENR(I)-LENRL(I)-1)
      IF (KCOST.GE.JCOST) GO TO 260
C PIVOT HAS BEST MARKOWITZ COUNT SO FAR ... NOW CHECK ITS
C     SUITABILITY ON NUMERIC GROUNDS BY EXAMINING THE OTHER NON-ZEROS
C     IN ITS ROW.
      J1=IPTR(I)+LENRL(I)
      J2=IPTR(I)+LENR(I)-1
C WE NEED A STABILITY CHECK ON SINGLETON COLUMNS BECAUSE OF POSSIBLE
C     PROBLEMS WITH UNDERDETERMINED SYSTEMS.
      AMAX=ZERO
      DO 250 JJ=J1,J2
C     AMAX=AMAX1(AMAX,ABS(A(JJ)))                                    IS/
      AMAX=DMAX1(AMAX,DABS(A(JJ)))
  250 IF (ICN(JJ).EQ.J) JPOS=JJ
C     IF (ABS(A(JPOS)).LE.AMAX*U.AND.L.EQ.1) GO TO 260               IS/
      IF (DABS(A(JPOS)).LE.AMAX*U.AND.L.EQ.1) GO TO 260
      JCOST=KCOST
      IPIV=I
      JPIV=J
      IJPOS=JPOS
      IF (JCOST.LE.NZ*(NZ-1)) GO TO 380
  260 CONTINUE
C
  270 CONTINUE
C
  280 CONTINUE
C
C MATRIX IS NUMERICALLY OR STRUCTURALLY SINGULAR  ... WHICH IT IS WILL
C     BE DIAGNOSED LATER.
      IRANK=IRANK-1
  290 CONTINUE
C ASSIGN REST OF ROWS AND COLUMNS TO ORDERING ARRAY.
C MATRIX IS STRUCTURALLY SINGULAR.
      IF (IFLAG.NE.2.AND.IFLAG.NE.-5) IFLAG=1
      IRANK=IRANK-ILAST+PIVOT+1
      IF (.NOT.ABORT1) GO TO 300
      IDISP(2)=IACTIV
      IFLAG=-1
      IF (LP.NE.0) WRITE(LP,50)
      GO TO 1110
  300 K=PIVOT-1
      DO 350 I=ISTART,ILAST
      IF (LASTR(I).NE.0) GO TO 350
      K=K+1
      LASTR(I)=K
      IF (LENRL(I).EQ.0) GO TO 340
      MINICN=MAX0(MINICN,NZROW+IBEG-1+MOREI+LENRL(I))
      IF (IACTIV-IBEG.GE.LENRL(I)) GO TO 320
C     CALL MA30D(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)  IS/
      CALL MA30DD(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)
C CHECK NOW TO SEE IF MA30D/DD HAS CREATED ENOUGH AVAILABLE SPACE.
      IF (IACTIV-IBEG.GE.LENRL(I)) GO TO 320
C CREATE MORE SPACE BY DESTROYING PREVIOUSLY CREATED LU FACTORS.
      MOREI=MOREI+IBEG-IDISP(1)
      IBEG=IDISP(1)
      IF (LP.NE.0) WRITE(LP,310)
      IFLAG=-5
  310 FORMAT(48H LU DECOMPOSITION DESTROYED TO CREATE MORE SPACE)
      IF (ABORT3) GO TO 1030
  320 J1=IPTR(I)
      J2=J1+LENRL(I)-1
      IPTR(I)=0
      DO 330 JJ=J1,J2
      A(IBEG)=A(JJ)
      ICN(IBEG)=ICN(JJ)
      ICN(JJ)=0
  330 IBEG=IBEG+1
      NZROW=NZROW-LENRL(I)
  340 IF (K.EQ.ILAST) GO TO 360
  350 CONTINUE
  360 K=PIVOT-1
      DO 370 I=ISTART,ILAST
      IF (LASTC(I).NE.0) GO TO 370
      K=K+1
      LASTC(I)=-K
      IF (K.EQ.ILAST) GO TO 940
  370 CONTINUE
C
C THE PIVOT HAS NOW BEEN FOUND IN POSITION (IPIV,JPIV) IN LOCATION
C     IJPOS IN ROW FILE.
C UPDATE COLUMN AND ROW ORDERING ARRAYS TO CORRESPOND WITH REMOVAL
C     OF THE ACTIVE PART OF THE MATRIX.
  380 ISING=PIVOT
      IF (A(IJPOS).NE.ZERO) GO TO 390
C NUMERICAL SINGULARITY IS RECORDED HERE.
      ISING=-ISING
      IF (IFLAG.NE.-5) IFLAG=2
      IF (.NOT.ABORT2) GO TO 390
      IDISP(2)=IACTIV
      IFLAG=-2
      IF (LP.NE.0) WRITE(LP,70)
      GO TO 1110
  390 OLDPIV=IPTR(IPIV)+LENRL(IPIV)
      OLDEND=IPTR(IPIV)+LENR(IPIV)-1
C CHANGES TO COLUMN ORDERING.
      DO 460 JJ=OLDPIV,OLDEND
      J=ICN(JJ)
      LC=LASTC(J)
      NC=NEXTC(J)
      IF (NC.NE.0) LASTC(NC)=LC
      IF (LC.EQ.0) GO TO 440
      NEXTC(LC)=NC
      GO TO 460
 440  NZ=LENC(J)
      ISW=IFIRST(NZ)
      IF (ISW.GT.0) LASTR(ISW)=-NC
      IF (ISW.LT.0) IFIRST(NZ)=-NC
  460 CONTINUE
C CHANGES TO ROW ORDERING.
      I1=IPC(JPIV)
      I2=I1+LENC(JPIV)-1
      DO 530 II=I1,I2
      I=IRN(II)
      LR=LASTR(I)
      NR=NEXTR(I)
      IF (NR.NE.0) LASTR(NR)=LR
      IF (LR.LE.0) GO TO 500
      NEXTR(LR)=NR
      GO TO 530
 500  NZ=LENR(I)-LENRL(I)
      IF (NR.NE.0) IFIRST(NZ)=NR
      IF (NR.EQ.0) IFIRST(NZ)=LR
  530 CONTINUE
C     RECORD THE COLUMN PERMUTATION IN LASTC(JPIV) AND THE ROW
C     PERMUTATION IN LASTR(IPIV).
      LASTC(JPIV)=ISING
      LASTR(IPIV)=PIVOT
C
C MOVE PIVOT TO POSITION LENRL+1 IN PIVOT ROW AND MOVE PIVOT ROW
C     TO THE BEGINNING OF THE AVAILABLE STORAGE.
C THE L PART AND THE PIVOT IN THE OLD COPY OF THE PIVOT ROW IS
C     NULLIFIED WHILE, IN THE STRICTLY UPPER TRIANGULAR PART, THE
C     COLUMN INDICES, J SAY, ARE OVERWRITTEN BY THE CORRESPONDING
C     ELEMENT OF IQ (IQ(J)) AND IQ(J) IS SET TO THE NEGATIVE OF THE
C     DISPLACEMENT OF THE COLUMN INDEX FROM THE PIVOT ELEMENT.
      IF (OLDPIV.EQ.IJPOS) GO TO 540
      AU=A(OLDPIV)
      A(OLDPIV)=A(IJPOS)
      A(IJPOS)=AU
      ICN(IJPOS)=ICN(OLDPIV)
      ICN(OLDPIV)=JPIV
C CHECK TO SEE IF THERE IS SPACE IMMEDIATELY AVAILABLE IN A/ICN TO
C     HOLD NEW COPY OF PIVOT ROW.
  540 MINICN=MAX0(MINICN,NZROW+IBEG-1+MOREI+LENR(IPIV))
      IF (IACTIV-IBEG.GE.LENR(IPIV)) GO TO 550
C     CALL MA30D(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)  IS/
      CALL MA30DD(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)
      OLDPIV=IPTR(IPIV)+LENRL(IPIV)
      OLDEND=IPTR(IPIV)+LENR(IPIV)-1
C CHECK NOW TO SEE IF MA30D/DD HAS CREATED ENOUGH AVAILABLE SPACE.
      IF (IACTIV-IBEG.GE.LENR(IPIV)) GO TO 550
C CREATE MORE SPACE BY DESTROYING PREVIOUSLY CREATED LU FACTORS.
      MOREI=MOREI+IBEG-IDISP(1)
      IBEG=IDISP(1)
      IF (LP.NE.0) WRITE(LP,310)
      IFLAG=-5
      IF (ABORT3) GO TO 1030
      IF (IACTIV-IBEG.GE.LENR(IPIV)) GO TO 550
C THERE IS STILL NOT ENOUGH ROOM IN A/ICN.
      IFLAG=-4
      GO TO 1030
C COPY PIVOT ROW AND SET UP IQ ARRAY.
  550 IJPOS=0
      J1=IPTR(IPIV)
C
      DO 570 JJ=J1,OLDEND
      A(IBEG)=A(JJ)
      ICN(IBEG)=ICN(JJ)
      IF (IJPOS.NE.0) GO TO 560
      IF (ICN(JJ).EQ.JPIV) IJPOS=IBEG
      ICN(JJ)=0
      GO TO 570
  560 K=IBEG-IJPOS
      J=ICN(JJ)
      ICN(JJ)=IQ(J)
      IQ(J)=-K
  570 IBEG=IBEG+1
C
      IJP1=IJPOS+1
      PIVEND=IBEG-1
      LENPIV=PIVEND-IJPOS
      NZROW=NZROW-LENRL(IPIV)-1
      IPTR(IPIV)=OLDPIV+1
      IF (LENPIV.EQ.0) IPTR(IPIV)=0
C
C REMOVE PIVOT ROW (INCLUDING PIVOT) FROM COLUMN ORIENTED FILE.
      DO 600 JJ=IJPOS,PIVEND
      J=ICN(JJ)
      I1=IPC(J)
      LENC(J)=LENC(J)-1
C I2 IS LAST POSITION IN NEW COLUMN.
      I2=IPC(J)+LENC(J)-1
      IF (I2.LT.I1) GO TO 590
      DO 580 II=I1,I2
      IF (IRN(II).NE.IPIV) GO TO 580
      IRN(II)=IRN(I2+1)
      GO TO 590
  580 CONTINUE
  590 IRN(I2+1)=0
  600 CONTINUE
      NZCOL=NZCOL-LENPIV-1
C
C GO DOWN THE PIVOT COLUMN AND FOR EACH ROW WITH A NON-ZERO ADD
C     THE APPROPRIATE MULTIPLE OF THE PIVOT ROW TO IT.
C WE LOOP ON THE NUMBER OF NON-ZEROS IN THE PIVOT COLUMN SINCE
C     MA30D/DD MAY CHANGE ITS ACTUAL POSITION.
C
      NZPC=LENC(JPIV)
      IF (NZPC.EQ.0) GO TO 870
      DO 820 III=1,NZPC
      II=IPC(JPIV)+III-1
      I=IRN(II)
C SEARCH ROW I FOR NON-ZERO TO BE ELIMINATED, CALCULATE MULTIPLIER,
C     AND PLACE IT IN POSITION LENRL+1 IN ITS ROW.
      J1=IPTR(I)+LENRL(I)
      IEND=IPTR(I)+LENR(I)-1
      DO 610 JJ=J1,IEND
      IF (ICN(JJ).NE.JPIV) GO TO 610
C IF PIVOT IS ZERO, REST OF COLUMN IS AND SO MULTIPLIER IS ZERO.
      AU=ZERO
      IF (A(IJPOS).NE.ZERO) AU=-A(JJ)/A(IJPOS)
      A(JJ)=A(J1)
      A(J1)=AU
      ICN(JJ)=ICN(J1)
      ICN(J1)=JPIV
      LENRL(I)=LENRL(I)+1
      GO TO 620
  610 CONTINUE
C GO TO 870 IF PIVOT ROW IS A SINGLETON.
  620 IF (LENPIV.EQ.0) GO TO 820
C NOW PERFORM NECESSARY OPERATIONS ON REST OF NON-PIVOT ROW I.
      ROWI=J1+1
      IOP=0
C IF ALL THE PIVOT ROW CAUSES FILL-IN GO TO 640
      IF (ROWI.GT.IEND) GO TO 640
C PERFORM OPERATIONS ON CURRENT NON-ZEROS IN ROW I.
C INNERMOST LOOP.
      DO 630 JJ=ROWI,IEND
      J=ICN(JJ)
      IF (IQ(J).GT.0) GO TO 630
      IOP=IOP+1
      PIVROW=IJPOS-IQ(J)
      A(JJ)=A(JJ)+AU*A(PIVROW)
      ICN(PIVROW)=-ICN(PIVROW)
  630 CONTINUE
  640 IFILL=LENPIV-IOP
C IF THERE IS NO FILL-IN GO TO 740.
      IF (IFILL.EQ.0) GO TO 740
C NOW FOR THE FILL-IN.
      MINICN=MAX0(MINICN,MOREI+IBEG-1+NZROW+IFILL+LENR(I))
C SEE IF THERE IS ROOM FOR FILL-IN.
C GET MAXIMUM SPACE FOR ROW I IN SITU.
      DO 650 JDIFF=1,IFILL
      JNPOS=IEND+JDIFF
      IF (JNPOS.GT.LICN) GO TO 660
      IF (ICN(JNPOS).NE.0) GO TO 660
  650 CONTINUE
C THERE IS ROOM FOR ALL THE FILL-IN AFTER THE END OF THE ROW SO IT
C     CAN BE LEFT IN SITU.
C NEXT AVAILABLE SPACE FOR FILL-IN.
      IEND=IEND+1
      GO TO 740
C JMORE SPACES FOR FILL-IN ARE REQUIRED IN FRONT OF ROW.
  660 JMORE=IFILL-JDIFF+1
      I1=IPTR(I)
C WE NOW LOOK IN FRONT OF THE ROW TO SEE IF THERE IS SPACE FOR
C     THE REST OF THE FILL-IN.
      DO 670 JDIFF=1,JMORE
      JNPOS=I1-JDIFF
      IF (JNPOS.LT.IACTIV) GO TO 680
      IF (ICN(JNPOS).NE.0) GO TO 690
  670 CONTINUE
  680 JNPOS=I1-JMORE
      GO TO 700
C WHOLE ROW MUST BE MOVED TO THE BEGINNING OF AVAILABLE STORAGE.
  690 JNPOS=IACTIV-LENR(I)-IFILL
C IF THERE IS SPACE IMMEDIATELY AVAILABLE FOR THE SHIFTED ROW GO TO 720.
  700 IF (JNPOS.GE.IBEG) GO TO 720
C     CALL MA30D(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)  IS/
      CALL MA30DD(A,ICN,IPTR(ISTART),N,IACTIV,ITOP,.TRUE.)
      I1=IPTR(I)
      IEND=I1+LENR(I)-1
      JNPOS=IACTIV-LENR(I)-IFILL
      IF (JNPOS.GE.IBEG) GO TO 720
C NO SPACE AVAILABLE SO TRY TO CREATE SOME BY THROWING AWAY PREVIOUS
C     LU DECOMPOSITION.
      MOREI=MOREI+IBEG-IDISP(1)-LENPIV-1
      IF (LP.NE.0) WRITE(LP,310)
      IFLAG=-5
      IF (ABORT3) GO TO 1030
C KEEP RECORD OF CURRENT PIVOT ROW.
      IBEG=IDISP(1)
      ICN(IBEG)=JPIV
      A(IBEG)=A(IJPOS)
      IJPOS=IBEG
      DO 710 JJ=IJP1,PIVEND
      IBEG=IBEG+1
      A(IBEG)=A(JJ)
  710 ICN(IBEG)=ICN(JJ)
      IJP1=IJPOS+1
      PIVEND=IBEG
      IBEG=IBEG+1
      IF (JNPOS.GE.IBEG) GO TO 720
C THIS STILL DOES NOT GIVE ENOUGH ROOM.
      IFLAG=-4
      GO TO 1030
  720 IACTIV=MIN0(IACTIV,JNPOS)
C MOVE NON-PIVOT ROW I.
      IPTR(I)=JNPOS
      DO 730 JJ=I1,IEND
      A(JNPOS)=A(JJ)
      ICN(JNPOS)=ICN(JJ)
      JNPOS=JNPOS+1
  730 ICN(JJ)=0
C FIRST NEW AVAILABLE SPACE.
      IEND=JNPOS
  740 NZROW=NZROW+IFILL
C INNERMOST FILL-IN LOOP WHICH ALSO RESETS ICN.
      DO 810 JJ=IJP1,PIVEND
      J=ICN(JJ)
      IF (J.LT.0) GO TO 800
      A(IEND)=AU*A(JJ)
      ICN(IEND)=J
      IEND=IEND+1
C
C PUT NEW ENTRY IN COLUMN FILE.
      MINIRN=MAX0(MINIRN,NZCOL+LENC(J)+1)
      JEND=IPC(J)+LENC(J)
      JROOM=NZPC-III+1+LENC(J)
      IF (JEND.GT.LIRN) GO TO 750
      IF (IRN(JEND).EQ.0) GO TO 790
  750 IF (JROOM.LT.DISPC) GO TO 760
C COMPRESS COLUMN FILE TO OBTAIN SPACE FOR NEW COPY OF COLUMN.
C     CALL MA30D(A,IRN,IPC(ISTART),N,DISPC,LIRN,.FALSE.)  IS/
      CALL MA30DD(A,IRN,IPC(ISTART),N,DISPC,LIRN,.FALSE.)
      IF (JROOM.LT.DISPC) GO TO 760
      JROOM=DISPC-1
      IF (JROOM.GE.LENC(J)+1) GO TO 760
C COLUMN FILE IS NOT LARGE ENOUGH.
      GO TO 1050
C COPY COLUMN TO BEGINNING OF FILE.
  760 JBEG=IPC(J)
      JEND=IPC(J)+LENC(J)-1
      JZERO=DISPC-1
      DISPC=DISPC-JROOM
      IDISPC=DISPC
      DO 770 II=JBEG,JEND
      IRN(IDISPC)=IRN(II)
      IRN(II)=0
  770 IDISPC=IDISPC+1
      IPC(J)=DISPC
      JEND=IDISPC
      DO 780 II=JEND,JZERO
  780 IRN(II)=0
  790 IRN(JEND)=I
      NZCOL=NZCOL+1
      LENC(J)=LENC(J)+1
C END OF ADJUSTMENT TO COLUMN FILE.
      GO TO 810
C
  800 ICN(JJ)=-J
  810 CONTINUE
      LENR(I)=LENR(I)+IFILL
C END OF SCAN OF PIVOT COLUMN.
  820 CONTINUE
C
C
C REMOVE PIVOT COLUMN FROM COLUMN ORIENTED STORAGE AND UPDATE ROW
C     ORDERING ARRAYS.
      I1=IPC(JPIV)
      I2=IPC(JPIV)+LENC(JPIV)-1
      NZCOL=NZCOL-LENC(JPIV)
      DO 860 II=I1,I2
      I=IRN(II)
      IRN(II)=0
      NZ=LENR(I)-LENRL(I)
      IF (NZ.NE.0) GO TO 830
      LASTR(I)=0
      GO TO 860
  830 IFIR=IFIRST(NZ)
      IFIRST(NZ)=I
      IF (IFIR) 840,855,850
 840  LASTR(I)=IFIR
      NEXTR(I)=0
      GO TO 860
 850  LASTR(I)=LASTR(IFIR)
      NEXTR(I)=IFIR
      LASTR(IFIR)=I
      GO TO 860
 855  LASTR(I)=0
      NEXTR(I)=0
      NZMIN=MIN0(NZMIN,NZ)
 860  CONTINUE
C RESTORE IQ AND NULLIFY U PART OF OLD PIVOT ROW.
  870 IPC(JPIV)=0
      IF (LENPIV.EQ.0) GO TO 930
      NZROW=NZROW-LENPIV
      JVAL=IJP1
      JZER=IPTR(IPIV)
      IPTR(IPIV)=0
      DO 880 JCOUNT=1,LENPIV
      J=ICN(JVAL)
      IQ(J)=ICN(JZER)
      ICN(JZER)=0
      JVAL=JVAL+1
  880 JZER=JZER+1
C ADJUST COLUMN ORDERING ARRAYS.
      DO 920 JJ=IJP1,PIVEND
      J=ICN(JJ)
      NZ=LENC(J)
      IF (NZ.NE.0) GO TO 890
      LASTC(J)=0
      GO TO 920
  890 IFIR=IFIRST(NZ)
      LASTC(J)=0
      IF (IFIR) 900,910,915
 900  IFIRST(NZ)=-J
      IFIR=-IFIR
      LASTC(IFIR)=J
      NEXTC(J)=IFIR
      GO TO 920
 910  IFIRST(NZ)=-J
      NEXTC(J)=0
      NZMIN=MIN0(NZMIN,NZ)
      GO TO 920
 915  LC=-LASTR(IFIR)
      LASTR(IFIR)=-J
      NEXTC(J)=LC
      IF (LC.NE.0) LASTC(LC)=J
  920 CONTINUE
  930 CONTINUE
C ********************************************
C ****    END OF MAIN ELIMINATION LOOP    ****
C ********************************************
C
C RESET IACTIV TO POINT TO THE BEGINNING OF THE NEXT BLOCK.
  940 IF (ILAST.NE.NN) IACTIV=IPTR(ILAST+1)
  950 CONTINUE
C
C ********************************************
C ****    END OF DEOMPOSITION OF BLOCK    ****
C ********************************************
C
C RECORD SINGULARITY (IF ANY) IN IQ ARRAY.
      IF (IRANK.EQ.NN) GO TO 970
      DO 960 I=1,NN
      IF (LASTC(I).GT.0) GO TO 960
      ISING=-LASTC(I)
      IQ(ISING)=-IQ(ISING)
      LASTC(I)=ISING
  960 CONTINUE
C
C RUN THROUGH LU DECOMPOSITION CHANGING COLUMN INDICES TO THAT OF NEW
C     ORDER AND PERMUTING LENR AND LENRL ARRAYS ACCORDING TO PIVOT
C     PERMUTATIONS.
  970 ISTART=IDISP(1)
      IEND=IBEG-1
      DO 980 JJ=ISTART,IEND
      JOLD=ICN(JJ)
  980 ICN(JJ)=LASTC(JOLD)
      DO 990 II=1,NN
      I=LASTR(II)
      NEXTR(I)=LENR(II)
  990 NEXTC(I)=LENRL(II)
      DO 1000 I=1,NN
      LENRL(I)=NEXTC(I)
 1000 LENR(I)=NEXTR(I)
C
C UPDATE PERMUTATION ARRAYS IP AND IQ.
      DO 1010 II=1,NN
      I=LASTR(II)
      J=LASTC(II)
      NEXTR(I)=IABS(IP(II)+0)
 1010 NEXTC(J)=IABS(IQ(II)+0)
      DO 1020 I=1,NN
      IF (IP(I).LT.0) NEXTR(I)=-NEXTR(I)
      IP(I)=NEXTR(I)
      IF (IQ(I).LT.0) NEXTC(I)=-NEXTC(I)
 1020 IQ(I)=NEXTC(I)
      IP(NN)=IABS(IP(NN)+0)
      IDISP(2)=IEND
      GO TO 1110
C
C   ***    ERROR RETURNS    ***
 1030 IDISP(2)=IACTIV
      IF (LP.EQ.0) GO TO 1110
      WRITE(LP,1040)
 1040 FORMAT(55H ERROR RETURN FROM MA30A/AD BECAUSE LICN NOT BIG ENOUGH)
      GO TO 1080
 1050 IF (IFLAG.EQ.-5) IFLAG=-6
      IF (IFLAG.NE.-6) IFLAG=-3
      IDISP(2)=IACTIV
      IF (LP.EQ.0) GO TO 1110
      IF (IFLAG.EQ.-3) WRITE(LP,1060)
      IF (IFLAG.EQ.-6) WRITE(LP,1070)
 1060 FORMAT(55H ERROR RETURN FROM MA30A/AD BECAUSE LIRN NOT BIG ENOUGH)
 1070 FORMAT(51H ERROR RETURN FROM MA30A/AD LIRN AND LICN TOO SMALL)
 1080 PIVOT=PIVOT-ISTART+1
      WRITE(LP,1090) PIVOT,NBLOCK,ISTART,ILAST
 1090 FORMAT(10H AT STAGE ,I5,10H IN BLOCK ,I5,
     116H WITH FIRST ROW ,I5,14H AND LAST ROW ,I5)
      IF (PIVOT.EQ.0) WRITE(LP,1100) MINIRN
 1100 FORMAT(34H TO CONTINUE SET LIRN TO AT LEAST ,I8)
C
C
 1110 RETURN
      END
C     SUBROUTINE MA30D(A,ICN,IPTR,N,IACTIV,ITOP,REALS)  IS/
      SUBROUTINE MA30DD(A,ICN,IPTR,N,IACTIV,ITOP,REALS)
C     REAL A(ITOP)                                                   IS/
      DOUBLE PRECISION A(ITOP)
      INTEGER IPTR(N)
      LOGICAL REALS
      INTEGER   ICN(ITOP)
C     INTEGER*2 ICN(ITOP)                                            ID/
C     COMMON /MA30F/ IRNCP,ICNCP,IRANK,MINIRN,MINICN  IS/
      COMMON /MA30FD/ IRNCP,ICNCP,IRANK,MINIRN,MINICN
      SAVE /MA30FD/
C IACTIV IS THE FIRST POSITION IN ARRAYS A/ICN FROM WHICH THE
C     COMPRESS STARTS.
C ON EXIT IACTIV EQUALS THE POSITION OF THE FIRST ELEMENT IN THE
C     COMPRESSED PART OF A/ICN
      IF (REALS) ICNCP=ICNCP+1
      IF (.NOT.REALS) IRNCP=IRNCP+1
C SET THE FIRST NON-ZERO ELEMENT IN EACH ROW TO THE NEGATIVE OF THE
C     ROW/COL NUMBER AND HOLD THIS ROW/COL INDEX IN THE ROW/COL
C     POINTER.  THIS IS SO THAT THE BEGINNING OF EACH ROW/COL CAN
C     BE RECOGNIZED IN THE SUBSEQUENT SCAN.
      DO 10 J=1,N
      K=IPTR(J)
      IF (K.LT.IACTIV) GO TO 10
      IPTR(J)=ICN(K)
      ICN(K)=-J
   10 CONTINUE
      KN=ITOP+1
      KL=ITOP-IACTIV+1
C GO THROUGH ARRAYS IN REVERSE ORDER COMPRESSING TO THE BACK SO
C     THAT THERE ARE NO ZEROS HELD IN POSITIONS IACTIV TO ITOP IN ICN.
C     RESET FIRST ELEMENT OF EACH ROW/COL AND POINTER ARRAY IPTR.
      DO 30 K=1,KL
      JPOS=ITOP-K+1
      IF (ICN(JPOS).EQ.0) GO TO 30
      KN=KN-1
      IF (REALS) A(KN)=A(JPOS)
      IF (ICN(JPOS).GE.0) GO TO 20
C FIRST NON-ZERO OF ROW/COL HAS BEEN LOCATED
      J=-ICN(JPOS)
      ICN(JPOS)=IPTR(J)
      IPTR(J)=KN
   20 ICN(KN)=ICN(JPOS)
   30 CONTINUE
      IACTIV=KN
      RETURN
      END
C     SUBROUTINE MA30B(N,ICN,A,LICN,LENR,LENRL,IDISP,IP,IQ,W,IW,  IS/
C    1IFLAG)  IS/
      SUBROUTINE MA30BD(N,ICN,A,LICN,LENR,LENRL,IDISP,IP,IQ,W,IW,
     1IFLAG)
C     REAL A(LICN),W(N)                                              IS/
      DOUBLE PRECISION A(LICN),W(N),AU,EPS,ROWMAX,ZERO,ONE,RMIN
      INTEGER IW(N),IDISP(2),PIVPOS
      LOGICAL ABORT1,ABORT2,ABORT3,STAB
      INTEGER   ICN(LICN),LENR(N),LENRL(N),IP(N),IQ(N)
C     INTEGER*2 ICN(LICN),LENR(N),LENRL(N),IP(N),IQ(N)               ID/
C     COMMON /MA30E/ LP,ABORT1,ABORT2,ABORT3  IS/
      COMMON /MA30ED/ LP,ABORT1,ABORT2,ABORT3
C     COMMON /MA30G/ EPS,RMIN  IS/
      COMMON /MA30GD/ EPS,RMIN
      SAVE /MA30ED/,/MA30GD/
C     DATA ZERO/0.0/,ONE/1.0/  IS/
      DATA ZERO/0.0D0/,ONE/1.0D0/
      STAB=EPS.LE.ONE
      RMIN=EPS
      ISING=0
      IFLAG=0
C 170 = RETURN...
      IF (N.EQ.1) GO TO 170
      DO 10 I=1,N
   10 W(I)=ZERO
C SET UP POINTERS TO THE BEGINNING OF THE ROWS.
      IW(1)=IDISP(1)
      DO 20 I=2,N
   20 IW(I)=IW(I-1)+LENR(I-1)
C
C   ****   START  OF MAIN LOOP    ****
C AT STEP I, ROW I OF A IS TRANSFORMED TO ROW I OF L/U BY ADDING
C     APPROPRIATE MULTIPLES OF ROWS 1 TO I-1.
C     .... USING ROW-GAUSS ELIMINATION.
      DO 140 I=1,N
C ISTART IS BEGINNING OF ROW I OF A AND ROW I OF L.
      ISTART=IW(I)
C IFIN IS END OF ROW I OF A AND ROW I OF U.
      IFIN=ISTART+LENR(I)-1
C ILEND IS END OF ROW I OF L.
      ILEND=ISTART+LENRL(I)-1
      IF (ISTART.GT.ILEND) GO TO 70
C LOAD ROW I OF A INTO VECTOR W.
      DO 30 JJ=ISTART,IFIN
      J=ICN(JJ)
   30 W(J)=A(JJ)
C
C ADD MULTIPLES OF APPROPRIATE ROWS OF  I TO I-1  TO ROW I.
      DO 50 JJ=ISTART,ILEND
      J=ICN(JJ)
C IPIVJ IS POSITION OF PIVOT IN ROW J.
      IPIVJ=IW(J)+LENRL(J)
C FORM MULTIPLIER AU.
      AU=-W(J)/A(IPIVJ)
      W(J)=AU
C AU * ROW J (U PART) IS ADDED TO ROW I.
      IPIVJ=IPIVJ+1
      JFIN=IW(J)+LENR(J)-1
      IF (IPIVJ.GT.JFIN) GO TO 50
C INNERMOST LOOP.
      DO 40 JAYJAY=IPIVJ,JFIN
      JAY=ICN(JAYJAY)
   40 W(JAY)=W(JAY)+AU*A(JAYJAY)
C
   50 CONTINUE
C RELOAD W BACK INTO A (NOW L/U)
      DO 60 JJ=ISTART,IFIN
      J=ICN(JJ)
      A(JJ)=W(J)
   60 W(J)=ZERO
C WE NOW PERFORM THE STABILITY CHECKS.
   70 PIVPOS=ILEND+1
      IF (IQ(I).GT.0) GO TO 120
C MATRIX HAD SINGULARITY AT THIS POINT IN MA30A/AD.
C IS IT THE FIRST SUCH PIVOT IN CURRENT BLOCK ?
      IF (ISING.EQ.0) ISING=I
C DOES CURRENT MATRIX HAVE A SINGULARITY IN THE SAME PLACE ?
      IF (PIVPOS.GT.IFIN) GO TO 80
      IF (A(PIVPOS).NE.ZERO) GO TO 150
C IT DOES .. SO SET ISING IF IT IS NOT THE END OF THE CURRENT BLOCK
C CHECK TO SEE THAT APPROPRIATE PART OF L/U IS ZERO OR NULL.
   80 IF (ISTART.GT.IFIN) GO TO 100
      DO 90 JJ=ISTART,IFIN
      IF (ICN(JJ).LT.ISING) GO TO 90
      IF (A(JJ).NE.ZERO) GO TO 150
   90 CONTINUE
  100 IF (PIVPOS.LE.IFIN) A(PIVPOS)=ONE
      IF (IP(I).GT.0.AND.I.NE.N) GO TO 140
C END OF CURRENT BLOCK ... RESET ZERO PIVOTS AND ISING.
      DO 110 J=ISING,I
      IF ((LENR(J)-LENRL(J)).EQ.0) GO TO 110
      JJ=IW(J)+LENRL(J)
      A(JJ)=ZERO
  110 CONTINUE
      ISING=0
      GO TO 140
C MATRIX HAD NON-ZERO PIVOT IN MA30A/AD AT THIS STAGE.
  120 IF (PIVPOS.GT.IFIN) GO TO 150
      IF (A(PIVPOS).EQ.ZERO) GO TO 150
      IF (.NOT.STAB) GO TO 140
      ROWMAX=ZERO
      DO 130 JJ=PIVPOS,IFIN
C 130 ROWMAX=AMAX1(ROWMAX,ABS(A(JJ)))                                IS/
  130 ROWMAX=DMAX1(ROWMAX,DABS(A(JJ)))
C     IF (ABS(A(PIVPOS))/ROWMAX.GE.RMIN) GO TO 140 IS/
      IF (DABS(A(PIVPOS))/ROWMAX.GE.RMIN) GO TO 140
      IFLAG=I
C     RMIN=ABS(A(PIVPOS))/ROWMAX  IS/
      RMIN=DABS(A(PIVPOS))/ROWMAX
C   ****    END OF MAIN LOOP    ****
  140 CONTINUE
C
      GO TO 170
C   ***   ERROR RETURN   ***
  150 IF (LP.NE.0) WRITE(LP,160) I
  160 FORMAT(55H ERROR RETURN FROM MA30B/BD SINGULARITY DETECTED IN ROW,
     1I8)
      IFLAG=-I
C
  170 RETURN
      END
C     SUBROUTINE MA30C(N,ICN,A,LICN,LENR,LENRL,LENOFF,IDISP,IP,IQ,  IS/
C    1X,W,MTYPE) IS/
      SUBROUTINE MA30CD(N,ICN,A,LICN,LENR,LENRL,LENOFF,IDISP,IP,IQ,
     1X,W,MTYPE)
C     REAL A(LICN),X(N),W(N)                                         IS/
      DOUBLE PRECISION A(LICN),X(N),W(N),WII,WI,RESID,ZERO
      INTEGER IDISP(2)
      LOGICAL NEG,NOBLOC
      INTEGER   ICN(LICN),LENR(N),LENRL(N),LENOFF(N),IP(N),IQ(N)
C     INTEGER*2 ICN(LICN),LENR(N),LENRL(N),LENOFF(N),IP(N),IQ(N)     ID/
C     COMMON /MA30H/ RESID  IS/
      COMMON /MA30HD/ RESID
      SAVE /MA30HD/
C     DATA ZERO/0.0/   IS/
      DATA ZERO/0.0D0/
C THE FINAL VALUE OF RESID IS THE MAXIMUM RESIDUAL FOR AN INCONSISTENT
C     SET OF EQUATIONS.
      RESID=ZERO
C NOBLOC IS .TRUE. IF SUBROUTINE BLOCK HAS BEEN USED PREVIOUSLY AND
C     IS .FALSE. OTHERWISE.  THE VALUE .FALSE. MEANS THAT LENOFF
C     WILL NOT BE SUBSEQUENTLY ACCESSED.
      NOBLOC=LENOFF(1).LT.0
      IF (MTYPE.EQ.2) GO TO 140
C
C WE NOW SOLVE   A * X = B.
C NEG IS USED TO INDICATE WHEN THE LAST ROW IN A BLOCK HAS BEEN
C     REACHED.  IT IS THEN SET TO TRUE WHEREAFTER BACKSUBSTITUTION IS
C     PERFORMED ON THE BLOCK.
      NEG=.FALSE.
C IP(N) IS NEGATED SO THAT THE LAST ROW OF THE LAST BLOCK CAN BE
C     RECOGNISED.  IT IS RESET TO ITS POSITIVE VALUE ON EXIT.
      IP(N)=-IP(N)
C PREORDER VECTOR ... W(I) = X(IP(I))
      DO 10 II=1,N
      I=IP(II)
      I=IABS(I)
   10 W(II)=X(I)
C LT HOLDS THE POSITION OF THE FIRST NON-ZERO IN THE CURRENT ROW OF THE
C     OFF-DIAGONAL BLOCKS.
      LT=1
C IFIRST HOLDS THE INDEX OF THE FIRST ROW IN THE CURRENT BLOCK.
      IFIRST=1
C IBLOCK HOLDS THE POSITION OF THE FIRST NON-ZERO IN THE CURRENT ROW
C     OF THE LU DECOMPOSITION OF THE DIAGONAL BLOCKS.
      IBLOCK=IDISP(1)
C IF I IS NOT THE LAST ROW OF A BLOCK, THEN A PASS THROUGH THIS LOOP
C     ADDS THE INNER PRODUCT OF ROW I OF THE OFF-DIAGONAL BLOCKS AND W
C     TO W AND PERFORMS FORWARD ELIMINATION USING ROW I OF THE LU
C     DECOMPOSITION.   IF I IS THE LAST ROW OF A BLOCK THEN, AFTER
C     PERFORMING THESE AFOREMENTIONED OPERATIONS, BACKSUBSTITUTION IS
C     PERFORMED USING THE ROWS OF THE BLOCK.
      DO 120 I=1,N
      WI=W(I)
      IF(NOBLOC)GO TO 30
      IF (LENOFF(I).EQ.0) GO TO 30
C OPERATIONS USING LOWER TRIANGULAR BLOCKS.
C LTEND IS THE END OF ROW I IN THE OFF-DIAGONAL BLOCKS.
      LTEND=LT+LENOFF(I)-1
      DO 20 JJ=LT,LTEND
      J=ICN(JJ)
   20 WI=WI-A(JJ)*W(J)
C LT IS SET THE BEGINNING OF THE NEXT OFF-DIAGONAL ROW.
      LT=LTEND+1
C SET NEG TO .TRUE. IF WE ARE ON THE LAST ROW OF THE BLOCK.
   30 IF (IP(I).LT.0) NEG=.TRUE.
      IF (LENRL(I).EQ.0) GO TO 50
C FORWARD ELIMINATION PHASE.
C IEND IS THE END OF THE L PART OF ROW I IN THE LU DECOMPOSITION.
      IEND=IBLOCK+LENRL(I)-1
      DO 40 JJ=IBLOCK,IEND
      J=ICN(JJ)
   40 WI=WI+A(JJ)*W(J)
C IBLOCK IS ADJUSTED TO POINT TO THE START OF THE NEXT ROW.
   50 IBLOCK=IBLOCK+LENR(I)
      W(I)=WI
      IF (.NOT.NEG) GO TO 120
C BACK SUBSTITUTION PHASE.
C J1 IS POSITION IN A/ICN AFTER END OF BLOCK BEGINNING IN ROW IFIRST
C     AND ENDING IN ROW I.
      J1=IBLOCK
C ARE THERE ANY SINGULARITIES IN THIS BLOCK?  IF NOT, CONTINUE WITH
C     THE BACKSUBSTITUTION.
C IF MTYPE=3 FIRST SINGULARITY IS SUPRESSED BY SETTING PIVOT TO ONE
      IB=I
      IF (IQ(I).GT.0 .OR. MTYPE.EQ.3) GO TO 70
      DO 60 III=IFIRST,I
      IB=I-III+IFIRST
      IF (IQ(IB).GT.0) GO TO 70
      J1=J1-LENR(IB)
C     RESID=AMAX1(RESID,ABS(W(IB)))                                  IS/
      RESID=DMAX1(RESID,DABS(W(IB)))
      W(IB)=ZERO
   60 CONTINUE
C ENTIRE BLOCK IS SINGULAR.
      GO TO 110
C EACH PASS THROUGH THIS LOOP PERFORMS THE BACK-SUBSTITUTION
C     OPERATIONS FOR A SINGLE ROW, STARTING AT THE END OF THE BLOCK AND
C     WORKING THROUGH IT IN REVERSE ORDER.
   70 DO 100 III=IFIRST,IB
      II=IB-III+IFIRST
C J2 IS END OF ROW II.
      J2=J1-1
C J1 IS BEGINNING OF ROW II.
      J1=J1-LENR(II)
C JPIV IS THE POSITION OF THE PIVOT IN ROW II.
      JPIV=J1+LENRL(II)
      JPIVP1=JPIV+1
C IF ROW  II OF U HAS NO NON-ZEROS GO TO 90.
      IF (J2.LT.JPIVP1) GO TO 90
      WII=W(II)
      DO 80 JJ=JPIVP1,J2
      J=ICN(JJ)
   80 WII=WII-A(JJ)*W(J)
      W(II)=WII
90    CONTINUE
      IF (MTYPE.NE.3) W(II)=W(II)/A(JPIV)
      MTYPE=1
  100 CONTINUE
  110 IFIRST=I+1
      NEG=.FALSE.
  120 CONTINUE
C
C REORDER SOLUTION VECTOR ... X(I) = W(IQINVERSE(I))
      DO 130 II=1,N
      I=IQ(II)
      I=IABS(I)
  130 X(I)=W(II)
      IP(N)=-IP(N)
      GO TO 310
C
C
C WE NOW SOLVE   ATRANSPOSE * X = B.
C PREORDER VECTOR ... W(I)=X(IQ(I))
  140 DO 150 II=1,N
      I=IQ(II)
      I=IABS(I)
  150 W(II)=X(I)
C LJ1 POINTS TO THE BEGINNING THE CURRENT ROW IN THE OFF-DIAGONAL
C     BLOCKS.
      LJ1=IDISP(1)
C IBLOCK IS INITIALIZED TO POINT TO THE BEGINNING OF THE BLOCK AFTER
C     THE LAST ONE 
      IBLOCK=IDISP(2)+1
C ILAST IS THE LAST ROW IN THE CURRENT BLOCK.
      ILAST=N
C IBLEND POINTS TO THE POSITION AFTER THE LAST NON-ZERO IN THE
C     CURRENT BLOCK.
      IBLEND=IBLOCK
C EACH PASS THROUGH THIS LOOP OPERATES WITH ONE DIAGONAL BLOCK AND
C     THE OFF-DIAGONAL PART OF THE MATRIX CORRESPONDING TO THE ROWS
C     OF THIS BLOCK.  THE BLOCKS ARE TAKEN IN REVERSE ORDER AND THE
C     NUMBER OF TIMES THE LOOP IS ENTERED IS MIN(N,NO. BLOCKS+1).
      DO 280 NUMBLK=1,N
      IF (ILAST.EQ.0) GO TO 290
      IBLOCK=IBLOCK-LENR(ILAST)
C THIS LOOP FINDS THE INDEX OF THE FIRST ROW IN THE CURRENT BLOCK..
C     IT IS FIRST AND IBLOCK IS SET TO THE POSITION OF THE BEGINNING
C     OF THIS FIRST ROW.
      DO 160 K=1,N
      II=ILAST-K
      IF (II.EQ.0) GO TO 170
      IF (IP(II).LT.0) GO TO 170
      IBLOCK=IBLOCK-LENR(II)
  160 CONTINUE
  170 IFIRST=II+1
C J1 POINTS TO THE POSITION OF THE BEGINNING OF ROW I (LT PART) OR PIVOT
      J1=IBLOCK
C FORWARD ELIMINATION.
C EACH PASS THROUGH THIS LOOP PERFORMS THE OPERATIONS FOR ONE ROW OF THE
C     BLOCK.  IF THE CORRESPONDING ELEMENT OF W IS ZERO THEN THE
C     OPERATIONS CAN BE AVOIDED.
      DO 200 I=IFIRST,ILAST
      IF (W(I).EQ.ZERO) GO TO 195
C IS ROW I SINGULAR?  IF SO, GO TO 210
      IF (IQ(I).LT.0) GO TO 210
C J2 FIRST POINTS TO THE PIVOT IN ROW I AND THEN IS MADE TO POINT TO THE
C     FIRST NON-ZERO IN THE U TRANSPOSE PART OF THE ROW.
      J2=J1+LENRL(I)
      WI=W(I)/A(J2)
      IF (LENR(I)-LENRL(I).EQ.1) GO TO 190
      J2=J2+1
C J3 POINTS TO THE END OF ROW I.
      J3=J1+LENR(I)-1
      DO 180 JJ=J2,J3
      J=ICN(JJ)
  180 W(J)=W(J)-A(JJ)*WI
  190 W(I)=WI
  195 J1=J1+LENR(I)
  200 CONTINUE
      GO TO 230
C DEALS WITH REST OF BLOCK WHICH IS SINGULAR.
  210 DO 220 II=I,ILAST
C     RESID=AMAX1(RESID,ABS(W(II)))                                  IS/
      RESID=DMAX1(RESID,DABS(W(II)))
      W(II)=ZERO
  220 CONTINUE
C BACK SUBSTITUTION.
C THIS LOOP DOES THE BACK SUBSTITUTION ON THE ROWS OF THE BLOCK IN
C     THE REVERSE ORDER DOING IT SIMULTANEOUSLY ON THE L TRANSPOSE PART
C     OF THE DIAGONAL BLOCKS AND THE OFF-DIAGONAL BLOCKS.
  230 J1=IBLEND
      DO 270 IBACK=IFIRST,ILAST
      I=ILAST-IBACK+IFIRST
C J1 POINTS TO THE BEGINNING OF ROW I.
      J1=J1-LENR(I)
      IF (LENRL(I).EQ.0) GO TO 250
C J2 POINTS TO THE END OF THE L TRANSPOSE PART OF ROW I.
      J2=J1+LENRL(I)-1
      DO 240 JJ=J1,J2
      J=ICN(JJ)
  240 W(J)=W(J)+A(JJ)*W(I)
  250 IF(NOBLOC)GO TO 270
C OPERATIONS USING LOWER TRIANGULAR BLOCKS.
      IF(LENOFF(I).EQ.0)GO TO 270
C LJ2 POINTS TO THE END OF ROW I OF THE OFF-DIAGONAL BLOCKS.
      LJ2=LJ1-1
C LJ1 POINTS TO THE BEGINNING OF ROW I OF THE OFF-DIAGONAL BLOCKS.
      LJ1=LJ1-LENOFF(I)
      DO 260 JJ=LJ1,LJ2
      J=ICN(JJ)
  260 W(J)=W(J)-A(JJ)*W(I)
  270 CONTINUE
      IBLEND=J1
      ILAST=IFIRST-1
  280 CONTINUE
C REORDER SOLUTION VECTOR ... X(I)=W(IPINVERSE(I))
  290 DO 300 II=1,N
      I=IP(II)
      I=IABS(I)
  300 X(I)=W(II)
C
  310 RETURN
      END
C I IS IBM SP AND S IS STANDARD SP    IS/
      SUBROUTINE MC21A(N,ICN,LICN,IP,LENR,IPERM,NUMNZ,IW)
      INTEGER IP(N)
      INTEGER ICN(LICN),LENR(N),IPERM(N),IW(N,4)
C     INTEGER*2 ICN(LICN),LENR(N),IPERM(N),IW(N,4)                    I/
      CALL MC21B(N,ICN,LICN,IP,LENR,IPERM,NUMNZ,IW(1,1),IW(1,2),IW(1,3),
     1IW(1,4))
      RETURN
      END
      SUBROUTINE MC21B(N,ICN,LICN,IP,LENR,IPERM,NUMNZ,PR,ARP,CV,OUT)
      INTEGER IP(N)
C   PR(I) IS THE PREVIOUS ROW TO I IN THE DEPTH FIRST SEARCH.
C IT IS USED AS A WORK ARRAY IN THE SORTING ALGORITHM.
C   ELEMENTS (IPERM(I),I) I=1, ... N  ARE NON-ZERO AT THE END OF THE
C ALGORITHM UNLESS N ASSIGNMENTS HAVE NOT BEEN MADE.  IN WHICH CASE
C (IPERM(I),I) WILL BE ZERO FOR N-NUMNZ ENTRIES.
C   CV(I) IS THE MOST RECENT ROW EXTENSION AT WHICH COLUMN I
C WAS VISITED.
C   ARP(I) IS ONE LESS THAN THE NUMBER OF NON-ZEROS IN ROW I
C WHICH HAVE NOT BEEN SCANNED WHEN LOOKING FOR A CHEAP ASSIGNMENT.
C   OUT(I) IS ONE LESS THAN THE NUMBER OF NON-ZEROS IN ROW I
C WHICH HAVE NOT BEEN SCANNED DURING ONE PASS THROUGH THE MAIN LOOP.
      INTEGER ICN(LICN),LENR(N),IPERM(N),PR(N),CV(N),
     1ARP(N),OUT(N)
C     INTEGER*2 ICN(LICN),LENR(N),IPERM(N),PR(N),CV(N),               I/
C    1ARP(N),OUT(N)                                                   I/
C
C   INITIALIZATION OF ARRAYS.
      DO 10 I=1,N
      ARP(I)=LENR(I)-1
      CV(I)=0
   10 IPERM(I)=0
      NUMNZ=0
C
C
C   MAIN LOOP.
C   EACH PASS ROUND THIS LOOP EITHER RESULTS IN A NEW ASSIGNMENT
C OR GIVES A ROW WITH NO ASSIGNMENT.
      DO 130 JORD=1,N
      J=JORD
      PR(J)=-1
      DO 100 K=1,JORD
C LOOK FOR A CHEAP ASSIGNMENT
      IN1=ARP(J)
      IF (IN1.LT.0) GO TO 60
      IN2=IP(J)+LENR(J)-1
      IN1=IN2-IN1
      DO 50 II=IN1,IN2
      I=ICN(II)
      IF (IPERM(I).EQ.0) GO TO 110
   50 CONTINUE
C   NO CHEAP ASSIGNMENT IN ROW.
      ARP(J)=-1
C   BEGIN LOOKING FOR ASSIGNMENT CHAIN STARTING WITH ROW J.
   60 OUT(J)=LENR(J)-1
C INNER LOOP.  EXTENDS CHAIN BY ONE OR BACKTRACKS.
      DO 90 KK=1,JORD
      IN1=OUT(J)
      IF (IN1.LT.0) GO TO 80
      IN2=IP(J)+LENR(J)-1
      IN1=IN2-IN1
C FORWARD SCAN.
      DO 70 II=IN1,IN2
      I=ICN(II)
      IF (CV(I).EQ.JORD) GO TO 70
C   COLUMN I HAS NOT YET BEEN ACCESSED DURING THIS PASS.
      J1=J
      J=IPERM(I)
      CV(I)=JORD
      PR(J)=J1
      OUT(J1)=IN2-II-1
      GO TO 100
   70 CONTINUE
C
C   BACKTRACKING STEP.
   80 J=PR(J)
      IF (J.EQ.-1) GO TO 130
   90 CONTINUE
C
  100 CONTINUE
C
C   NEW ASSIGNMENT IS MADE.
  110 IPERM(I)=J
      ARP(J)=IN2-II-1
      NUMNZ=NUMNZ+1
      DO 120 K=1,JORD
      J=PR(J)
      IF (J.EQ.-1) GO TO 130
      II=IP(J)+LENR(J)-OUT(J)-2
      I=ICN(II)
      IPERM(I)=J
  120 CONTINUE
C
  130 CONTINUE
C
C   IF MATRIX IS STRUCTURALLY SINGULAR, WE NOW COMPLETE THE
C PERMUTATION IPERM.
      IF (NUMNZ.EQ.N) RETURN
      DO 140 I=1,N
  140 ARP(I)=0
      K=0
      DO 160 I=1,N
      IF (IPERM(I).NE.0) GO TO 150
      K=K+1
      OUT(K)=I
      GO TO 160
  150 J=IPERM(I)
      ARP(J)=I
  160 CONTINUE
      K=0
      DO 170 I=1,N
      IF (ARP(I).NE.0) GO TO 170
      K=K+1
      IOUTK=OUT(K)
      IPERM(IOUTK)=I
  170 CONTINUE
      RETURN
      END
C I AND J ARE IBM SINGLE AND DOUBLE PRECISION CODES RESP.  JISD/
C S AND D ARE STANDARD SINGLE AND DOUBLE PRECISION CODES RESP.
C     SUBROUTINE MA28A(N,NZ,A,LICN,IRN,LIRN,ICN,U,IKEEP,  IS/
C    1IW,W,IFLAG)  IS/
      SUBROUTINE MA28AD(N,NZ,A,LICN,IRN,LIRN,ICN,U,IKEEP,
     1IW,W,IFLAG)
C THE PARAMETERS ARE AS FOLLOWS.....
C N     INTEGER  ORDER OF MATRIX  NOT ALTERED BY SUBROUTINE.
C NZ    INTEGER  NUMBER OF NON-ZEROS IN INPUT MATRIX  NOT ALTERED
C     BY SUBROUTINE.
C A     REAL/DOUBLE PRECISION ARRAY  LENGTH LICN.  HOLDS NON-ZEROS OF
C     MATRIX ON ENTRY AND NON-ZEROS OF FACTORS ON EXIT.  REORDERED BY
C     MC20A/AD AND MC23A/AD AND ALTERED BY MA30A/AD.
C LICN  INTEGER  LENGTH OF ARRAYS A AND ICN.  NOT ALTERED BY
C     SUBROUTINE.
C IRN   INTEGER*2 ARRAY  LENGTH LIRN.  HOLDS ROW INDICES ON INPUT ...
C     USED AS WORKSPACE BY MA30A/AD TO HOLD COLUMN ORIENTATION OF
C     MATRIX.
C LIRN  INTEGER  LENGTH OF ARRAY IRN.
C ICN   INTEGER*2 ARRAY  LENGTH LICN.  HOLDS COLUMN INDICES ON ENTRY
C     AND COLUMN INDICES OF DECOMPOSED MATRIX ON EXIT. REORDERED BY
C     MC20A/AD AND MC23A/AD AND ALTERED BY MA30A/AD.
C U     REAL/DOUBLE PRECISION VARIABLE  SET BY USER TO CONTROL
C     BIAS TOWARDS NUMERIC OR SPARSITY PIVOTING.  U=1.0 GIVES PARTIAL
C     PIVOTING WHILE U=0. DOES NOT CHECK MULTIPLIERS AT ALL.
C     VALUES OF U GREATER THAN ONE ARE TREATED AS ONE WHILE NEGATIVE
C     VALUES ARE TREATED AS ZERO.  NOT ALTERED BY SUBROUTINE.
C IKEEP  INTEGER*2 ARRAY  LENGTH 5*N  USED AS WORKSPACE BY MA28A/AD
C     (SEE LATER COMMENTS).  IT IS NOT REQUIRED TO BE SET ON ENTRY
C     AND, ON EXIT, IT CONTAINS INFORMATION ABOUT THE DECOMPOSITION.
C     IT SHOULD BE PRESERVED BETWEEN THIS CALL AND SUBSEQUENT CALLS
C     TO MA28B/BD OR MA28C/CD.
C     IKEEP(I,1),I=1,N  HOLDS THE TOTAL LENGTH OF THE PART OF ROW I
C     IN THE DIAGONAL BLOCK.
C     ROW IKEEP(I,2),I=1,N  OF THE INPUT MATRIX IS THE ITH ROW IN
C     PIVOT ORDER.
C     COLUMN IKEEP(I,3),I=1,N  OF THE INPUT MATRIX IS THE ITH COLUMN
C     IN PIVOT ORDER.
C     IKEEP(I,4),I=1,N  HOLDS THE LENGTH OF THE PART OF ROW I IN
C     THE L PART OF THE L/U DECOMPOSITION.
C     IKEEP(I,5),I=1,N  HOLDS THE LENGTH OF THE PART OF ROW I IN THE
C     OFF-DIAGONAL BLOCKS.  IF THERE IS ONLY ONE DIAGONAL BLOCK,
C     IKEEP(1,5) WILL BE SET TO -1.
C IW    INTEGER*2 ARRAY  LENGTH 10*N.  TO OBTAIN CORRECT ALIGNMENT
C     FOR THIS ARRAY, BECAUSE PARTS OF IT ARE USED AS INTEGER*4
C     WORKSPACE, THE USER SHOULD HAVE DECLARED IT TO BE AN
C     INTEGER ARRAY OF LENGTH 5*N.  IN THE STANDARD VERSION THIS
C     DISTINCTION DISAPPEARS AND THE LENGTH OF THIS WORK-ARRAY (IN
C     THE USER'S PROGRAM) SHOULD CONSEQUENTLY BE CHANGED TO 8*N.
C W     REAL/DOUBLE PRECISION ARRAY  LENGTH N.  USED BY MC24A/AD BOTH
C     AS WORKSPACE AND TO RETURN GROWTH ESTIMATE IN W(1).  THE USE OF
C     THIS ARRAY BY MA28A/AD IS THUS OPTIONAL DEPENDING ON COMMON
C     BLOCK LOGICAL VARIABLE GROW.
C IFLAG  INTEGER VARIABLE  USED AS ERROR FLAG BY ROUTINE.  A POSITIVE
C     OR ZERO VALUE ON EXIT INDICATES SUCCESS.  POSSIBLE NEGATIVE
C     VALUES ARE -1 THROUGH -14.
C     REAL A(LICN),W(N)  IS/
      DOUBLE PRECISION A(LICN),U,W(N),UPRIV,RMIN,EPS,RESID,ZERO,
     1THEMAX
      INTEGER IDISP(2),IPRIV4
      INTEGER   ICN(LICN),IRN(LIRN),IKEEP(N,5),IW(N,8),IPRIV2(2) 
C     INTEGER*2 ICN(LICN),IRN(LIRN),IKEEP(N,5),IW(N,10),IPRIV2(2)    ID/
      LOGICAL GROW,LBLOCK,ABORT,ABORT1,ABORT2,ABORT3,ABORTA,ABORTB
C COMMON BLOCKS ... COMMON BLOCK MA28F/FD IS USED MERELY
C     TO COMMUNICATE WITH COMMON BLOCK MA30F/FD  SO THAT THE USER
C     NEED NOT DECLARE THIS COMMON BLOCK IN HIS MAIN PROGRAM.
C THE COMMON BLOCK VARIABLES ARE AS FOLLOWS ...
C LP,MP  INTEGER  DEFAULT VALUE 6 (LINE PRINTER).  UNIT NUMBER
C     FOR ERROR MESSAGES AND DUPLICATE ELEMENT WARNING RESP.
C NLP,MLP  INTEGER  UNIT NUMBER FOR MESSAGES FROM MA30A/AD AND
C     MC23A/AD RESP.  SET BY MA28A/AD TO VALUE OF LP.
C LBLOCK  LOGICAL  DEFAULT VALUE TRUE.  IF TRUE MC23A/AD IS USED
C     TO FIRST PERMUTE THE MATRIX TO BLOCK LOWER TRIANGULAR FORM.
C GROW    LOGICAL  DEFAULT VALUE TRUE.  IF TRUE THEN AN ESTIMATE
C     OF THE INCREASE IN SIZE OF MATRIX ELEMENTS DURING L/U
C     DECOMPOSITION IS GIVEN BY MC24A/AD.
C EPS,RMIN,RESID  REAL/DOUBLE PRECISION VARIABLES NOT REFERENCED
C     BY MA28A/AD.
C IRNCP,ICNCP  INTEGER  SET TO NUMBER OF COMPRESSES ON ARRAYS IRN AND
C     ICN/A RESPECTIVELY.
C MINIRN,MINICN  INTEGER  MINIMUM LENGTH OF ARRAYS IRN AND ICN/A
C     RESPECTIVELY, FOR SUCCESS ON FUTURE RUNS.
C IRANK  INTEGER   ESTIMATED RANK OF MATRIX.
C MIRNCP,MICNCP,MIRANK,MIRN,MICN INTEGER VARIABLES.  USED TO
C     COMMUNICATE BETWEEN MA30F/FD AND MA28F/FD VALUES OF ABOVENAMED
C     VARIABLES WITH SOMEWHAT SIMILAR NAMES.
C ABORT1,ABORT2  LOGICAL VARIABLES WITH DEFAULT VALUE TRUE.  IF FALSE
C     THEN DECOMPOSITION WILL BE PERFORMED EVEN IF THE MATRIX IS
C     STRUCTURALLY OR NUMERICALLY SINGULAR RESPECTIVELY.
C ABORTA,ABORTB  LOGICAL VARIABLES USED TO COMMUNICATE VALUES OF
C     ABORT1 AND ABORT2 TO MA30A/AD.
C ABORT  LOGICAL  USED TO COMMUNICATE VALUE OF ABORT1 TO MC23A/AD.
C ABORT3  LOGICAL VARIABLE NOT REFERENCED BY MA28A/AD.
C IDISP   INTEGER ARRAY  LENGTH 2.  USED TO COMMUNICATE INFORMATION
C     ON DECOMPOSITION BETWEEN THIS CALL TO MA28A/AD AND SUBSEQUENT
C     CALLS TO MA28B/BD AND MA28C/CD.  ON EXIT, IDISP(1) AND
C     IDISP(2) INDICATE POSITION IN ARRAYS A AND ICN OF THE
C     FIRST AND LAST ELEMENTS IN THE L/U DECOMPOSITION OF THE
C     DIAGONAL BLOCKS, RESPECTIVELY.
C NUMNZ  INTEGER  STRUCTURAL RANK OF MATRIX.
C NUM    INTEGER  NUMBER OF DIAGONAL BLOCKS.
C LARGE  INTEGER  SIZE OF LARGEST DIAGONAL BLOCK.
C     INTERNAL VARIABLES AND WORKSPACE USED IN  MA28A/AD ARE DEFINED
C     WITHIN THE SUBROUTINE IMMEDIATELY PRIOR TO THEIR FIRST USE.
C     COMMON /MA28E/ LP,MP,LBLOCK,GROW  IS/
      COMMON /MA28ED/ LP,MP,LBLOCK,GROW
C     COMMON /MA28F/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,  IS/
C    1IRANK,ABORT1,ABORT2  IS/
      COMMON /MA28FD/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     1IRANK,ABORT1,ABORT2
C     COMMON /MA28G/ IDISP  IS/
      COMMON /MA28GD/ IDISP
C     COMMON /MA30E/ NLP,ABORTA,ABORTB,ABORT3  IS/
      COMMON /MA30ED/ NLP,ABORTA,ABORTB,ABORT3
C     COMMON /MA30F/ MIRNCP,MICNCP,MIRANK,MIRN,MICN  IS/
      COMMON /MA30FD/ MIRNCP,MICNCP,MIRANK,MIRN,MICN
C     COMMON /MC23B/ MLP,NUMNZ,NUM,LARGE,ABORT  IS/
      COMMON /MC23BD/ MLP,NUMNZ,NUM,LARGE,ABORT
      SAVE /MA28ED/,/MA28FD/,/MA28GD/,/MA30ED/,/MA30FD/,/MC23BD/
C EQUIVALENCE IS USED TO OBTAIN INTEGER*4 INFORMATION FROM INTEGER*2
C     ARRAYS  ...  SEE LATER COMMENTS ON THE SETTING OF IPRIV4.
      EQUIVALENCE(IPRIV4,IPRIV2(1))
C     DATA ZERO /0.0E0/  IS/
      DATA ZERO /0.0D0/
C SOME  INITIALIZATION AND TRANSFER OF INFORMATION BETWEEN
C     COMMON BLOCKS (SEE EARLIER COMMENTS).
      IFLAG=0
      ABORTA=ABORT1
      ABORTB=ABORT2
      ABORT=ABORT1
      MLP=LP
      NLP=LP
C UPRIV PRIVATE COPY OF U IS USED IN CASE IT IS OUTSIDE
C     RANGE  ZERO TO ONE  AND  IS THUS ALTERED BY MA30A/AD.
      UPRIV=U
C SIMPLE DATA CHECK ON INPUT VARIABLES AND ARRAY DIMENSIONS.
      IF (N.GT.0) GO TO 2
C     IF (N.GT.0.AND.N.LE.32767) GO TO 2                             ID/
      IFLAG=-8
      IF (LP.NE.0) WRITE(LP,1) N
 1    FORMAT(36X,17HN OUT OF RANGE = ,I10)
      GO TO 999
 2    IF (NZ.GT.0) GO TO 4
      IFLAG=-9
      IF (LP.NE.0) WRITE(LP,3) NZ
 3    FORMAT(36X,18HNZ NON POSITIVE = ,I10)
      GO TO 999
 4    IF (LICN.GE.NZ) GO TO 6
      IFLAG=-10
      IF (LP.NE.0) WRITE(LP,5) LICN
 5    FORMAT(36X,17HLICN TOO SMALL = ,I10)
      GO TO 999
 6    IF (LIRN.GE.NZ) GO TO 8
      IFLAG=-11
      IF (LP.NE.0) WRITE(LP,7) LIRN
 7    FORMAT(36X,17HLIRN TOO SMALL = ,I10)
      GO TO 999
C
C DATA CHECK TO SEE IF ALL INDICES LIE BETWEEN 1 AND N.
 8    DO 30 I=1,NZ
      IF (IRN(I).GT.0.AND.IRN(I).LE.N.AND.ICN(I).GT.0.AND.ICN(I).LE.N)
     1GO TO 30
      IF (IFLAG.EQ.0.AND.LP.NE.0) WRITE(LP,10)
 10   FORMAT(62H ERROR RETURN FROM MA28A/AD BECAUSE INDICES FOUND OUT OF
     1 RANGE)
      IFLAG=-12
      IF (LP.NE.0) WRITE(LP,20) I,A(I),IRN(I),ICN(I)
 20   FORMAT(1X,I6,22HTH ELEMENT WITH VALUE ,1PD22.14,
     130H IS OUT OF RANGE WITH INDICES ,I8,2H ,,I8)
 30   CONTINUE
      IF (IFLAG.LT.0) GO TO 1000
C
C SORT ELEMENTS INTO ROW ORDER.
C     CALL MC20A(N,NZ,A,ICN,IW,IRN,0)  IS/
      CALL MC20AD(N,NZ,A,ICN,IW,IRN,0)
C
C THESE TWO STATEMENTS (TOGETHER WITH THE EARLIER EQUIVALENCE STATEMENT)
C SET IPRIV4 EQUAL TO THE INTEGER*4 VALUE CONSISTING OF THE
C CONCATENATION OF THE TWO INTEGER*2 WORDS IW(1,1) AND IW(2,1).
C THIS IS NECESSARY IN THE IBM VERSION BECAUSE MC20A/AD EXPECTS AN
C INTEGER*4 ARGUMENT IN IW.  THE STATEMENTS CAN BE LEFT IN
C WITHOUT AFFECTING A STANDARD FORTRAN VERSION (ASSUMING THE
C EQUIVALENCE STATEMENT IS KEPT).
      IPRIV2(1)=IW(1,1)
      IPRIV2(2)=IW(1,2)
      IF (N.GT.1) IPRIV2(2)=IW(2,1)
C PART OF IKEEP IS USED HERE AS A WORK-ARRAY.  IKEEP(I,2) IS
C     THE LAST ROW TO HAVE A NON-ZERO IN COLUMN I.  IKEEP(I,3)
C     IS THE OFF-SET OF COLUMN I FROM THE START OF THE ROW.
      DO 40 I=1,N
      IKEEP(I,2)=0
 40   IKEEP(I,1)=0
C
C CHECK FOR DUPLICATE ELEMENTS .. SUMMING ANY SUCH ENTRIES AND
C     PRINTING A WARNING MESSAGE ON UNIT MP.
C MOVE IS EQUAL TO THE NUMBER OF DUPLICATE ELEMENTS FOUND.
      MOVE=0
C THE LOOP ALSO CALCULATES THE LARGEST ELEMENT IN THE MATRIX, THEMAX.
      THEMAX=ZERO
C J1 IS POSITION IN ARRAYS OF FIRST NON-ZERO IN ROW.
      J1=IPRIV4
      DO 80 I=1,N
      IF (I.NE.N) GO TO 45
      IPRIV4=NZ+1
      GO TO 49
C THESE STATEMENTS ARE USED AS ABOVE TO SET IPRIV4.
C THIS TIME THE CHANGE INDICATED BY THE SPECIAL COMMENT CARD IS
C REQUIRED FOR SUCCESSFUL OPERATION OF THE STANDARD VERSION.
 45   DO 47 L=1,2
      K=1
      J=I+1
C     J=2*I+L                                                        ID/
      IF (J.LE.N) GO TO 46
      J=J-N
      K=2
 46   IPRIV2(L)=IW(J,K)
 47   CONTINUE
 49   LENGTH=IPRIV4-J1
      IF (LENGTH.EQ.0) GO TO 80
      J2=IPRIV4-1
      NEWJ1=J1-MOVE
      DO 70 JJ=J1,J2
      J=ICN(JJ)
C     THEMAX=AMAX1(THEMAX,ABS(A(JJ))) IS/
      THEMAX=DMAX1(THEMAX,DABS(A(JJ)))
      IF (IKEEP(J,2).EQ.I) GO TO 50
C FIRST TIME COLUMN HAS OCURRED IN CURRENT ROW.
      IKEEP(J,2)=I
      IKEEP(J,3)=JJ-MOVE-NEWJ1
      IF (MOVE.EQ.0) GO TO 70
C SHIFT NECESSARY BECAUSE OF  PREVIOUS DUPLICATE ELEMENT.
      NEWPOS=JJ-MOVE
      A(NEWPOS)=A(JJ)
      ICN(NEWPOS)=ICN(JJ)
      GO TO 70
C DUPLICATE ELEMENT.
 50   MOVE=MOVE+1
      LENGTH=LENGTH-1
      JAY=IKEEP(J,3)+NEWJ1
      IF (MP.NE.0) WRITE(MP,60) I,J,A(JJ)
 60   FORMAT(31H DUPLICATE ELEMENT IN POSITION ,I8,2H ,,I8,
     112H WITH VALUE ,1PD22.14)
      A(JAY)=A(JAY)+A(JJ)
C     THEMAX=AMAX1(THEMAX,ABS(A(JAY))) IS/
      THEMAX=DMAX1(THEMAX,DABS(A(JAY)))
 70   CONTINUE
      IKEEP(I,1)=LENGTH
      J1=IPRIV4
 80   CONTINUE
C
C KNUM IS ACTUAL NUMBER OF NON-ZEROS IN MATRIX WITH ANY MULTIPLE
C     ENTRIES COUNTED ONLY ONCE.
      KNUM=NZ-MOVE
      IF (.NOT.LBLOCK) GO TO 100
C
C PERFORM BLOCK TRIANGULARISATION.
C     CALL MC23A(N,ICN,A,LICN,IKEEP,IDISP,IKEEP(1,2),IKEEP(1,3), I/
C    1IKEEP(1,5),IW(1,5),IW)  I/
C     CALL MC23A(N,ICN,A,LICN,IKEEP,IDISP,IKEEP(1,2),IKEEP(1,3), S/
C    1IKEEP(1,5),IW(1,3),IW)  S/
      CALL MC23AD(N,ICN,A,LICN,IKEEP,IDISP,IKEEP(1,2),IKEEP(1,3),
     1IKEEP(1,5),IW(1,3),IW)
C     CALL MC23AD(N,ICN,A,LICN,IKEEP,IDISP,IKEEP(1,2),IKEEP(1,3), J/
C    1IKEEP(1,5),IW(1,5),IW)  J/
      IF (IDISP(1).GT.0) GO TO 130
      IFLAG=-7
      IF(IDISP(1).EQ.-1) IFLAG=-1
      IF (LP.NE.0) WRITE(LP,90)
 90   FORMAT(36X,26HERROR RETURN FROM MC23A/AD)
      GO TO 999
C
C BLOCK TRIANGULARIZATION NOT REQUESTED.
C MOVE STRUCTURE TO END OF DATA ARRAYS IN PREPARATION FOR
C     MA30A/AD.
C ALSO SET LENOFF(1) TO -1 AND SET PERMUTATION ARRAYS.
 100  DO 110 I=1,KNUM
      II=KNUM-I+1
      NEWPOS=LICN-I+1
      ICN(NEWPOS)=ICN(II)
 110  A(NEWPOS)=A(II)
      IDISP(1)=1
      IDISP(2)=LICN-KNUM+1
      DO 120 I=1,N
      IKEEP(I,2)=I
 120  IKEEP(I,3)=I
      IKEEP(1,5)=-1
C
C PERFORM L/U DECOMOSITION ON DIAGONAL BLOCKS.
C130  CALL MA30A(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),  S/
C    1IKEEP(1,3),IRN,LIRN,IW(1,2),IW(1,3),IW(1,4),IW(1,5),IW(1,6), S/
C    2IW(1,7),IW(1,8),IW,UPRIV,IFLAG)  S/
 130  CALL MA30AD(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),
     1IKEEP(1,3),IRN,LIRN,IW(1,2),IW(1,3),IW(1,4),IW(1,5),IW(1,6),
     2IW(1,7),IW(1,8),IW,UPRIV,IFLAG)
C130  CALL MA30A(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),  I/
C    1IKEEP(1,3),IRN,LIRN,IW(1,3),IW(1,4),IW(1,5),IW(1,6),IW(1,7), I/
C    2IW(1,8),IW(1,9),IW,UPRIV,IFLAG)  I/
C130  CALL MA30AD(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),  J/
C    1IKEEP(1,3),IRN,LIRN,IW(1,3),IW(1,4),IW(1,5),IW(1,6),IW(1,7), J/
C    2IW(1,8),IW(1,9),IW,UPRIV,IFLAG)  J/
C
C TRANSFER COMMON BLOCK INFORMATION.
      MINIRN=MAX0(MIRN,NZ)
      MINICN=MAX0(MICN,NZ)
      IRNCP=MIRNCP
      ICNCP=MICNCP
      IRANK=MIRANK
      IF (IFLAG.GE.0) GO TO 140
      IF (LP.NE.0) WRITE(LP,135)
 135  FORMAT(36X,26HERROR RETURN FROM MA30A/AD)
      GO TO 999
C
C REORDER OFF-DIAGONAL BLOCKS ACCORDING TO PIVOT PERMUTATION.
 140  I1=IDISP(1)-1
C     IF (I1.NE.0) CALL MC22A(N,ICN,A,I1,IKEEP(1,5),IKEEP(1,2), IS/
C    1IKEEP(1,3),IW,IRN) IS/
      IF (I1.NE.0) CALL MC22AD(N,ICN,A,I1,IKEEP(1,5),IKEEP(1,2),
     1IKEEP(1,3),IW,IRN)
C
C OPTIONALLY CALCULATE ELEMENT GROWTH ESTIMATE.
      I1=IDISP(1)
      IEND=LICN-I1+1
C     IF (GROW) CALL MC24A(N,ICN,A(I1),IEND,IKEEP,IKEEP(1,4),W) IS/
      IF (GROW) CALL MC24AD(N,ICN,A(I1),IEND,IKEEP,IKEEP(1,4),W)
C INCREMENT GROWTH ESTIMATE BY ORIGINAL MAXIMUM ELEMENT.
      IF (GROW) W(1)=W(1)+THEMAX
      IF (GROW.AND.N.GT.1) W(2)=THEMAX
C SET FLAG IF THE ONLY ERROR IS DUE TO DUPLICATE ELEMENTS.
      IF (IFLAG.GE.0.AND.MOVE.NE.0) IFLAG=-14
      GO TO 1000
 999  IF (LP.NE.0) WRITE(LP,998)
 998  FORMAT(36H+ERROR RETURN FROM MA28A/AD BECAUSE )
 1000 RETURN
      END
      BLOCK DATA
C     REAL EPS,RMIN,RESID        IS/
      DOUBLE PRECISION EPS,RMIN,RESID
      LOGICAL LBLOCK,GROW,ABORT1,ABORT2
C     COMMON /MA28E/ LP,MP,LBLOCK,GROW   IS/
      COMMON /MA28ED/ LP,MP,LBLOCK,GROW
C     COMMON /MA28F/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,   IS/
C    1IRANK,ABORT1,ABORT2    IS/
      COMMON /MA28FD/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     1IRANK,ABORT1,ABORT2
      SAVE /MA28ED/,/MA28FD/
C     DATA EPS/1.0E-4/         IS/
      DATA EPS/1.0D-4/
      DATA LP/6/,MP/6/
      DATA LBLOCK/.TRUE./,GROW/.TRUE./
      DATA ABORT1/.TRUE./,ABORT2/.TRUE./
      END
C     SUBROUTINE MA28B(N,NZ,A,LICN,IVECT,JVECT,ICN,IKEEP, IS/
C    1IW,W,IFLAG) IS/
      SUBROUTINE MA28BD(N,NZ,A,LICN,IVECT,JVECT,ICN,IKEEP,
     1IW,W,IFLAG)
C THE PARAMETERS ARE AS FOLLOWS ...
C N      INTEGER  ORDER OF MATRIX  NOT ALTERED BY SUBROUTINE.
C NZ     INTEGER  NUMBER OF NON-ZEROS IN INPUT MATRIX  NOT ALTERED
C     BY SUBROUTINE.
C A      REAL/DOUBLE PRECISION ARRAY  LENGTH LICN.  HOLDS NON-ZEROS OF
C     MATRIX ON ENTRY AND NON-ZEROS OF FACTORS ON EXIT.  REORDERED BY
C     MA28D/DD AND ALTERED BY SUBROUTINE MA30B/BD.
C LICN   INTEGER  LENGTH OF ARRAYS A AND ICN.  NOT ALTERED BY
C     SUBROUTINE.
C IVECT,JVECT  INTEGER*2 ARRAYS  LENGTH NZ.  HOLD ROW AND COLUMN
C     INDICES OF NON-ZEROS RESPECTIVELY.  NOT ALTERED BY SUBROUTINE.
C ICN    INTEGER*2 ARRAY  LENGTH LICN.  SAME ARRAY AS OUTPUT FROM
C     MA28A/AD.  UNCHANGED BY MA28B/BD.
C IKEEP  INTEGER*2 ARRAY  LENGTH 5*N.  SAME ARRAY AS OUTPUT FROM
C     MA28A/AD.  UNCHANGED BY MA28B/BD.
C IW     INTEGER ARRAY  LENGTH 4*N (5*N IN STANDARD VERSION).
C     USED AS WORKSPACE BY MA28D/DD AND MA30B/BD.
C W      REAL/DOUBLE PRECISION ARRAY  LENGTH N.  USED AS WORKSPACE
C     BY MA28D/DD,MA30B/BD AND (OPTIONALLY) MC24A/AD.
C IFLAG  INTEGER  USED AS ERROR FLAG WITH POSITIVE OR ZERO VALUE
C     INDICATING SUCCESS.
C     REAL A(LICN),W(N),MEPS,MRMIN    IS/
      DOUBLE PRECISION A(LICN),W(N),EPS,MEPS,RMIN,MRMIN
     1                ,RESID,WMAX
      INTEGER IDISP(2),IW(N,5)     
C     INTEGER IDISP(2),IW(N,4)    ID/
      INTEGER   IKEEP(N,5),IVECT(NZ),JVECT(NZ),ICN(LICN)
C     INTEGER*2  IKEEP(N,5),IVECT(NZ),JVECT(NZ),ICN(LICN) ID/
      LOGICAL GROW,LBLOCK,ABORTA,ABORTB,ABORT1,ABORT2,ABORT3
C UNLESS OTHERWISE STATED COMMON BLOCK VARIABLES ARE AS IN MA28A/AD.
C     THOSE VARIABLES REFERENCED BY MA28B/BD ARE MENTIONED BELOW.
C LP,MP  INTEGERS  USED AS IN MA28A/AD AS UNIT NUMBER FOR ERROR AND
C     WARNING MESSAGES, RESPECTIVELY.
C NLP    INTEGER VARIABLE USED TO GIVE VALUE OF LP TO MA30E/ED.
C EPS    REAL/DOUBLE PRECISION  MA30B/BD WILL OUTPUT A POSITIVE VALUE
C     FOR IFLAG IF ANY MODULUS OF THE RATIO OF PIVOT ELEMENT TO THE
C     LARGEST ELEMENT IN ITS ROW (U PART ONLY) IS LESS THAN EPS (UNLESS
C     EPS IS GREATER THAN 1.0 WHEN NO ACTION TAKES PLACE).
C RMIN   REAL/DOUBLE PRECISION  VARIABLE EQUAL TO THE VALUE OF THIS MINI
C     RATIO IN CASES WHERE EPS IS LESS THAN OR EQUAL TO 1.0
C MEPS,MRMIN  REAL/DOUBLE PRECISION VARIABLES USED BY THE SUBROUTINE
C     TO COMMUNICATE BETWEEN COMMON BLOCKS MA28F/FD AND MA30G/GD.
C IDISP  INTEGER ARRAY  LENGTH 2  THE SAME AS THAT USED BY MA28A/AD.
C     IT IS UNCHANGED BY MA28B/BD.
C     COMMON /MA28E/ MP,LP,LBLOCK,GROW  IS/
      COMMON /MA28ED/ MP,LP,LBLOCK,GROW
C     COMMON /MA28F/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,  IS/
C    1IRANK,ABORT1,ABORT2  IS/
      COMMON /MA28FD/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     1IRANK,ABORT1,ABORT2
C     COMMON /MA28G/ IDISP  IS/
      COMMON /MA28GD/ IDISP
C     COMMON /MA30E/ NLP,ABORTA,ABORTB,ABORT3  IS/
      COMMON /MA30ED/ NLP,ABORTA,ABORTB,ABORT3
C     COMMON /MA30G/ MEPS,MRMIN  IS/
      COMMON /MA30GD/ MEPS,MRMIN
      SAVE /MA28ED/,/MA28FD/,/MA28GD/,/MA30ED/,/MA30GD/
      IFLAG=0
      MEPS=EPS
      NLP=LP
C SIMPLE DATA CHECK ON VARIABLES.
      IF (N.GT.0) GO TO 2
C     IF (N.GT.0.AND.N.LE.32767) GO TO 2  ID/
      IFLAG=-11
      IF (LP.NE.0) WRITE(LP,1) N
 1    FORMAT(36X,17HN OUT OF RANGE = ,I10)
      GO TO 999
 2    IF (NZ.GT.0) GO TO 4
      IFLAG=-10
      IF (LP.NE.0) WRITE(LP,3) NZ
 3    FORMAT(36X,18HNZ NON POSITIVE = ,I10)
      GO TO 999
 4    IF (LICN.GE.NZ) GO TO 6
      IFLAG=-9
      IF (LP.NE.0) WRITE(LP,5) LICN
 5    FORMAT(36X,17HLICN TOO SMALL = ,I10)
      GO TO 999
C
C6    CALL MA28D(N,A,LICN,IVECT,JVECT,NZ,ICN,IKEEP,IKEEP(1,4),  IS/
C    1IKEEP(1,5),IKEEP(1,2),IKEEP(1,3),IW(1,3),IW,W(1),IFLAG)  IS/
 6    CALL MA28DD(N,A,LICN,IVECT,JVECT,NZ,ICN,IKEEP,IKEEP(1,4),
     1IKEEP(1,5),IKEEP(1,2),IKEEP(1,3),IW(1,3),IW,W(1),IFLAG)
C WMAX IS LARGEST ELEMENT IN MATRIX.
      WMAX=W(1)
C IDUP EQUALS ONE IF THERE WERE DUPLICATE ELEMENTS, ZERO OTHERWISE.
      IDUP=0
      IF (IFLAG.EQ.(N+1)) IDUP=1
      IF (IFLAG.LT.0) GO TO 999
C
C PERFORM ROW-GAUSS ELIMINATION ON THE STRUCTURE RECEIVED FROM MA28D/DD
C     CALL MA30B(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2), IS/
C    1IKEEP(1,3),W,IW,IFLAG) IS/
      CALL MA30BD(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IDISP,IKEEP(1,2),
     1IKEEP(1,3),W,IW,IFLAG)
C
C TRANSFER COMMON BLOCK INFORMATION.
      RMIN=MRMIN
      IF (IFLAG.GE.0) GO TO 200
      IFLAG=-2
      IF (LP.NE.0) WRITE(LP,100)
 100  FORMAT(36X,26HERROR RETURN FROM MA30B/BD)
      GO TO 999
C
C OPTIONALLY CALCULATE THE GROWTH PARAMETER.
 200  I1=IDISP(1)
      IEND=LICN-I1+1
C     IF (GROW) CALL MC24A(N,ICN,A(I1),IEND,IKEEP,IKEEP(1,4),W) IS/
      IF (GROW) CALL MC24AD(N,ICN,A(I1),IEND,IKEEP,IKEEP(1,4),W)
C INCREMENT ESTIMATE BY LARGEST ELEMENT IN INPUT MATRIX.
      IF (GROW) W(1)=W(1)+WMAX
C SET FLAG IF THE ONLY ERROR IS DUE TO DUPLICATE ELEMENTS.
      IF (IDUP.EQ.1.AND.IFLAG.GE.0) IFLAG=-14
      GO TO 1000
 999  IF (LP.NE.0) WRITE(LP,998)
 998  FORMAT(36H+ERROR RETURN FROM MA28B/BD BECAUSE )
 1000 RETURN
      END
C THIS SUBROUTINE NEED NEVER BE CALLED BY THE USER DIRECTLY.
C     IT SORTS THE USER'S MATRIX INTO THE STRUCTURE OF THE DECOMPOSED
C     FORM AND CHECKS FOR THE PRESENCE OF DUPLICATE ENTRIES OR
C     NON-ZEROS LYING OUTSIDE THE SPARSITY PATTERN OF THE DECOMPOSITION
C     IT ALSO CALCULATES THE LARGEST ELEMENT IN THE INPUT MATRIX.
C     SUBROUTINE MA28D(N,A,LICN,IVECT,JVECT,NZ,ICN,LENR,LENRL,  IS/
C    1LENOFF,IP,IQ,IW1,IW,W1,IFLAG)  IS/
      SUBROUTINE MA28DD(N,A,LICN,IVECT,JVECT,NZ,ICN,LENR,LENRL,
     1LENOFF,IP,IQ,IW1,IW,W1,IFLAG)
C     REAL A(LICN)    IS/
      DOUBLE PRECISION A(LICN),ZERO,W1,AA
      INTEGER IW(N,2),IDISP(2)
      INTEGER   ICN(LICN),IVECT(NZ),JVECT(NZ),IP(N),IQ(N),
     1LENR(N),IW1(N,3),LENRL(N),LENOFF(N)
C     INTEGER*2 ICN(LICN),IVECT(NZ),JVECT(NZ),IP(N),IQ(N),     ID/
C    1LENR(N),IW1(N,3),LENRL(N),LENOFF(N)  ID/
      LOGICAL LBLOCK,GROW,BLOCKL
C     COMMON /MA28E/ LP,MP,LBLOCK,GROW  IS/
      COMMON /MA28ED/ LP,MP,LBLOCK,GROW
C     COMMON /MA28G/ IDISP  IS/
      COMMON /MA28GD/ IDISP
      SAVE /MA28ED/,/MA28GD/
C     DATA ZERO/0.0E0/  IS/
      DATA ZERO/0.0D0/
      BLOCKL=LENOFF(1).GE.0
C IW1(I,3)  IS SET TO THE BLOCK IN WHICH ROW I LIES AND THE
C     INVERSE PERMUTATIONS TO IP AND IQ ARE SET IN IW1(.,1) AND
C     IW1(.,2) RESP.
C POINTERS TO BEGINNING OF THE PART OF ROW I IN DIAGONAL AND
C   OFF-DIAGONAL BLOCKS ARE SET IN IW(I,2) AND IW(I,1) RESP.
      IBLOCK=1
      IW(1,1)=1
      IW(1,2)=IDISP(1)
      DO 10 I=1,N
      IW1(I,3)=IBLOCK
      IF (IP(I).LT.0) IBLOCK=IBLOCK+1
      II=IABS(IP(I)+0)
      IW1(II,1)=I
      JJ=IQ(I)
      JJ=IABS(JJ)
      IW1(JJ,2)=I
      IF (I.EQ.1) GO TO 10
      IF(BLOCKL) IW(I,1)=IW(I-1,1)+LENOFF(I-1)
      IW(I,2)=IW(I-1,2)+LENR(I-1)
 10   CONTINUE
C PLACE EACH NON-ZERO IN TURN INTO ITS CORRECT LOCATION
C    IN THE A/ICN ARRAY.
      IDISP2=IDISP(2)
      DO 300 I=1,NZ
C NECESSARY TO AVOID REFERENCE TO UNASSIGNED ELEMENT OF ICN.
      IF(I.GT.IDISP2) GO TO 30
      IF (ICN(I).LT.0) GO TO 300
   30 IOLD=IVECT(I)
      JOLD=JVECT(I)
      AA=A(I)
C THIS IS A DUMMY LOOP FOR FOLLOWING A CHAIN OF INTERCHANGES.
C   IT WILL BE EXECUTED NZ TIMES IN TOTAL.
      DO 200 IDUMMY=1,NZ
C PERFORM SOME VALIDITY CHECKS ON IOLD AND JOLD.
      IF (IOLD.LE.N .AND. IOLD.GT.0
     1      .AND. JOLD.LE.N .AND. JOLD.GT.0) GO TO 60
      IF (LP.NE.0) WRITE(LP,40) I,A(I),IOLD,JOLD
 40   FORMAT(9H ELEMENT ,I6,12H WITH VALUE ,1PD22.14,
     1 13H HAS INDICES ,I8,2H ,,I8
     2 /36X,20HINDICES OUT OF RANGE)
      IFLAG=-12
      GO TO 340
 60   INEW=IW1(IOLD,1)
      JNEW=IW1(JOLD,2)
C ARE WE IN A VALID BLOCK AND IS IT DIAGONAL OR OFF-DIAGONAL?
      IF (IW1(INEW,3)-IW1(JNEW,3)) 70,100,90
 70   IFLAG=-13
      IF (LP.NE.0) WRITE(LP,80) IOLD,JOLD
 80   FORMAT(36X,8HNON-ZERO,I7,2H ,,I6,27H IN ZERO OFF-DIAGONAL BLOCK)
      GO TO 340
 90   J1=IW(INEW,1)
      J2=J1+LENOFF(INEW)-1
      GO TO 160
C ELEMENT IS IN DIAGONAL BLOCK.
 100  J1=IW(INEW,2)
      IF (INEW.GT.JNEW) GO TO 110
      J2=J1+LENR(INEW)-1
      J1=J1+LENRL(INEW)
      GO TO 160
 110  J2=J1+LENRL(INEW)
C BINARY SEARCH OF ORDERED LIST  .. ELEMENT IN L PART OF ROW.
      DO 140 JDUMMY=1,N
      MIDPT=(J1+J2)/2
      JCOMP=IABS(ICN(MIDPT)+0)
      IF (JNEW-JCOMP) 120,180,130
 120  J2=MIDPT
      GO TO 140
 130  J1=MIDPT
 140  CONTINUE
      IFLAG=-13
      IF (LP.NE.0) WRITE(LP,150) IOLD,JOLD
 150  FORMAT(36X,8H ELEMENT ,I6,2H ,,I6,23H WAS NOT IN L/U PATTERN)
      GO TO 340
C LINEAR SEARCH ... ELEMENT IN L PART OF ROW OR OFF-DIAGONAL BLOCKS.
 160  DO 170 MIDPT=J1,J2
      IF (IABS(ICN(MIDPT)+0).EQ.JNEW) GO TO 180
 170  CONTINUE
      IFLAG=-13
      IF (LP.NE.0) WRITE(LP,150) IOLD,JOLD
      GO TO 340
C EQUIVALENT ELEMENT OF ICN IS IN POSITION MIDPT.
 180  IF (ICN(MIDPT).LT.0) GO TO 250
      IF (MIDPT.GT.NZ.OR.MIDPT.LE.I) GO TO 220
      W1=A(MIDPT)
      A(MIDPT)=AA
      AA=W1
      IOLD=IVECT(MIDPT)
      JOLD=JVECT(MIDPT)
      ICN(MIDPT)=-ICN(MIDPT)
 200  CONTINUE
 220  A(MIDPT)=AA
      ICN(MIDPT)=-ICN(MIDPT)
      GO TO 300
 250  A(MIDPT)=A(MIDPT)+AA
C SET FLAG FOR DUPLICATE ELEMENTS.
      IFLAG=N+1
 300  CONTINUE
C RESET ICN ARRAY  AND ZERO ELEMENTS IN L/U BUT NOT IN A.
C ALSO CALCULATE MAXIMUM ELEMENT OF A.
  340 W1=ZERO
      DO 400 I=1,IDISP2
      IF (ICN(I).LT.0) GO TO 350
      A(I)=ZERO
      GO TO 400
 350  ICN(I)=-ICN(I)
C     W1=AMAX1(W1,ABS(A(I)))                IS/
      W1=DMAX1(W1,DABS(A(I)))
 400  CONTINUE
      RETURN
      END
C     SUBROUTINE MA28C(N,A,LICN,ICN,IKEEP,RHS,W,MTYPE)  IS/
      SUBROUTINE MA28CD(N,A,LICN,ICN,IKEEP,RHS,W,MTYPE)
C THE PARAMETERS ARE AS FOLLOWS ....
C N     INTEGER  ORDER OF MATRIX  NOT ALTERED BY SUBROUTINE.
C A      REAL/DOUBLE PRECISION ARRAY  LENGTH LICN.  THE SAME ARRAY AS
C     WAS USED IN THE MOST RECENT CALL TO MA28A/AD OR MA28B/BD.
C LICN  INTEGER  LENGTH OF ARRAYS A AND ICN.  NOT ALTERED BY
C     SUBROUTINE.
C ICN    INTEGER*2 ARRAY  LENGTH LICN.  SAME ARRAY AS OUTPUT FROM
C     MA28A/AD.  UNCHANGED BY MA28C/CD.
C IKEEP  INTEGER*2 ARRAY  LENGTH 5*N.  SAME ARRAY AS OUTPUT FROM
C     MA28A/AD.  UNCHANGED BY MA28C/CD.
C RHS    REAL/DOUBLE PRECISION ARRAY  LENGTH N.  ON ENTRY, IT HOLDS THE
C     RIGHT HAND SIDE.  ON EXIT, THE SOLUTION VECTOR.
C W      REAL/DOUBLE PRECISION ARRAY  LENGTH N. USED AS WORKSPACE BY
C     MA30C/CD.
C MTYPE  INTEGER  USED TO TELL MA30C/CD TO SOLVE THE DIRECT EQUATION
C     (MTYPE.NE.2) OR ITS TRANSPOSE (MTYPE.EQ.2).
C     IF MTYPE=3 FIRST SINGULARITY IS SUPRESSED BY SETTING PIVOT TO ONE
C     IN THIS CASE MTYPE REMAINS NOT UNCHANGED
C     REAL A(LICN),RHS(N),W(N),MRESID  IS/
      DOUBLE PRECISION A(LICN),RHS(N),W(N),MRESID,EPS,RMIN,RESID
      INTEGER IDISP(2)
      INTEGER   ICN(LICN),IKEEP(N,5)
C     INTEGER*2 ICN(LICN),IKEEP(N,5)  ID/
      LOGICAL ABORT1,ABORT2
C UNLESS OTHERWISE STATED COMMON BLOCK VARIABLES ARE AS IN MA28A/AD.
C     THOSE VARIABLES REFERENCED BY MA28C/CD ARE MENTIONED BELOW.
C RESID  REAL/DOUBLE PRECISION  VARIABLE RETURNS MAXIMUM RESIDUAL OF
C     EQUATIONS WHERE PIVOT WAS ZERO.
C MRESID  REAL/DOUBLE PRECISION VARIABLE USED BY MA28C/CD TO
C     COMMUNICATE BETWEEN MA28F/FD AND MA30H/HD.
C IDISP  INTEGER ARRAY  LENGTH 2  THE SAME AS THAT USED BY MA28A/AD.
C     IT IS UNCHANGED BY MA28B/BD.
C     COMMON /MA28F/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,  IS/
C    1IRANK,ABORT1,ABORT2  IS/
      COMMON /MA28FD/ EPS,RMIN,RESID,IRNCP,ICNCP,MINIRN,MINICN,
     1IRANK,ABORT1,ABORT2
C     COMMON /MA28G/ IDISP  IS/
      COMMON /MA28GD/ IDISP
C     COMMON /MA30H/ MRESID  IS/
      COMMON /MA30HD/ MRESID
      SAVE /MA28FD/,/MA28GD/,/MA30HD/
C
C THIS SUBROUTINE PERFORMS THE SOLUTION OF THE SET OF EQUATIONS.
C     CALL MA30C(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IKEEP(1,5),IDISP, IS/
C    1IKEEP(1,2),IKEEP(1,3),RHS,W,MTYPE)  IS/
      CALL MA30CD(N,ICN,A,LICN,IKEEP,IKEEP(1,4),IKEEP(1,5),IDISP,
     1IKEEP(1,2),IKEEP(1,3),RHS,W,MTYPE)
C
C TRANSFER COMMON BLOCK INFORMATION.
      RESID=MRESID
      RETURN
      END
