/*----------------------------------------------------------------------------
    QRMat.c	QR decomposition by Householder reflections, Mar 18, 1992

    by Andreas Hohmann and Claudia Wulff, ZIB, Hohmann@sc.ZIB-Berlin.DE

-----------------------------------------------------------------------------

        This module contains all programs needed for solution of a linear
        least squares problem Norm(Ax-b) = min using a
        Householder-triangulation, followed by a
        Cholesky-decomposition (only in case of an underdetermined 
        least square) and some back-substitutions.

    literature

    1)  P. Deuflhard and A. Hohmann, 'Numerische Mathematik -- Eine
        algorithmisch orientierte Einfuehrung', de Gruyter Verlag,
        Berlin, 1991

    2)  P. Deuflhard and W. Sautter, 'On rank-deficient Pseudoinverses',
        Lin. Alg. Appl., 29:91-111, 1980

-----------------------------------------------------------------------------*/

#include <stdio.h>
#include <math.h>
#include <malloc.h>

#include "mystd.h"
#include "message.h"
#include "matvec.h"
#include "qrmat.h"

/*----------------------------------------------------------------------------
    globalized data
-----------------------------------------------------------------------------*/

static RealMat A, V, L;
static RealVec d, b;
static RealVec domainScale, imageScale;
static Real cond, det;
static IntVec pivot;
static Int m, n, p, signum;


/*----------------------------------------------------------------------------
    allocation and deallocation
-----------------------------------------------------------------------------*/

QRMat *NewQRMat(RealMat A, Int m, Int n) {
    QRMat *qrMat;
    qrMat = (QRMat *) malloc(sizeof(QRMat));
    qrMat->AUser 	= A;
    qrMat->A		= NewRealMat(1, m, 1, n);
    if (m<n) {
	qrMat->V 	= NewRealMat(1, m, 1, n-m);
	qrMat->L 	= NewRealLowMat(1, n-m, 1, n-m);
    }
    qrMat->domainScale	= nil;
    qrMat->imageScale	= nil;
    qrMat->d 		= NewRealVec(1, n);
    qrMat->domainPivot 	= NewIntVec(1, n);
    qrMat->b		= NewRealVec(1, m);
    qrMat->cond 	= 0;
    qrMat->condMax 	= 1.0 / sqrtEpsMach;
    qrMat->m 		= m;
    qrMat->n 		= n;
    qrMat->p 		= MIN(m, n);
    qrMat->signum 	= 0;
    return qrMat;
}

void NewRank(QRMat *qrMat, Int p) {
    Int n =qrMat->n, m=qrMat->m, pOld = qrMat->p;
    if (pOld != p) {
       if (p>n || p>m) Error("in NewRank: rank too big");
       else {
         if (pOld<n && pOld != 0) {
            FreeRealMat(qrMat->V, 1, pOld, 1, n-pOld);
            FreeRealMat(qrMat->L, 1, n-pOld, 1, n-pOld);
         }
         if (p<n && p!= 0) {
            qrMat->V 	= NewRealMat(1, p, 1, n-p);
            qrMat->L 	= NewRealLowMat(1, n-p, 1, n-p);
         }
         qrMat->p = p;
       }
    }
}

void FreeQRMat(QRMat *qrMat) {
    Int m = qrMat->m, n = qrMat->n, p = qrMat->p;
    FreeRealMat(qrMat->A, 1, m, 1, n);
    if (p<n && p !=0) {
	FreeRealMat(qrMat->V, 1, p, 1, n-p);
	FreeRealMat(qrMat->L, 1, n-p, 1, n-p);
    }
    FreeRealVec(qrMat->d, 1, n);
    FreeRealVec(qrMat->b, 1, m);
    FreeIntVec(qrMat->domainPivot, 1, n);
}

/*----------------------------------------------------------------------------
    globalize user's data    
-----------------------------------------------------------------------------*/

static void Globalize(QRMat *qrMat) {
    n 		= qrMat->n;
    m 		= qrMat->m;
    p 		= qrMat->p;
    A 		= qrMat->A;
    V 		= qrMat->V;
    L 		= qrMat->L;
    det		= qrMat->det;
    cond 	= qrMat->cond;
    pivot 	= qrMat->domainPivot;
    signum 	= qrMat->signum;
    d 		= qrMat->d;
    b 		= qrMat->b;
    domainScale	= qrMat->domainScale;
    imageScale	= qrMat->imageScale;
}

/*----------------------------------------------------------------------------
    scaling procedures    
-----------------------------------------------------------------------------*/

static void ScaleImageVec(RealVec y, RealVec yScaled) {
    Int i;

    if (imageScale!=nil) {
	for (i=1; i<=m; i++) yScaled[i] = y[i] / imageScale[i];
    } 
}

static void DescaleImageVec(RealVec y, RealVec yScaled) {
    Int i;

    if (imageScale!=nil) {
	for (i=1; i<=m; i++) y[i] = yScaled[i] * imageScale[i];
    } 
}

static void ScaleDomainVec(RealVec y, RealVec yScaled) {
    Int j;

    if (domainScale!=nil) {
	for (j=1; j<=n; j++) yScaled[j] = y[j] / domainScale[j];
    } 
}

static void DescaleDomainVec(RealVec y, RealVec yScaled) {
    Int j;

    if (domainScale!=nil) {
	for (j=1; j<=n; j++) y[j] = yScaled[j] * domainScale[j];
    } 
}

static void ScaleMat(RealMat A) {
    Int i, j;

    if (domainScale!=nil) {
	for (i=1; i<=m; i++) {
	    for (j=1; j<=n; j++) A[i][j] *= domainScale[j];
	}
    }
    if (imageScale!=nil) {
	for (i=1; i<=m; i++) {
	    for (j=1; j<=n; j++) A[i][j] /= imageScale[i];
	}
    }
}

/*----------------------------------------------------------------------------
    QR decomposition by Householder reflections    
-----------------------------------------------------------------------------*/

static Real ColumnProduct(RealMat a, Int j, Int k, Int ml, Int mh) {
    Int i;
    Real temp = 0;
    for (i=ml; i<=mh; i++) temp += a[i][j] * a[i][k];
    return temp;
}

static Real RowProduct(RealMat a, Int i, Int k, Int nl, Int nh) {
    Int j;
    Real temp = 0;
    for (j=nl; j<=nh; j++) temp += a[i][j] * a[k][j];
    return temp;
}

static Bool Householder() {
    Int i, j, k, jmax, pMax;
    Real h, temp, t, dd;

    for (i=1; i<=n; i++) pivot[i] = i;
    signum = 1;
    pMax = p;

    if (m<=1 && n<=1) { d[1] = A[1][1]; cond = 1.0; return true; }
    for (j=1; j<=n; j++) d[j] = ColumnProduct(A, j, j, 1, m);

    for (k=1; k<=p; k++) {
	if (k!=n) {
	    h = d[k];
	    jmax = k;
	    for (j=k+1; j<=n; j++) {
		if (d[j] > h) { h = d[j]; jmax = j; }
	    }
	    if (jmax != k) {
		INTSWAP(pivot[k], pivot[jmax]);
		SwapColumns(A, jmax, k, 1, m);
		d[jmax] = d[k];
		signum = - signum;
	    }                       
	}
	h = ColumnProduct(A, k, k, k, m);
	t = sqrt(h);
	if (k==1) dd = t/cond;
	if (t<=dd) {
           p = k-1;               /* rank-reduction */
        }
	else if (k<m) {
	    temp = A[k][k];
	    d[k] = t = (temp>0) ? -t : t;
	    A[k][k] = temp - t;
	    if (k!=n) { 
		t = 1/(h-temp*t);
		for (j=k+1; j<=n; j++) {
		    temp = t * ColumnProduct(A, k, j, k, m);
		    for (i=k; i<=m; i++) A[i][j] -= temp * A[i][k];
		    d[j] = d[j] - SQR(A[k][j]);
		}   
	    }
	}   
	else d[k] = A[k][k];    
    }
    if (p==pMax) {
        cond = fabs(d[1] / d[p]);
        det = signum;
        for(i=1;i<=pMax;i++) det *=d[i];
        det = ( p%2 ) ? det : -det;
        if ( n>p) det = ((n+ pivot[n]) %2) ? -det: det;
    } else {
        det = 0.0;
    }
    return (p==pMax);
}

static void PrepareSolution() {
    Int i, j, k, q;
    Real temp;

    if (p>0 && p<n) {
        q = n-p;
        for (j=1; j<=q; j++) {
            for (i=p; i>=1; i--){
                temp = A[i][p+j];
                for (k=i+1; k<=p; k++) temp -= A[i][k] * V[k][j];
                V[i][j] = temp / d[i];
            }
            for (i=1; i<=j; i++) {
                temp = ColumnProduct(V, i, j, 1, p)
                         - RowProduct(L, i, j, 1, i-1);
                L[j][i] = (i!=j) ? temp/L[i][i] : sqrt(1+temp);
            }
        }       
    }
} /* PrepareSolution */

Bool QRDecompose(QRMat *qrMat) {
    Int pOld;
    Bool rankDefect;

    qrMat->cond = qrMat->condMax;
    Globalize(qrMat);

    pOld = p;
    CopyRealMat(qrMat->AUser, 1, m, 1, n, A);
    ScaleMat(A);

    rankDefect = !Householder();
    if (rankDefect) {
        Warning("QRDecompose: Matrix rank defect, assumed rank: %d,  numeric rank: %d", pOld, p);
	if ((pOld != 0) && (pOld != n)) { 
           FreeRealMat(V, 1, pOld, 1, n-pOld);
	   FreeRealMat(L, 1, n-pOld, 1, n-pOld);
        }
        if (p>pOld) p = pOld; /* shoudn't happen */
        if (p!=0) {
	   V = NewRealMat(1, p, 1, n-p);
	   L = NewRealLowMat(1, n-p, 1, n-p);
        }
    }
    PrepareSolution();

    qrMat->p 		= p;
    qrMat->V 		= V;
    qrMat->L 		= L;
    qrMat->det		= det;
    qrMat->cond 	= cond;
    qrMat->signum 	= signum;

    return !rankDefect;
}

/*----------------------------------------------------------------------------
    solution of the linear least squares problem (pseudo inverse)
-----------------------------------------------------------------------------*/
        
static void Solution(RealVec bIn, RealVec xOut)
{
    RealVec x = NewRealVec(1, n);
    RealVec u, y = x, z = x+p;
    Real  temp;
            
    Int i, j, q = n-p;
    
    if (p==0) for (i=1; i<=n; i++) xOut[i] = 0;
    else {
	u = NewRealVec(1, p);

	CopyRealVec(bIn, 1, m, b);
	ScaleImageVec(b, b);
	
    /* apply HH-transformation to the right side, b := Q * b   */
    
	for (j=1; j<=p && j<m; j++) {
	    temp = 0;
	    for (i=j; i<=m; i++) temp += A[i][j] * b[i];
	    temp = temp / (d[j] * A[j][j]);
	    for (i=j; i<=m; i++) b[i] += temp * A[i][j];
	}
	
    /* u := R^(-1) * c, where c = b[1..p]  */
	
	for (i=p; i>=1; i--) {
	    temp = b[i];
	    for (j=i+1; j<=p; j++) temp -= A[i][j] * u[j];
	    u[i] = temp/d[i];
	}
	
	MatTransVecMult(V, u, p, q, z);  
	SolveTransposedRealLowMat(L, z, q, z);
	SolveRealLowMat(L, z, q, z);
	    
    /* y := u - V * z     */
    
	for (i=1; i<=p; i++) {
	    y[i] = u[i];
	    for (j=1; j<=q; j++) y[i] -= V[i][j] * z[j];
	}   
	
    /* xOut := pivot^T * x  */
    
	for (i=1; i<=n; i++) xOut[pivot[i]] = x[i];
	DescaleDomainVec(xOut, xOut);
    }    
    FreeRealVec(x, 1, n);
    if (p!=0) FreeRealVec(u, 1, p);
}         

Bool QRSolve(QRMat *qrMat, RealVec b, RealVec x) {
    Globalize(qrMat);
    Solution(b, x);
    return true;
}


void QRKernelBasis(QRMat *qrMat, RealMat t)
{
    Real tmp;
    Int n, m, p, q, i, j;
    IntVec pivot = qrMat->domainPivot;

    n = qrMat->n;
    m = qrMat->m;
    p = qrMat->p;
    q = n-p;
    if (q <= 0) Warning("KernelBasis: Kernel = <0>");
    for (j=1; j<=q; j++) {
        for (i=1; i<=p; i++)    t[j][pivot[i]] = - qrMat->V[i][j];
        for (i=1; i<=q; i++)    t[j][pivot[p+i]] = (i==j) ? 1.0 : 0.0;
        OrthoProject(t, t[j], nil, j-1, n, t[j]);
        tmp = 1.0/Norm(t[j], 1, n);
        ScalarVecMult(t[j], tmp, n, t[j]);
    }
    for (j=1; j<=q; j++) {
       DescaleDomainVec(t[j], t[j]);
    }
}

Real QRLastResidual(QRMat *q) {
   Real tmp = 0.0;
   Int  i = q->p;

   for (++i;i<=q->m;i++) tmp += SQR(q->b[i]);
   return sqrt(tmp);
}
