/*
C
C  _______________________________________________________________
C
C*   Licence
C    =======
C
C    You may use or modify this code for your own non commercial
C    purposes for an unlimited time. 
C    In any case you should not deliver this code without a special 
C    permission of ZIB.
C    In case you intend to use the code commercially, we oblige you
C    to sign an according licence agreement with ZIB.
C
C
C  _______________________________________________________________
C
*/

#include <stdio.h>
#include <math.h>

#include "kask.h"
#include "kasktri.h"
#include "kaskgraph.h"

#define ABS(x)    (((x) <  0 ) ? -(x) : (x))
#define MAX(x,y)  (((x) < (y)) ?  (y) : (x))
#define MIN(x,y)  (((x) < (y)) ?  (x) : (y))
#define SIGN(x) ((x<ZERO)?-ONE:ONE)

#define EPS       0.000001

#define RUBBERBAND  0
#define RUBBERRECT  1

struct L_Tr_E
  {
   struct TR  *t;
   REAL        FarZ;
  };
typedef struct L_Tr_E L_Tr_E;

struct iIE
  {
   struct EDG  *e;
   REAL        x, y;
  };
typedef struct iIE iIE;

static L_Tr_E  *TrL  = nil;
static iIE     *iIEL = nil;

static REAL bCubeX[8] = { -1.0, -1.0,  1.0,  1.0, -1.0, -1.0,  1.0,  1.0 };
static REAL bCubeY[8] = { -1.0, -1.0, -1.0, -1.0,  1.0,  1.0,  1.0,  1.0 };
static REAL bCubeZ[8] = { -1.0,  1.0,  1.0, -1.0, -1.0,  1.0,  1.0, -1.0 };
static REAL tr_bCubeX[8], tr_bCubeY[8], tr_bCubeZ[8];
static int  bCubeSurf[6][5] = { 0, 1, 2, 3, 0,
                                0, 1, 5, 4, 0,
                                1, 2, 6, 5, 1,
                                2, 3, 7, 6, 2,
                                3, 0, 4, 7, 3,
                                4, 5, 6, 7, 4 };
static REAL bCubeSZ[8];

static int  len_TrL, No_L_Tr_E, No_iIE;

static REAL cDiff, fxDiff, fyDiff;

static REAL Xoff, Yoff, Zoff, scalX, scalY, scalZ;
static REAL V11, V12, V13, V21, V22, V23, V31, V32, V33, V34;
static REAL SinAng, CosAng;
static REAL cutX1, cutY1, cutX2, cutY2, lcutXi, lcutYi, lcutSol;
static REAL Minjac, Maxjac, Totjac, Meanjac;
static int  InnerCol;
static int  DrawatF;

static int toDraw = R_SOL;
extern int errorPlot;
 
static int UnMarkEdg(ed)
  EDG    *ed;
  {
   (ed->mark) = false;
   return true;
  } /* UnMarkEdg */

void ComputeScaling(graph,driver)
  GRAPHIC *graph;
  DRIVER *driver;
  {
	REAL dx = driver->right-driver->left,
		 dy = driver->top-driver->bottom,
		 udx = graph->right-graph->left,
		 udy = graph->top-graph->bottom,
		 xMargin = dx/25.0, yMargin = dx/25.0;
    REAL scaling, xScal, yScal, xTrans, yTrans;
 
    dx -= xMargin*2.0*SIGN(dx);
	dy -= yMargin*2.0*SIGN(dy);
	xScal = dx/udx;
    yScal = dy/udy;
    scaling = (fabs(xScal)>fabs(yScal))?yScal:xScal; /* Common scaling factor */
	scaling = fabs(scaling);

	if (fabs(dx/dy)>fabs(udx/udy))
	  xMargin += (fabs(dx)-fabs(udx)*scaling)*0.5*SIGN(xScal);
	else yMargin += (fabs(dy)-fabs(udy)*scaling)*0.5*SIGN(yScal);

	xTrans = (dx>0)?xMargin:xMargin-dx;
	yTrans = (dy>0)?yMargin:yMargin-dy;

	driver->xScal = SIGN(xScal)*scaling;
	driver->yScal = SIGN(yScal)*scaling;
	driver->xTrans = xTrans-(driver->xScal)*(graph->left);
	driver->yTrans = yTrans-(driver->yScal)*(graph->bottom);
    return;
  }
 
void DrawFrame()
  {
	(actDriver->Settings)(PENCOLOR,BLACK);
    (actDriver->Line)(actGraph->left, actGraph->bottom,
					   actGraph->right, actGraph->bottom);
    (actDriver->Line)(actGraph->right, actGraph->bottom,
					   actGraph->right, actGraph->top);
    (actDriver->Line)(actGraph->right, actGraph->top,
					   actGraph->left, actGraph->top);
    (actDriver->Line)(actGraph->left, actGraph->top,
					   actGraph->left, actGraph->bottom);
    return;
  }

static int AnySonNothingP(ed)
  EDG *ed;
  {
    EDG *son = ed->firstSon;
	if (son==nil) return (ed->type)==T_NOTHING;
	if (AnySonNothingP(son))
	  {
		son = son->next;
		if ((son->type)==T_NOTHING) return true;
		(actDriver->Line)((son->p1)->x, (son->p1)->y,
					 	 (son->p2)->x, (son->p2)->y);
		return true;
	  }
	son = son->next;
	if (AnySonNothingP(son))
	  {
		son = son->last;
		if ((son->type)==T_NOTHING) return true;
		(actDriver->Line)((son->p1)->x, (son->p1)->y,
					 	 (son->p2)->x, (son->p2)->y);
		return true;
	  }
	return false;
  }
static int DrEdge(ed)
  EDG *ed;
  {
    if (((ed->father)!=nil)||AnySonNothingP(ed)) return true;
	if ((ed->bound)==nil)
      (actDriver->Line)((ed->p1)->x, (ed->p1)->y,
						(ed->p2)->x, (ed->p2)->y);
    else
      (actDriver->Arc)((ed->p1)->x, (ed->p1)->y,
					   (ed->p2)->x, (ed->p2)->y,
					   (ed->bound)->x, (ed->bound)->y);
    return true;
  }
void DrawBound()
  {
    (actDriver->Settings)(PENCOLOR, BLUE);
    (actDriver->Settings)(PENSIZE, BIGSIZE);
    ApplyE(DrEdge, boundInit);
    return;
  }
 
void DrawTri()
  {
    (actDriver->Settings)(PENCOLOR, RED);
    (actDriver->Settings)(PENSIZE, SMALLSIZE);
	ApplyE(DrEdge, all);

    (actDriver->Settings)(PENCOLOR, MAGENTA);
    (actDriver->Settings)(PENSIZE, MEDIUMSIZE);
	ApplyE(DrEdge, initial);

    return;
  }
 
static int DrPoint(p)
  PT *p;
  {
    (actDriver->Line)((p->x)-cDiff, (p->y)-cDiff,
					   (p->x)+cDiff, (p->y)+cDiff);
    (actDriver->Line)((p->x)-cDiff, (p->y)+cDiff,
        			   (p->x)+cDiff, (p->y)-cDiff);
    return true;
  }
void DrawPoint()
  {
    cDiff = fabs((actGraph->top)-(actGraph->bottom))/50.0;
	(actDriver->Settings)(PENCOLOR, MAGENTA);
    ApplyP(DrPoint, all);
    return;
  }
 
static int DrIndexP(p)
  PT *p;
  { 
    sprintf(globBuf, "%d", p->indexP);
    (actDriver->String)((p->x)+fxDiff, (p->y)+fyDiff, globBuf);
    return true;
  }
void DrawIndex()
  { 
    fxDiff = ((actGraph->maxRight)-(actGraph->maxLeft))/50.0;
	fyDiff = ((actGraph->maxTop)-(actGraph->maxBottom))/50.0;
	(actDriver->Settings)(FONTSIZE, SMALLSIZE);
    ApplyP(DrIndexP, all);
    return;
  }
 
static void CompLineLevels()
  {
    REAL xMax = RMIN, xMin = RMAX, diff, xVal;
    int k;
	PT *p = actTriang->firstPoint;
 
    while (p!=nil)
      {
        if (RA(p,toDraw)>xMax) xMax = RA(p,toDraw);
        if (RA(p,toDraw)<xMin) xMin = RA(p,toDraw);
		p = p->next;
      }
    if (actGraph->percentage) xMin = 0.0;
	diff = (xMax-xMin)/((actGraph->levels)+1);
    xVal = xMin+diff*((actGraph->percentage)?1.0:0.5);
    for (k = 0; k<(actGraph->levels); k++)
      {
        (actGraph->levelsAt)[k] = xVal;
        xVal += diff;
      }
	sprintf(globBuf, "Graph: computed level lines (%9.3e - %9.3e)%9.3e\n",
			xMin,xMax,diff);
	ZIBStdOut(globBuf);
    return;
  }

static void DrContourLine(x,ed1,ed2)
  REAL x;
  EDG *ed1, *ed2;
  {
    double x1,y1,x2,y2,fac,quot;
    PT *p11 = ed1->p1, *p12 = ed1->p2, *p21 = ed2->p1,
       *p22 = ed2->p2;

    quot=RA(p12,toDraw)-RA(p11,toDraw);
    if (fabs(quot)<SMALL) quot=0.5;
    fac = (x-RA(p11,toDraw))/quot;
    x1 = (p11->x)+fac*((p12->x)-(p11->x));
    y1 = (p11->y)+fac*((p12->y)-(p11->y));
    quot=RA(p22,toDraw)-RA(p21,toDraw);
    if (fabs(quot)<SMALL) quot=0.5;
    fac = (x-RA(p21,toDraw))/quot;
    x2 = (p21->x)+fac*((p22->x)-(p21->x));
    y2 = (p21->y)+fac*((p22->y)-(p21->y));
    (actDriver->Line)(x1, y1, x2, y2);
 
    return;
  }

static int DrSol(t)
  TR *t;
  {
   int  dec, i;
   REAL zl, z1, z2, z3;

   z1 = RA(t->p1,toDraw); z2 = RA(t->p2,toDraw); z3 = RA(t->p3,toDraw);
 
   for (i = 0; i < (actGraph->levels); i++) /* loop through all levels */
       {
        dec = 0;
        zl = (actGraph->levelsAt)[i];
        if (z1 > zl) dec += 1;
        if (z2 > zl) dec += 2;
        if (z3 > zl) dec += 4;

        if (dec == 0)  break;
        else if ((dec == 1) || (dec == 6))  DrContourLine(zl,t->e3,t->e2);
        else if ((dec == 2) || (dec == 5))  DrContourLine(zl,t->e3,t->e1);
        else if ((dec == 3) || (dec == 4))  DrContourLine(zl,t->e1,t->e2);
       }
   return true;
  } /* DrSol */
 
void DrawSol()
  {
	int localP = false;

	if ((actGraph->levelsAt)==nil)
	  {
	    actGraph->levelsAt = (REAL*)ZIBAlloc((long)(actGraph->levels)*sizeof(REAL));
		if ((actGraph->levelsAt)==nil)
		  { ZIBStdOut("Graph: not enough memory (DrawSol)\n"); return; }
		CompLineLevels();
		localP = true;
	  }
	else
	  {
		sprintf(globBuf,"User defined level lines (%e - %e)%e\n",
			 	(actGraph->levelsAt)[0],
				(actGraph->levelsAt)[(actGraph->levels)-1],
				(actGraph->levelsAt)[1]-(actGraph->levelsAt)[0]);
		ZIBStdOut(globBuf);
	  }
    (actDriver->Settings)(PENCOLOR, BLACK);
    (actDriver->Settings)(PENSIZE, MEDIUMSIZE);
	toDraw = errorPlot?R_RES:R_SOL;
    ApplyT(DrSol,all);
	if (localP)
	  {
	    ZIBFree((PTR)(actGraph->levelsAt));
		actGraph->levelsAt = nil;
	  }
    return;
  }

static void CompInters(x,y,p1,p2,z)
  REAL *x, *y;
  PT   *p1, *p2;
  REAL z;
  {
    double fac, quot;

    quot = RA(p2,toDraw) - RA(p1,toDraw);
    if (fabs(quot) < SMALL) quot = 0.5;
    fac = (z - RA(p1,toDraw)) / quot;
    *x = (p1->x) + fac * ((p2->x) - (p1->x));
    *y = (p1->y) + fac * ((p2->y) - (p1->y));
 
    return;
  } /* CompInters */

static int DrTemp(t)
  TR *t;
  {
   int  Trifilled, dec, col, i;
   REAL zl, z1, z2, z3, x1, y1, x2, y2, xAr[4], yAr[4];

   Trifilled = false;
   z1 = RA(t->p1,toDraw); z2 = RA(t->p2,toDraw); z3 = RA(t->p3,toDraw);
 
   for (i = 0; i < (actGraph->levels); i++) /* loop through all levels */
       {
        dec = 0;
        zl = (actGraph->levelsAt)[i];
        if (z1 > zl) dec += 1;
        if (z2 > zl) dec += 2;
        if (z3 > zl) dec += 4;

        col = (actDriver->firstGray) + (actDriver->noOfGrays) - i - 1;

        if (!Trifilled && (dec < 7))
           {
            xAr[0] = (t->p1)->x; yAr[0] = (t->p1)->y;
            xAr[1] = (t->p2)->x; yAr[1] = (t->p2)->y;
            xAr[2] = (t->p3)->x; yAr[2] = (t->p3)->y;
            (actDriver->Fill)(xAr, yAr, 3, col);
            Trifilled = true;
           }
        if (dec == 0)
           {
            break;
           }
        else if (dec == 1)
           {
            CompInters(&x1,&y1,t->p1,t->p2,zl);
            CompInters(&x2,&y2,t->p1,t->p3,zl);
            xAr[0] = x1; yAr[0] = y1;
            xAr[1] = x2; yAr[1] = y2;
            xAr[2] = (t->p1)->x; yAr[2] = (t->p1)->y;
            (actDriver->Fill)(xAr, yAr, 3, col-1);
           }
        else if (dec == 2)
           {
            CompInters(&x1,&y1,t->p2,t->p1,zl);
            CompInters(&x2,&y2,t->p2,t->p3,zl);
            xAr[0] = x1; yAr[0] = y1;
            xAr[1] = x2; yAr[1] = y2;
            xAr[2] = (t->p2)->x; yAr[2] = (t->p2)->y;
            (actDriver->Fill)(xAr, yAr, 3, col-1);
           }
        else if (dec == 3)
           {
            CompInters(&x1,&y1,t->p3,t->p2,zl);
            CompInters(&x2,&y2,t->p3,t->p1,zl);
            xAr[0] = x1; yAr[0] = y1;
            xAr[1] = x2; yAr[1] = y2;
            xAr[2] = (t->p1)->x; yAr[2] = (t->p1)->y;
            xAr[3] = (t->p2)->x; yAr[3] = (t->p2)->y;
            (actDriver->Fill)(xAr, yAr, 4, col-1);
           }
        else if (dec == 4)
           {
            CompInters(&x1,&y1,t->p3,t->p2,zl);
            CompInters(&x2,&y2,t->p3,t->p1,zl);
            xAr[0] = x1; yAr[0] = y1;
            xAr[1] = x2; yAr[1] = y2;
            xAr[2] = (t->p3)->x; yAr[2] = (t->p3)->y;
            (actDriver->Fill)(xAr, yAr, 3, col-1);
           }
        else if (dec == 5)
           {
            CompInters(&x1,&y1,t->p2,t->p1,zl);
            CompInters(&x2,&y2,t->p2,t->p3,zl);
            xAr[0] = x1; yAr[0] = y1;
            xAr[1] = x2; yAr[1] = y2;
            xAr[2] = (t->p3)->x; yAr[2] = (t->p3)->y;
            xAr[3] = (t->p1)->x; yAr[3] = (t->p1)->y;
            (actDriver->Fill)(xAr, yAr, 4, col-1);
           }
        else if (dec == 6)
           {
            CompInters(&x1,&y1,t->p1,t->p2,zl);
            CompInters(&x2,&y2,t->p1,t->p3,zl);
            xAr[0] = x1; yAr[0] = y1;
            xAr[1] = x2; yAr[1] = y2;
            xAr[2] = (t->p3)->x; yAr[2] = (t->p3)->y;
            xAr[3] = (t->p2)->x; yAr[3] = (t->p2)->y;
            (actDriver->Fill)(xAr, yAr, 4, col-1);
           }
        else if (dec == 7)
           {
            if (i == (actGraph->levels)-1)
               {
                xAr[0] = (t->p1)->x; yAr[0] = (t->p1)->y;
                xAr[1] = (t->p2)->x; yAr[1] = (t->p2)->y;
                xAr[2] = (t->p3)->x; yAr[2] = (t->p3)->y;
                (actDriver->Fill)(xAr, yAr, 3, col-1);
                Trifilled = true;
               }
           }
       } /* loop through all levels */
   return true;
  } /* DrTemp */

void DrawTemp()
  {
	int lev;
	lev = actDriver->noOfGrays - 1;
	if ((lev<(actGraph->levels))&&((actGraph->levelsAt)!=nil))
     {
	   ZIBFree((PTR)(actGraph->levelsAt));
	   actGraph->levelsAt = nil;
	 }
	(actGraph->levels) = lev;
	if ((actGraph->levelsAt) == nil)
      {
       actGraph->levelsAt = (REAL*)ZIBAlloc((long)(lev*sizeof(REAL)));
       if ((actGraph->levelsAt) == nil)
    	  { ZIBStdOut("Graph: not enough memory (DrawTemp)\n"); return; }
      }
   toDraw = errorPlot?R_RES:R_SOL;
   CompLineLevels();
   ApplyT(DrTemp,all);

   return;
  }

static REAL dist(x1,y1,x2,y2)
   REAL    x1, y1, x2, y2;
  {
   REAL    dx, dy;

   dx = x1 - x2; dy = y1 - y2;
   return (sqrt(dx * dx + dy * dy));
  } /* dist */

static void VTr_bCube()
  {
   int     i, j, k;
   REAL    factor = REALPI / 180.;
   REAL    aux;
   double  angle, dx, dy;

   for (i = 0; i < 8; i++)
       {
        tr_bCubeX[i] = V11*bCubeX[i] + V12*bCubeY[i] + V13*bCubeZ[i];
        tr_bCubeY[i] = V21*bCubeX[i] + V22*bCubeY[i] + V23*bCubeZ[i];
        tr_bCubeZ[i] = V31*bCubeX[i] + V32*bCubeY[i] + V33*bCubeZ[i] + V34;
       }
   dx = (double) (tr_bCubeX[4] - tr_bCubeX[0]);
   dy = (double) (tr_bCubeY[4] - tr_bCubeY[0]);
   angle = ABS(dy) > EPS ? (-atan2(dx,dy)) : (0.0);
   SinAng = (REAL) sin(angle); CosAng = (REAL) cos(angle);
   angle = angle / factor;
   for (i = 0; i < 8; i++)
       {
        aux = tr_bCubeX[i];
        tr_bCubeX[i] = aux *  CosAng + tr_bCubeY[i] *  SinAng;
        tr_bCubeY[i] = aux * -SinAng + tr_bCubeY[i] *  CosAng;
       }
   for (i = 0; i < 6; i++)
       {
        j = bCubeSurf[i][0]; k = bCubeSurf[i][2];
        bCubeSZ[i] = (tr_bCubeZ[j] + tr_bCubeZ[k]) / 2.0;
       }
  } /* VTr_bCube */

static int ViewTransPoint(p)
  PT *p;
  {
   REAL    xn, yn, zn, auxx, auxy;

   xn = (((p->x) - Xoff) / scalX - 1.0);
   zn = -(((p->y) - Yoff) / scalY - 1.0);
   yn = ((RA(p,R_SOL) - Zoff) / scalZ - 1.0);
   auxx = V11 * xn + V12 * yn + V13 * zn;
   auxy = V21 * xn + V22 * yn + V23 * zn;
   RA(p,R_TRX) = auxx *  CosAng + auxy *  SinAng; 
   RA(p,R_TRY) = auxx * -SinAng + auxy *  CosAng;
   RA(p,R_TRZ) = V31 * xn + V32 * yn + V33 * zn + V34;

   return true;
  } /* ViewTransPoint */

static void Dr_bCube(background)
  int background;
  {
   int     i, j, k, l;

   if (background)
      {
       for (i = 0; i < 6; i++)
           {
            if (bCubeSZ[i] < 0.0)
               {
                for (j = 0; j < 4; j++)
                    {
                     k = bCubeSurf[i][j]; l = bCubeSurf[i][j+1];
                     if ((k == 1) || (l == 1))
                         (actDriver->Settings)(PENSIZE, BIGSIZE);
                     else
                         (actDriver->Settings)(PENSIZE, SMALLSIZE);
                     (actDriver->Line)(tr_bCubeX[k],tr_bCubeY[k],
                                       tr_bCubeX[l],tr_bCubeY[l]);
                    }
               }
           }
      }
   else
      {
       for (i = 0; i < 6; i++)
           {
            if (bCubeSZ[i] > 0.0)
               {
                for (j = 0; j < 4; j++)
                    {
                     k = bCubeSurf[i][j]; l = bCubeSurf[i][j+1];
                     if ((k == 1) || (l == 1))
                         (actDriver->Settings)(PENSIZE, BIGSIZE);
                     else
                         (actDriver->Settings)(PENSIZE, SMALLSIZE);
                     (actDriver->Line)(tr_bCubeX[k],tr_bCubeY[k],
                                       tr_bCubeX[l],tr_bCubeY[l]);
                    }
               }
           }
      }
  } /* Dr_bCube */

static REAL GrJacobi(p1x,p1y,p2x,p2y,p3x,p3y)
  REAL  p1x, p1y, p2x, p2y, p3x, p3y;
  {
    return (p2x - p1x) * (p3y - p1y) - (p3x - p1x) * (p2y - p1y);
  } /* GrJacobi */

static REAL IpZ(x,y,x1,y1,z1,x2,y2,z2)
   REAL    x, y, x1, y1, z1, x2, y2, z2;
  {
   REAL    fac;

   fac = dist(x1,y1,x,y) / dist(x1,y1,x2,y2);
   return (z1 + fac * (z2 - z1));
  } /* IpZ */

static int lineinters1(in,xi,yi,l1x1,l1y1,l1x2,l1y2,l2x1,l2y1,l2x2,l2y2)
   int     *in;
   REAL    *xi, *yi, l1x1, l1y1, l1x2, l1y2, l2x1, l2y1, l2x2, l2y2;
  {
   int     found;
   REAL    del, rmu;
   
   del = (l1x1 - l1x2) * (l2y2 - l2y1) - (l1y1 - l1y2) * (l2x2 - l2x1);
   if (fabs(del) > EPS)
      {
       rmu = ((l2y2-l2y1) * (l2x2-l1x2) - (l2x2-l2x1) * (l2y2-l1y2)) / del;
       *xi = rmu * l1x1 + (1.0 - rmu) * l1x2;
       *yi = rmu * l1y1 + (1.0 - rmu) * l1y2;
       /* dirty, but it works */
       if (fabs(l1x1-l1x2) < EPS) { l1x1 = l1x1-EPS/2.; l1x2 = l1x2+EPS/2.; }
       if (fabs(l2x1-l2x2) < EPS) { l2x1 = l2x1-EPS/2.; l2x2 = l2x2+EPS/2.; }
       if (fabs(l1y1-l1y2) < EPS) { l1y1 = l1y1-EPS/2.; l1y2 = l1y2+EPS/2.; }
       if (fabs(l2y1-l2y2) < EPS) { l2y1 = l2y1-EPS/2.; l2y2 = l2y2+EPS/2.; }
       found = ((*xi>=l1x1 && *xi<=l1x2) || (*xi<=l1x1 && *xi>=l1x2)) &&
               ((*xi>=l2x1 && *xi<=l2x2) || (*xi<=l2x1 && *xi>=l2x2)) &&
               ((*yi>=l1y1 && *yi<=l1y2) || (*yi<=l1y1 && *yi>=l1y2)) &&
               ((*yi>=l2y1 && *yi<=l2y2) || (*yi<=l2y1 && *yi>=l2y2));
       *in = false;
      }
   else
      {
       del = dist(l1x1,l1y1,l1x2,l1y2);
       if (fabs(del - (dist(l2x1,l2y1,l1x1,l1y1) +
                       dist(l2x1,l2y1,l1x2,l1y2))) < EPS)
          { found = true; *in = true; *xi = l2x1; *yi = l2y1; }
       else if (fabs(del - (dist(l2x2,l2y2,l1x1,l1y1) +
                            dist(l2x2,l2y2,l1x2,l1y2))) < EPS)
          { found = true; *in = true; *xi = l2x2; *yi = l2y2; }
       else
          { found = false; *in = false; }
      }

   return found;
  } /* lineinters1 */

static int lineinters2(in,xi,yi,l1x1,l1y1,l1x2,l1y2,l2x1,l2y1,l2x2,l2y2)
   int     *in;
   REAL    *xi, *yi, l1x1, l1y1, l1x2, l1y2, l2x1, l2y1, l2x2, l2y2;
  {
   int     found;
   REAL    del, rmu;
   
   del = (l1x1 - l1x2) * (l2y2 - l2y1) - (l1y1 - l1y2) * (l2x2 - l2x1);
   if (fabs(del) > EPS)
      {
       rmu = ((l2y2-l2y1) * (l2x2-l1x2) - (l2x2-l2x1) * (l2y2-l1y2)) / del;
       *xi = rmu * l1x1 + (1.0 - rmu) * l1x2;
       *yi = rmu * l1y1 + (1.0 - rmu) * l1y2;
       /* dirty, but it works */
       if (fabs(l2x1-l2x2) < EPS) { l2x1 = l2x1-EPS/2.; l2x2 = l2x2+EPS/2.; }
       if (fabs(l2y1-l2y2) < EPS) { l2y1 = l2y1-EPS/2.; l2y2 = l2y2+EPS/2.; }
       found = ((*xi>=l2x1 && *xi<=l2x2) || (*xi<=l2x1 && *xi>=l2x2)) &&
               ((*yi>=l2y1 && *yi<=l2y2) || (*yi<=l2y1 && *yi>=l2y2));
       *in = false;
      }
   else
      {
       del = dist(l1x1,l1y1,l1x2,l1y2);
       if (fabs(del - (dist(l2x1,l2y1,l1x1,l1y1) +
                       dist(l2x1,l2y1,l1x2,l1y2))) < EPS)
          { found = true; *in = true; *xi = l2x1; *yi = l2y1; }
       else if (fabs(del - (dist(l2x2,l2y2,l1x1,l1y1) +
                            dist(l2x2,l2y2,l1x2,l1y2))) < EPS)
          { found = true; *in = true; *xi = l2x2; *yi = l2y2; }
       else
          { found = false; *in = false; }
      }

   return found;
  } /* lineinters2 */

static int TesT(t)
  TR *t;
  {
   REAL    xt[3], yt[3], jac;

   xt[0] = RA(t->p1,R_TRX); yt[0] = RA(t->p1,R_TRY);
   xt[1] = RA(t->p2,R_TRX); yt[1] = RA(t->p2,R_TRY);
   xt[2] = RA(t->p3,R_TRX); yt[2] = RA(t->p3,R_TRY);
   jac = GrJacobi(xt[0],yt[0],xt[1],yt[1],xt[2],yt[2]);
   jac = fabs(jac);
   Minjac = MIN(Minjac,jac);
   Maxjac = MAX(Maxjac,jac);
   Totjac += jac;

   return true;
  } /* TesT */

static int SortTri(t,currFarZ)
  TR      *t;
  REAL    currFarZ;
  {
   L_Tr_E  *le, *len;
   REAL    midFarZ;
   int     lwb, upb, mid, InsPos, Poff, i;
   int     NotFound;

   if (No_L_Tr_E == 0)
      {
       le = &TrL[No_L_Tr_E++];
       (le->t) = t;
       (le->FarZ) = currFarZ;
      }
   else
      {
       lwb = 0; upb = No_L_Tr_E - 1;
       NotFound = true;
       while (NotFound)
             {
              mid = (upb + lwb) / 2;
              le = &TrL[mid];
              midFarZ = (le->FarZ);
              if (currFarZ > midFarZ)
                 { lwb = mid; Poff = 1; }
              else 
                 { upb = mid; Poff = 0; }
              if (upb == lwb)
                 {
                  NotFound = false;
                 }
              else if (upb == lwb+1)
                 {
                  if (Poff > 0)
                     {
                      le = &TrL[upb];
                      if (currFarZ >= (le->FarZ))  Poff += 1;
                     }
                  else
                     {
                      le = &TrL[lwb];
                      if (currFarZ <= (le->FarZ))  Poff -= 1;
                     }
                  NotFound = false;
                 }
             } /* while */
       if (No_L_Tr_E < len_TrL)
          {
           InsPos = mid + Poff;
           for (i = No_L_Tr_E; i > InsPos; i--)
               {
                le = &TrL[i-1]; len = &TrL[i];
                (len->t) = (le->t);
                (len->FarZ) = (le->FarZ);
               }
           le = &TrL[InsPos];
           (le->t) = t;
           (le->FarZ) = currFarZ;
           No_L_Tr_E += 1;
          }
      }

   return true;
  } /* SortTri */

static void QuartSort(t,p1x,p1y,p1z,p2x,p2y,p2z,p3x,p3y,p3z,jac)
   TR     *t;
   REAL   p1x, p1y, p1z, p2x, p2y, p2z, p3x, p3y, p3z, jac;
  {
   REAL   p12x, p12y, p12z, p23x, p23y, p23z, p31x, p31y, p31z, njac;
   REAL   currFarZ;

   p12x = (p1x + p2x) / 2.0; p12y = (p1y + p2y) / 2.0; p12z = (p1z + p2z) / 2.0;
   p23x = (p2x + p3x) / 2.0; p23y = (p2y + p3y) / 2.0; p23z = (p2z + p3z) / 2.0;
   p31x = (p3x + p1x) / 2.0; p31y = (p3y + p1y) / 2.0; p31z = (p3z + p1z) / 2.0;
   njac = jac / 4.0;
   if (njac > Meanjac)
      {
       QuartSort(t,p1x,p1y,p1z,p12x,p12y,p12z,p31x,p31y,p31z,njac);
       QuartSort(t,p2x,p2y,p2z,p23x,p23y,p23z,p12x,p12y,p12z,njac);
       QuartSort(t,p3x,p3y,p3z,p31x,p31y,p31z,p23x,p23y,p23z,njac);
       QuartSort(t,p12x,p12y,p12z,p23x,p23y,p23z,p31x,p31y,p31z,njac);
      }
   else
      {
       currFarZ = (p1z + p12z + p31z) / 3.0;
       (t->mark) += 1;
       SortTri(t,currFarZ);
       currFarZ = (p2z + p23z + p12z) / 3.0;
       (t->mark) += 1;
       SortTri(t,currFarZ);
       currFarZ = (p3z + p31z + p23z) / 3.0;
       (t->mark) += 1;
       SortTri(t,currFarZ);
       currFarZ = (p12z + p23z + p31z) / 3.0;
       (t->mark) += 1;
       SortTri(t,currFarZ);
      }
  } /* QuartSort */

static int DivEtSort(t)
   TR *t;
  {
   REAL    xt[3], yt[3], jac, currFarZ;

   (t->mark) = 0;
   xt[0] = RA(t->p1,R_TRX); yt[0] = RA(t->p1,R_TRY);
   xt[1] = RA(t->p2,R_TRX); yt[1] = RA(t->p2,R_TRY);
   xt[2] = RA(t->p3,R_TRX); yt[2] = RA(t->p3,R_TRY);
   jac = GrJacobi(xt[0],yt[0],xt[1],yt[1],xt[2],yt[2]);
   jac = fabs(jac);
   if (jac > Meanjac)
      {
       QuartSort(t,xt[0],yt[0],RA(t->p1,R_TRZ),xt[1],yt[1],RA(t->p2,R_TRZ),
                 xt[2],yt[2],RA(t->p3,R_TRZ),jac);
      }
   else
      {
       currFarZ = (RA(t->p1,R_TRZ) + RA(t->p2,R_TRZ) + RA(t->p3,R_TRZ)) / 3.0;
       (t->mark) += 1;
       SortTri(t,currFarZ);
      }

   return true;
  } /* DivEtSort */

static int Dr3D(t)
  TR *t;
  {
   REAL    xt[3], yt[3], jac;
   int     col, drawOK;

   drawOK = (DrawatF && ((t->mark) > 0)) || (!DrawatF && (--(t->mark) == 0));
   if (drawOK)
      {
       (t->mark) = 0; 
       xt[0] = RA(t->p1,R_TRX); yt[0] = RA(t->p1,R_TRY);
       xt[1] = RA(t->p2,R_TRX); yt[1] = RA(t->p2,R_TRY);
       xt[2] = RA(t->p3,R_TRX); yt[2] = RA(t->p3,R_TRY);
       jac = GrJacobi(xt[0],yt[0],xt[1],yt[1],xt[2],yt[2]);
       col = jac <= 0.0 ? InnerCol : WHITE;
       (actDriver->Fill)(xt,yt,3,col);
       (actDriver->Line)(xt[0],yt[0],xt[1],yt[1]);
       (actDriver->Line)(xt[1],yt[1],xt[2],yt[2]);
       (actDriver->Line)(xt[2],yt[2],xt[0],yt[0]);
      }

   return true;
  } /* Dr3D */

static int getIntersE(e)
   EDG     *e;
  {
   iIE     *le;
   REAL    xi, yi;
   int     found, in;

   found = lineinters2(&in,&xi,&yi,cutX1,cutY1,cutX2,cutY2,((e->p1)->x),
                       ((e->p1)->y),((e->p2)->x),((e->p2)->y));
   if (found)
      {
       le = &iIEL[No_iIE++];
       (le->e) = e; (le->x) = xi; (le->y) = yi;
      }

   return true;
  } /* getIntersE */

static EDG *fadr_path(currEdge,prevTri,lx,ly,lSol,ll)
   EDG     *currEdge;
   TR      *prevTri;
   REAL    lx, ly, lSol, ll;
  {
   EDG     *nEdge[2], *bndEdge;
   TR      *currTri;
   REAL    xi, yi, Soli, li, d1, d2;
   int     found, in, i;

   (currEdge->mark) = true;
   currTri  = (currEdge->t1) == prevTri  ? (currEdge->t2) : (currEdge->t1);
   nEdge[0] = (currTri->e1)  == currEdge ? (currTri->e2)  : (currTri->e1);
   nEdge[1] = (currTri->e3)  == currEdge ? (currTri->e2)  : (currTri->e3);
   i = 0;
   do {
       found = lineinters2(&in,&xi,&yi,cutX1,cutY1,cutX2,cutY2,
                           ((nEdge[i]->p1)->x),((nEdge[i]->p1)->y),
                           ((nEdge[i]->p2)->x),((nEdge[i]->p2)->y));
      } while (!(found || in) && (++i < 2));
   if (found && !(nEdge[i]->mark))
      {
       if (in)
          {
           d1 = dist(((nEdge[i]->p1)->x),((nEdge[i]->p1)->y),lx,ly);
           d2 = dist(((nEdge[i]->p2)->x),((nEdge[i]->p2)->y),lx,ly);
           if (d1 < d2)
              {
               xi = ((nEdge[i]->p1)->x); yi = ((nEdge[i]->p1)->y);
               Soli = RA((nEdge[i]->p1),R_SOL);
               li = dist(cutX1,cutY1,xi,yi);
               (actDriver->Line)(ll,lSol,li,Soli);
               lx = xi; lSol = Soli; ll = li;
               xi = ((nEdge[i]->p2)->x); yi = ((nEdge[i]->p2)->y);
               Soli = RA((nEdge[i]->p2),R_SOL);
              }
           else
              {
               xi = ((nEdge[i]->p2)->x); yi = ((nEdge[i]->p2)->y);
               Soli = RA((nEdge[i]->p2),R_SOL);
               li = dist(cutX1,cutY1,xi,yi);
               (actDriver->Line)(ll,lSol,li,Soli);
               lx = xi; lSol = Soli; ll = li;
               xi = ((nEdge[i]->p1)->x); yi = ((nEdge[i]->p1)->y);
               Soli = RA((nEdge[i]->p1),R_SOL);
              }
          }
       else
          {
           Soli = IpZ(xi,yi,((nEdge[i]->p1)->x),((nEdge[i]->p1)->y),
                      RA((nEdge[i]->p1),R_SOL),((nEdge[i]->p2)->x),
                      ((nEdge[i]->p2)->y),RA((nEdge[i]->p2),R_SOL));
          }
       lcutXi = xi; lcutYi = yi; lcutSol = Soli;
       li = dist(cutX1,cutY1,xi,yi);
       (actDriver->Line)(ll,lSol,li,Soli);
       if ((nEdge[i]->boundP) == INTERIOR)
          {
           bndEdge = fadr_path(nEdge[i],currTri,xi,yi,Soli,li);
           if ((bndEdge == nil) && (i == 0))
              {
               found = lineinters2(&in,&xi,&yi,cutX1,cutY1,cutX2,cutY2,
                                   ((nEdge[1]->p1)->x),((nEdge[1]->p1)->y),
                                   ((nEdge[1]->p2)->x),((nEdge[1]->p2)->y));
               if (found && !(nEdge[1]->mark))
                  {
                   if (in)
                      {
                       d1 = dist(((nEdge[1]->p1)->x),((nEdge[1]->p1)->y),lx,ly);
                       d2 = dist(((nEdge[1]->p2)->x),((nEdge[1]->p2)->y),lx,ly);
                       if (d1 < d2)
                          {
                           xi = ((nEdge[1]->p1)->x); yi = ((nEdge[1]->p1)->y);
                           Soli = RA((nEdge[1]->p1),R_SOL);
                           li = dist(cutX1,cutY1,xi,yi);
                           (actDriver->Line)(ll,lSol,li,Soli);
                           lx = xi; lSol = Soli; ll = li;
                           xi = ((nEdge[1]->p2)->x); yi = ((nEdge[1]->p2)->y);
                           Soli = RA((nEdge[1]->p2),R_SOL);
                          }
                       else
                          {
                           xi = ((nEdge[1]->p2)->x); yi = ((nEdge[1]->p2)->y);
                           Soli = RA((nEdge[1]->p2),R_SOL);
                           li = dist(cutX1,cutY1,xi,yi);
                           (actDriver->Line)(ll,lSol,li,Soli);
                           lx = xi; lSol = Soli; ll = li;
                           xi = ((nEdge[1]->p1)->x); yi = ((nEdge[1]->p1)->y);
                           Soli = RA((nEdge[1]->p1),R_SOL);
                          }
                      }
                   else
                      {
                       Soli = IpZ(xi,yi,((nEdge[1]->p1)->x),((nEdge[1]->p1)->y),
                                  RA((nEdge[1]->p1),R_SOL),((nEdge[1]->p2)->x),
                                  ((nEdge[1]->p2)->y),RA((nEdge[1]->p2),R_SOL));
                      }
                   lcutXi = xi; lcutYi = yi; lcutSol = Soli;
                   li = dist(cutX1,cutY1,xi,yi);
                   (actDriver->Line)(ll,lSol,li,Soli);
                   if ((nEdge[1]->boundP) == INTERIOR)
                      {
                       bndEdge = fadr_path(nEdge[1],currTri,xi,yi,Soli,li);
                      }
                   else
                      {
                       bndEdge = nEdge[1];
                      }
                  }
               else
                  {
                   bndEdge = nil;
                  }
              }
          }
       else
          {
           bndEdge = nEdge[i];
          }
      }
   else
      {
       bndEdge = nil;
      }

   return bndEdge;
  } /* fadr_path */

static EDG *DrCut(e)
   EDG     *e;
  {
   EDG     *currEdge, *prevEdge, *bndEdge;
   REAL    xi, yi, Soli, li;
   int     found, in;

   currEdge = e;
   while (currEdge != nil)
       {
        found = lineinters2(&in,&xi,&yi,cutX1,cutY1,cutX2,cutY2,
                            ((currEdge->p1)->x),((currEdge->p1)->y),
                            ((currEdge->p2)->x),((currEdge->p2)->y));
        if (found)
           {
            prevEdge = currEdge; currEdge = (currEdge->firstSon);
           }
        else
           {
            prevEdge = currEdge; currEdge = (currEdge->next);
           }
       }
   if (found || in)
      {
       currEdge = prevEdge;
       Soli = IpZ(xi,yi,((currEdge->p1)->x),((currEdge->p1)->y),
                  RA((currEdge->p1),R_SOL),((currEdge->p2)->x),
                  ((currEdge->p2)->y),RA((currEdge->p2),R_SOL));
       (actDriver->Settings)(PENCOLOR, BLACK);
       (actDriver->Settings)(PENSIZE, SMALLSIZE);
       li = dist(cutX1,cutY1,xi,yi);
       (actDriver->Line)(li,(actGraph->bottom),li,Soli);
       (actDriver->Settings)(PENSIZE, MEDIUMSIZE);
       bndEdge = fadr_path(currEdge,(TR*)nil,xi,yi,Soli,li);
       if (bndEdge != nil)
          {
           (actDriver->Settings)(PENSIZE, SMALLSIZE);
           li = dist(cutX1,cutY1,lcutXi,lcutYi);
           (actDriver->Line)(li,(actGraph->bottom),li,lcutSol);
           while ((bndEdge->father) != nil) bndEdge = (bndEdge->father);
          }
      }
   else
      {
       bndEdge = nil;
      }

   return bndEdge;
  } /* DrCut */

void DrawCut()
  {
   EDG     *currEdge, *bndEdgefF;
   iIE     *le, *lec;
   PT      *p;
   REAL    MaxZ, MinZ, lenCut;
   REAL    save_left, save_bottom, save_right, save_top;
   REAL    xt, yt, xin1, yin1, xin2, yin2, aux;
   int     i, j;

   if (iIEL != nil) ZIBFree((PTR)iIEL);
   iIEL = (iIE *)ZIBAlloc((long)(actTriang->noOfInitEdges) * sizeof(iIE));
   if (iIEL == nil)
      { ZIBStdOut("Graph: not enough memory (Cut)\n"); return; }
   No_iIE = 0;

   if ((actDriver->GetXY) != nil) /* Get Cutline */
      {
       DrawTri(); DrawSol();
       xt = (actGraph->left) + ((actGraph->right) - (actGraph->left)) / 40.0;
       yt = (actGraph->bottom) + ((actGraph->top) - (actGraph->bottom)) / 40.0;
       (actDriver->String)(xt,yt,"Enter Cutline or hit <Return> for Default");
       if ((actDriver->GetXY)(&xin1,&yin1))
          {
           if ((actDriver->GetRXY)(&xin2,&yin2,xin1,yin1,RUBBERBAND))
              {
               (actGraph->cutX1) = xin1; (actGraph->cutY1) = yin1;
               (actGraph->cutX2) = xin2; (actGraph->cutY2) = yin2;
              }
          }
       cutX1 = (actGraph->cutX1); cutY1 = (actGraph->cutY1);
       cutX2 = (actGraph->cutX2); cutY2 = (actGraph->cutY2);
       (actDriver->Line)(cutX1,cutY1,cutX2,cutY2);
       (actDriver->NewPict()); DrawFrame();
      }
   else
      {
       cutX1 = (actGraph->cutX1); cutY1 = (actGraph->cutY1);
       cutX2 = (actGraph->cutX2); cutY2 = (actGraph->cutY2);
      }

   lenCut = dist(cutX1,cutY1,cutX2,cutY2);
   p = actTriang->firstPoint;
   MaxZ = RA(p,R_SOL);
   MinZ = RA(p,R_SOL);
   p = p->next;
   while (p!=nil)
     {
      if (RA(p,R_SOL) > MaxZ) MaxZ = RA(p,R_SOL);
      if (RA(p,R_SOL) < MinZ) MinZ = RA(p,R_SOL);
      p = p->next;
     }
   ApplyE(getIntersE,boundInit);
   for (i = 0; i < No_iIE; i++)
       {
        for (j = i + 1; j < No_iIE; j++)
            {
             le = &iIEL[i]; lec = &iIEL[j];
             aux = dist((le->x),(le->y),(lec->x),(lec->y));
             if (aux > lenCut)
                {
                 lenCut = aux;
                 if ((le->x) < (lec->x))
                    { cutX1 = (le->x); cutY1 = (le->y); }
                 else
                    { cutX1 = (lec->x); cutY1 = (lec->y); }
                }
            }
       }
   save_left  = (actGraph->left);  save_bottom = (actGraph->bottom);
   save_right = (actGraph->right); save_top    = (actGraph->top);
   (actGraph->left)  = 0.0;    (actGraph->bottom) = MinZ;
   (actGraph->right) = lenCut; (actGraph->top)    = MaxZ;
   ComputeScaling(actGraph,actDriver);

   while (No_iIE > 0)
       {
        le = &iIEL[0]; currEdge = (le->e);
        bndEdgefF = DrCut(currEdge);
        for (i = 1, j = 0; i < No_iIE; i++)
            {
             lec = &iIEL[i];
             if ((lec->e) != bndEdgefF)
                { le = &iIEL[j++]; (le->e) = (lec->e); }
            }
        No_iIE = j;
       }

   ApplyE(UnMarkEdg,all);
   (actGraph->left)  = save_left;  (actGraph->bottom) = save_bottom;
   (actGraph->right) = save_right; (actGraph->top)    = save_top;

  } /* DrawCut */

void Draw3D()
  {
   PT      *p;
   REAL    MaxZ, MinZ;
   REAL    save_left, save_bottom, save_right, save_top;
   REAL    SinTheta, CosTheta, SinPhi, CosPhi;
   REAL    factor = REALPI / 180.;
   double  Theta, Phi;
   L_Tr_E  *le;
   int     i;

   InnerCol = actDriver->firstGray + actDriver->noOfGrays / 3;
   Theta = (double)((actGraph->rotX) * factor);
   Phi   = (double)((actGraph->rotY) * factor);
   DrawatF = (int)((actGraph->rotZ) + 0.5);
   SinTheta = (REAL) sin(Theta); CosTheta = (REAL) cos(Theta);
   SinPhi   = (REAL) sin(Phi);   CosPhi   = (REAL) cos(Phi);
   V11 =  CosPhi * CosTheta; V12 =  SinPhi * CosTheta; V13 = -SinTheta;
   V21 = -SinPhi           ; V22 =  CosPhi;            V23 = 0;
   V31 =  CosPhi * SinTheta; V32 =  SinPhi * SinTheta; V33 =  CosTheta;
                                              V34 = 0.; /* (actGraph->rotZ); */
   p = actTriang->firstPoint;
   MaxZ = RA(p,R_SOL);
   MinZ = RA(p,R_SOL);
   p = p->next;
   while (p!=nil)
     {
      if (RA(p,R_SOL) > MaxZ) MaxZ = RA(p,R_SOL);
      if (RA(p,R_SOL) < MinZ) MinZ = RA(p,R_SOL);
      p = p->next;
     }
   Xoff = actGraph->left;
   Yoff = actGraph->bottom;
   Zoff = MinZ;
   scalX = (actGraph->right - actGraph->left) / 2.0;
   scalY = (actGraph->top - actGraph->bottom) / 2.0;
   scalZ = (MaxZ - MinZ) / 2.0;
   save_left  = (actGraph->left);  save_bottom = (actGraph->bottom);
   save_right = (actGraph->right); save_top    = (actGraph->top);
   (actGraph->left)  = -1.7321; (actGraph->bottom) = -1.7321;
   (actGraph->right) =  1.7321; (actGraph->top)    =  1.7321;
   ComputeScaling(actGraph,actDriver);
   (actDriver->Settings)(PENCOLOR, BLACK);
   (actDriver->Settings)(PENSIZE, SMALLSIZE);
   VTr_bCube();
   ApplyP(ViewTransPoint,all);

   len_TrL = actTriang->noOfTriangles * 8;
   if (TrL != nil) ZIBFree((PTR)TrL);
   TrL = (L_Tr_E *)ZIBAlloc((long)len_TrL * sizeof(L_Tr_E));
   if (TrL == nil)
      { ZIBStdOut("Graph: not enough memory (Draw3D)\n"); return; }
   No_L_Tr_E = 0;
   Minjac = 9999999.; Maxjac = 0.; Totjac = 0.0;
   ApplyT(TesT,all);
   Meanjac = (Totjac / (actTriang->noOfTriangles)) / 2.0;
   ApplyT(DivEtSort,all);
   Dr_bCube(true);
   for (i = 0; i < No_L_Tr_E; i++) { le = &TrL[i]; Dr3D(le->t); }
   Dr_bCube(false);

   (actGraph->left)  = save_left;  (actGraph->bottom) = save_bottom;
   (actGraph->right) = save_right; (actGraph->top)    = save_top;

   return;
  } /* Draw3D */
