"""
t3MRES.py

Provides a class representing a multimedia resource block in a t3 image file.
The class also enables ripping of resources from image files.

The MRESMASK constant is the bitmask that is used to obscure the filenames of
multimedia resources in t3 image files.
"""

import struct
from t3Block import *

MRESMASK = 0xFF

class MRES_Block(Block):
	"""
	A multimedia resource block. There may be multiple such blocks in an image
	file, but there is a single global namespace for the names defined within
	each one.
	"""
	name = "Multimedia Resource Block"

	def __init__(self, datastream, image, attribs):
		Block.__init__(self, datastream, image, attribs)
		self.read_block()
		
	def read_block(self, ripresources=False):
		"""
		Reads the data included in the block.
		"""
		self.data["l_tableOfContents"] = self.interpret_toc()
		if self.image.options.get("RIP", False):
			self.ripresources()
		self.datastream.seek(self.datastreamidx + self.i_size)
		
	def interpret_toc(self):
		toc = []
		n = struct.unpack("<H", self.datastream.read(2))[0]
		for i in xrange(n):
			entry = {}
			offset, bytes, lenname = struct.unpack("<IIB", self.datastream.read(9))
			entry["i_offset"] = offset
			entry["i_bytes"] = bytes
			entry["i_lengthOfName"] = lenname
			encodedname = self.datastream.read(lenname)
			decodedname = []
			for i in xrange(lenname):
				b = struct.unpack("<B", encodedname[i])[0]
				decodedname.append(chr(b ^ MRESMASK))
			entry["s_name"] = "".join(decodedname)
			toc.append(entry)
		return toc
		
	def ripresources(self, ds=False):
		for resdict in self.data["l_tableOfContents"]:
			offset = resdict["i_offset"]
			length = resdict["i_bytes"]
			name = resdict["s_name"]
			self.datastream.seek(self.datastreamidx + offset)
			f = open(name, "wb")
			f.write(self.datastream.read(length))
			f.close()

	def report_data(self):
		sl = []
		for entry in self.data["l_tableOfContents"]:
			sl.append("offset: " + str(entry["i_offset"]) + "\n" + 
						"length: " + str(entry["i_bytes"]) + "\n" + 
						"name: " + str(entry["s_name"]) + "\n")
		return "".join(sl)