/*
 RAP 1.0 Kernel

 Reactive Agent Planner for TADS
 (the Text Adventure Development System)
 ----------------------------------------

 Kernel
 ------

  Main decision engine, planbase class definitions,
  and supporting utility functions


 Defines

   rapper: object
   rOpcode: object
   rToken: object
   rCond: rToken
   rAct: rToken
   
   rToString: function

*/




class rapper: object

 /* this is the main object for all RAP functions
    Use multiple inheritance to use it - rapper doesn't define
    any Actor bahaviours, so you can use it with whatever actor class
    you have.  In a standard adv.t game you would define your NPCs
    as rapper, Actor.

   To minimise namespace conflicts, all RAP methods and properties
   are prefixed with either r or rap

   methods:

     rLoudAction        should actor actions be described to player?
     rLoudActionWhen    formula for calcing when rLoudAction is true

     rToggleTrace       toggles decision tracing for debugging   

     rIdleCode          what the actor does when no action works
     rIdleDesc          what the actor describes when no action works

     rTravelToDesc      message to describe travel into a room
     rTravelFromDesc    message to describe travel out of a room

     rapAct             main RAP API, searches a goal and makes an action

     rFind              core RAP goalsearch algorithm

     rSetAnswer         returns action recommendation from rFind
                        and stores it in rAnswer

     rFound             is rAnswer non nil?
     rAction            returns action component of rAnswer
     rParam             returns parameter component of rAnswer

     rAddGoals          rFind utility, adds plans for parent goals
     rFindDupe          rFind utility, kills duplicate goals
     rStampStack        rFind utility, puts entry on rStack
     rFireAction        rFind utility, adds action to rActions


   properties:

     rStack             main decision stack (list)
     rActions           stack of actions returned (list)

     rLoudActionAlways  for actor tracing, describes actions even
                        when player is not nearby

     rTrace             is decision tracing on?

     rAnswer            returned action recommendation from last call
                        made to rFind


*/



// rLoudAction: method, returns whether actor's rAction should be notified
// to the player.  Normally this is true only if the actor is visible
// and it's not dark.
// setting rLoudActionAlways to true overrides rLoudAction
// override rLoudActionWhen to define nonstandard rLoudAction behaviour

rLoudActionAlways = nil
rLoudAction(cond, param) =
{
 if (self.rLoudActionAlways) return (true);
 else return(self.rLoudActionWhen(cond,param));
}
rLoudActionWhen(cond, param) =
{
  return (self.isVisible(Me) and Me.location.islit);
}


// rTrace: debug parameter
// shows a trace of RAP inference path
// for testing the knowledge base

// rToggleTrace toggles tracing on or off

rTrace = nil

rToggleTrace =
{
  // toggles decision tracing mode
  // also turns on loud actions, so that every move and action
  // made by the actor is visible to the player despite not being
  // nearby

  self.rTrace := not self.rTrace;
  self.rLoudActionAlways := self.rTrace;
  "RAP decision tracing for <<self.thedesc>> is
  now <<self.rTrace ? 'ON' : 'OFF'>>.";

}


 // rStack: internal goal stack
 // stores goals as: [condition parameter parent plan step]
 // where parent is the goal which placed this goal
 // plan is the plan index for [cond param] and
 // step is the index of the step in plan reached before ending search

 rStack = []

// actions: internal stack of actions generated by rFind
// in ver 1.0 only one action is generated
// format: [ [action param] ] 

 rActions = []

// idle methods - called by rNoAction (when rap search finds no goals)

 rIdleCode = {}
 rIdleDesc =

    "\b\^<<self.thedesc>> does nothing."


 rTravelToDesc(loc) = "\b\^<<self.thedesc>> heads off toward <<loc.traveldesc>>."
 rTravelFromDesc(loc) = "\b\^<<self.thedesc>> approaches from <<loc.traveldesc>>."


// rapAct: main Rap function
// calls rFind, if an action is returned runs it otherwise does rNoAction


rapAct(cond,param) =
{
  local x, thiscond,thisparam;

//  "\n<<self.sdesc>>:\ rapAct\n";

  // call main Rap goalsearch method (rfind) to find an appropriate action

  x:=self.rFind(cond,param);

  // run the action that was returned
  // a nil return means rNoAction

  if (datatype(x)=7) // list
  {
    thiscond:=x[1];
    thisparam:=x[2];
  }
    else
  {
    thiscond:=rNoAction;
    thisparam:=nil;
  }

// "\nrapAct: thiscond=<<rToString(thiscond)>>, thisparam=<<rToString(thisparam)>>\n";

    thiscond.rAction(self,thisparam);
}




 // rFind: Rap agent kernel
 // find condition, return true if exists, nil if not findable,
 // otherwise return best next action step (list, cond+param)
 // Also puts copy of answer in self.rAnswer, and sets self.rFound
 // self.rAction, self.rParam




 rFind (cond, param) =
 {
    local x;
    local test;
    local j,s,temploop;
    local done, donesteps;
    local thisparent,thisopcode, thiscond,thisparam,thisplan,thisstep;
    local parentcond, parentparam, maxsteps;

    local dupes;

//    "\nrFind: started\n";

    if (self.rTrace)  "\bRAP decision trace for <<self.thedesc>>\n";

    x:=cond.rTest(self,param);
    if (datatype(x)=8) // true
      return(true);
    else if (datatype(x)=1) // number
    {

     // set up stack with main goal

     self.rActions:=[];
     self.rStack:=[[cond param nil nil nil]];
     self.rAddGoals(cond,param,1);

    // main loop: search all goals, expanding if not found

     done:=nil; 

      for(j:=2;(j<=length(self.rStack)) and (not done); j++)
      {

        // read next goal off stack
        // parent is this goal's originator
        // plan,step is this goal's index in rulebase for parent

//        "\nrFind: main loop j <<j>> \n";

        x:=self.rStack[j];
//        "\nrFind: x <<rToString(x)>>\n";
        thiscond:=x[1];     // empty
        thisparam:=x[2];    // empty
        thisparent:=x[3];
        thisplan:=x[4];
        thisstep:=x[5];     // empty

        parentcond:=self.rStack[thisparent][1];
        parentparam:=self.rStack[thisparent][2];


//        "\nrFind: thisparent <<rToString(thisparent)>> thisplan <<
//        rToString(thisplan)>> parentcont <<rToString(parentcond)
//        >> parentparam <<rToString(parentparam)>>\n";


        maxsteps:=parentcond.rNumSteps(self,parentparam,thisplan);



        // loop through all steps for this goal, testing

        donesteps:=nil;
        for (s:=1;(s<=maxsteps) and (not donesteps);s++)
        {
//                "\nrFind: step loop s <<s>> of <<maxsteps>> donesteps <<rToString(donesteps)>>\n";

                thisopcode:=parentcond.rStepOpcode(self,parentparam,thisplan,s);
                thiscond:=parentcond.rStepCond(self,parentparam,thisplan,s);
                thisparam:=parentcond.rStepParam(self,parentparam,thisplan,s);

              if (self.rTrace)
                "\ntrying <<rToString(thisopcode)>> <<rToString(thiscond)>> <<rToString(thisparam)>> for <<rToString(parentcond)>> <<rToString(parentparam)>>";


//                "\nrFind: thisopcode <<rToString(thisopcode)
//                  >> thiscond <<rToString(thiscond)>> thisparam <<rToString(thisparam)>>\n";






                if (thisopcode=rDo)
                {

//                  "\nrFind: rDo opcode\n";

                  // fire an action and stop searching

                  if (self.rTrace) "...firing";

                  self.rStampStack(j,thiscond,thisparam,s);
                  self.rFireAction(thiscond,thisparam);
                  donesteps:=true;
                  done:=true; // remove this if we want multiple actions
                }



                else // rIf or rBe
                {
//                    "\nrFind: rIf or rBe opcode\n";

                     test:=thiscond.rTest(self,thisparam);

//                     "\nrFind: test <<rToString(test)>>\n";



  

                     if (datatype(test)=8) // true
                     {
//                        "\nrFind: test true\n";


                        if (self.rTrace)  "...success";

                         // do nothing

                     }


   
                     else if ((datatype(test)=1) and (thisopcode=rBe))

                     // add goal to stack

                     {





//                       "\nrFind: test <<test>>\n";

                       dupes:=self.rFindDupe(thiscond,thisparam);
//                       "\nrFind: dupes <<rToString(dupes)>>\n";
                       self.rStampStack(j,thiscond,thisparam,s);

                       if (dupes = nil)
                         self.rAddGoals(thiscond,thisparam,j);

                       else

                       {
//                         "\nrFind: goal already on stack, not adding\n";

                       if (self.rTrace)
                           "...dupe";
                       }

 
                       donesteps:=true;

//                      "\nrFind: dupes <<rToString(dupes)>> donesteps <<rToString(donesteps)>>\n";



                     }

                     else // no plans found - abort processing for this goal
                     {
//                        "\nrFind: test nil\n";

                       self.rStampStack(j,thiscond,thisparam,s);
                       donesteps:=true;
                     }

                }

  


        } // end of steps loop

      } // end of goals loop


      // choose an action to return

//      "\nrFind: choosing action\n";

      if (length(self.rActions)>0)
      {
        return(self.rSetAnswer(self.rActions[1]));
      }
      else
        return(self.rSetAnswer(nil));

      
    }

    else // nil, or undefined datatype
      return(self.rSetAnswer(nil));

 }



 

 rSetAnswer(x) =
 {
   self.rAnswer:=x;
   return(x);
 }

 rAnswer = nil
 rFound =
 {
   return (self.rAnswer = true);
 }

 rAction =
 {
   if (datatype(self.rAnswer)=7) // list
    return (self.rAnswer[1]);
   else return (nil);
 }

 rParam =
 {
   if (datatype(self.rAnswer)=7) // list
    return (self.rAnswer[2]);
   else return (nil);
 }


// rAddGoals: add plan entries for a specified parent goal

 rAddGoals(cond,param,parent) =
 {
   local j,maxplans;

   maxplans:=cond.rNumPlans(self,param);

//   "\nrAddGoals: cond <<rToString(cond)>> param <<rToString(
//    param)>> parent <<rToString(parent)>> maxplans <<rToString(maxplans)>>\n";



//     "\nrAddGoals: adding plans\n";

     for (j:=1;j<=maxplans;j++)
     {
//       "\nrAddGoals: parent <<rToString(parent)>> plan <<rToString(j)>>\n";

       self.rStack+= [[nil nil parent j nil]];

//      "\nrAddGoals: rStack <<rToString(self.rStack)>>\n";
     }


 }


// rFindDupe: internal, searches stack for cond and param,
// returns true if found, nil otherwise

rFindDupe (cond,param) =
{
  local j, dupe, numgoals;

//   "\nrFindDupe: cond <<rToString(cond)>> param <<rToString(param)>>\n";

   dupe:=nil;
   numgoals:=length(self.rStack);
   for (j:=1;(j<=numgoals) and (not dupe);j++)
   {
     if (self.rStack[j][1]=cond)
      if (self.rStack[j][2]=param)
       dupe:=true;
   }

   return (dupe);
}

// rStampStack: internal, records cond, param and step in current working
// position on stack

 rStampStack (goal, cond,param,step) =
 {
//   "\nrStampStack: goal <<rToString(goal)>> actor <<rToString(self)>> cond <<rToString(cond)
//      >> param <<rToString(param)>> step <<rToString(step)>>\n";
     
   self.rStack[goal][1]:=cond;
   self.rStack[goal][2]:=param;
   self.rStack[goal][5]:=step;

//   "\nStack: <<goal>> <<rToString(cond)>> <<rToString(param)>> parent <<self.rStack[goal][3]>> plan <<self.rStack[goal][4]>> step <<step>>\n";

 }

 rFireAction(cond,param) =
 {
//   "\nrFireAction: cond <<rToString(cond)>> param <<rToString(param)>>\n";

   self.rActions += [[cond param]];
//   "\nrFireAction: rActions <<rToString(self.rActions)>>\n";

 }


;





// rOpcode class - used to indicate condition type in plans

class rOpcode: object
 sdesc = "rOpcode"
;

rDo: rOpcode
 sdesc = "rDo"
;

rBe: rOpcode
 sdesc = "rBe"
;

rIf: rOpcode
 sdesc = "rIf"
;


class rCond: rToken
;

class rAct: rToken
;



class rToken: object

/*
      rToken class incorporates behaviour for rCond and rAct classes -
      planbase conditions and actions

      After rapper, this is the most important class in RAP
      and is the only one the user should modify (although it will
      usually be in descendent classes).


  methods

        rTest           used by rFind, returns true or number of plans.
                        Do NOT override this method.

        rTrue           condition's truth method.  OVERRIDE THIS.

        rNumPlans       returns number of plans in this condition's
                        plan list

        rNumSteps       returns number of steps in a condition's plan

        rStep           returns one step in a condition's plan

        rStepAtom       internal function, returns atomic list entry
                        in a condition's specific plan step by offset

        rStepOpcode     returns opcode for specific plan step
        rStepCond       returns condition/action for specific plan step
        rStepParam      returns parameter for specific plan step

        rPlans          main planbase method, returns plan list for
                        a given actor and parameter.  OVERRIDE THIS.
        
        rAction         performs a specified action given actor and parameter
        rPreDesc        describes action before doing it.  OVERRIDE THIS.
        rPostDesc       describes action after doing it.  Not often used
                        but might be useful.
        rActionCode     the actual code for doing an action.  OVERRIDE THIS.



        

  properties

        sdesc           condition or action name

        stepsize        number of atomic list objects in a plan entry.
                        Change this and it breaks horribly.

        


*/


  sdesc = "rToken"

  // stepsize: internal constant, number of list atoms for each plan step

   stepsize = 3


  // rTest: evaluate condition, return true if exists, nil if no plan
  // available, else the number of plans available


  // rTest: exported test method, don't override this

  rTest(actor, param) =
  {
    local x;
//    "\nrTest: actor <<rToString(actor)>> cond <<rToString(self)>> param <<rToString(param)>>\n";
    if (self.rTrue(actor,param))
      return (true);
    else
      return (self.rNumPlans(actor,param));
  }

  // rTrue: actual test, returns true or false only
  // override this one at will

  rTrue (actor,param) =
  {
    return (nil);
  }


  //rNumPlans: returns number of plans for test or nil if no plans

  rNumPlans(actor,param) = 
  {
    local x;

//    "\nrNumPlans: actor <<rToString(actor)>> cond <<rToString(self)>> param <<rToString(param)>>\n";

    x:=self.rPlans(actor,param);

//    "\nrNumPlans: x <<rToString(x)>> length(x) <<length(x)>>\n";

    if (x)
    {
//     "\nrNumPlans: length(x) <<rToString(length(x))>>\n";
     return (length(x));
    }
    else
      return (nil);

  }

// rNumSteps: returns number of steps for plan or nil if no steps


rNumSteps(actor,param,plan) =
{
  local x;
//  "\nrNumSteps: actor <<rToString(actor)>> cond <<rToString(self)>> param <<rToString(param)>> plan <<rToString(plan)>>\n";
  x:=self.rPlans(actor,param)[plan];
    if (x)
     return (length(x)/self.stepsize);
    else
      return (nil);

}


/*
//rStep: return next plan step (list: opcode, cond, param)
// by plan and step

  rStep(actor,param,plan,step) =
  {
    local x, pos;
    x:=self.rPlans(actor,param)[plan];
    if (step<=length(x))
      return(self.rPlans(actor,param)[plan][step]);
    else return(nil);
  }

*/



  // rStepAtom: internal function, returns next condition/action
  // by plan, step and offset
 
  rStepAtom(actor,param,plan,step,offset) =
  {
    local x, pos;
    x:=self.rPlans(actor,param)[plan];
    pos:=((step-1)*self.stepsize)+offset;
    if (step<=length(x))
      return(self.rPlans(actor,param)[plan][pos]);
    else return(nil);
  }



  // rStepOpcode, rStepCond, rStepParam: returns atoms of plan step


  rStepOpcode(actor,param,plan,step) =
  {
    return (self.rStepAtom(actor,param,plan,step,1));
  }

  rStepCond(actor,param,plan,step) =
  {
    return (self.rStepAtom(actor,param,plan,step,2));
  }

  rStepParam(actor,param,plan,step) =
  {
    return (self.rStepAtom(actor,param,plan,step,3));
  }




  // rPlans: internal function returning planbase by actor and param
  // in this format:
  // [
  // [ rOpcode rCond param rOpcode rCond param...]
  // [ rOpcode... ]
  // ]
  
  rPlans(actor, param) = nil


  // rAction: run an action
  // call rPreDesc to describe action to player if visible
  // then call rActionCode to modify game state
  // then call rPostDesc for afterword description if required

  rAction(a,p) =
  {
//    "\n<<rToString(a)>>: rAction <<rToString(self)>> <<rToString(p)>>\n";

      if (a.rLoudAction(self,p)) self.rPreDesc(a,p);
      self.rActionCode(a,p);
      if (a.rLoudAction(self,p)) self.rPostDesc(a,p);
  }

  rPreDesc(a,p) = "\b\^<<a.thedesc>>: undescribed RAP action <<self.sdesc>> <<rToString(p)>>.\n"
  rPostDesc(a,p) = {}
  rActionCode(a,p) = "\b\^<<a.thedesc>>: uncoded RAP action <<self.sdesc>> <<rToString(p)>>.\n"   
   

;



// rToString is a function so useful it should be in the default libraries
// but I've left it with an r prefix in case the name conflicts.  It
// simply displays a text equivalent of any object reference passed to it,
// including nil pointers and lists. I couldn't live without it for debugging.
// Feel free to cut and paste this into your standard libraries.

rToString: function (value)
{
   local j;
    switch ((datatype(value)))
    {
     case 1:   say(value); " ";    break;
     case 2:   "\"";value.sdesc;"\""; " "; break;
     case 3:   "'"; say(value);"'"; " ";  break;
     case 5: "nil"; " "; break;
     case 7:
         "[ ";
         for (j:=1;j<=length(value);j++)
           rToString(value[j]);" ";
         "] ";  
         break;
     case 8: "true"; " "; break;
     case 9: "function_pointer"; " "; break;
     case 10: "property_pointer"; " ";  break;

    }
    return (nil);
}
;



